// -*- C++ -*-

//
// This file is a part of the Bayes Blocks library
//
// Copyright (C) 2001-2006 Markus Harva, Antti Honkela, Alexander
// Ilin, Tapani Raiko, Harri Valpola and Tomas stman.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License (included in file License.txt in the
// program package) for more details.
//
// $Id: Loader.h 7 2006-10-26 10:26:41Z ah $

#ifndef LOADER_H
#define LOADER_H

class Node;
class NodeBase;

#include <functional>
#include <algorithm>
#include "Templates.h"


/**********************************
**  Node type constants
*/
enum node_types {
  cn_Unknown      =  0,

  cn_Constant,
  cn_ConstantV,

  cn_Prod,
  cn_ProdV,
  cn_Sum2,
  cn_Sum2V,
  cn_SumN,
  cn_SumNV,
  cn_Rectification,
  cn_RectificationV,
  cn_DelayV,

  cn_Gaussian,
  cn_GaussianV,
  cn_DelayGaussV,
  cn_SparseGaussV,
  cn_RectifiedGaussian,
  cn_RectifiedGaussianV,
  cn_GaussRect,
  cn_GaussRectV,
  cn_GaussNonlin,
  cn_GaussNonlinV,
  cn_MoG,
  cn_MoGV,
  cn_Discrete,
  cn_DiscreteV,
  cn_DiscreteDirichlet,
  cn_DiscreteDirichletV,
  cn_Dirichlet,

  cn_Proxy,
  cn_Relay,
  cn_Evidence,
  cn_EvidenceV,

  cn_Memory,
  cn_OLDelayS,
  cn_OLDelayD,

};


class ConcreteLoader
{
public:
  virtual ~ConcreteLoader() {}
  virtual void   LoadIt() = 0;
  virtual int    StartEnumeratedContainer(string name) = 0;
  virtual int    StartNamedContainer(string name) = 0;
  virtual void   CloseEnumeratedContainer(string name) = 0;
  virtual void   CloseNamedContainer(string name) = 0;
  virtual int    StartNode(string & type) = 0;
  virtual void   CloseNode(string type) = 0;

  virtual int GetDouble(double & val) = 0;
  virtual int GetInt   (int    & val) = 0;
  virtual int GetBool  (bool   & val) = 0;
  virtual int GetString(string & val) = 0;
  virtual int GetLabel (Label  & val) = 0;
  virtual int GetDV    (DV     & val) = 0;
  virtual int GetDFlags(DFlags & val) = 0;
  virtual int GetDSSet (DSSet  & val) = 0;
  virtual int GetDVSet (DVSet  & val) = 0;
  virtual int GetDVH   (DVH    & val) = 0;
  virtual int GetDD    (DD     & val) = 0;
  virtual int GetVDD   (VDD    & val) = 0;
  virtual int GetIntV  (IntV   & val) = 0;

  virtual int GetNamedDouble(string name, double & val) = 0;
  virtual int GetNamedInt   (string name, int    & val) = 0;
  virtual int GetNamedBool  (string name, bool   & val) = 0;
  virtual int GetNamedString(string name, string & val) = 0;
  virtual int GetNamedLabel (string name, Label  & val) = 0;
  virtual int GetNamedDV    (string name, DV     & val) = 0;
  virtual int GetNamedDFlags(string name, DFlags & val) = 0;
  virtual int GetNamedDSSet (string name, DSSet  & val) = 0;
  virtual int GetNamedDVSet (string name, DVSet  & val) = 0;
  virtual int GetNamedDVH   (string name, DVH    & val) = 0;
  virtual int GetNamedDD    (string name, DD     & val) = 0;
  virtual int GetNamedVDD   (string name, VDD    & val) = 0;
  virtual int GetNamedIntV  (string name, IntV   & val) = 0;

};

class NetLoader
{
public:
  NetLoader(ConcreteLoader *theloader) : loader(theloader) { }

  void LoadIt() { loader->LoadIt(); }
  ~NetLoader() { delete loader; }

  void StartNet(string name) { loader->StartEnumeratedContainer(name); }

  int  StartNode(string & type) {
    parents.clear();
    return loader->StartNode(type);
  }

  void AddParent(Node *par) {
    parents.push_back(par);
  }

  void FinishNet(string name) { loader->CloseEnumeratedContainer(name); }

  void FinishNode(string type, std::binder1st<std::mem_fun1_t<void, NodeBase, Node*> > f) {
    loader->CloseNode(type);
    for_each(parents.begin(), parents.end(), f);
  }

  int  StartEnumCont(string name)
  { return( loader->StartEnumeratedContainer(name) ); }

  int  StartNamedCont(string name)
  { return( loader->StartNamedContainer(name) ); }

  void FinishEnumCont(string name)
  { loader->CloseEnumeratedContainer(name); }

  void FinishNamedCont(string name)
  { loader->CloseNamedContainer(name); }


  int GetDouble(double & val) { return( loader->GetDouble(val) ); }
  int GetInt   (int    & val) { return( loader->GetInt(val) ); }
  int GetBool  (bool   & val) { return( loader->GetBool(val) ); }
  int GetString(string & val) { return( loader->GetString(val) ); }
  int GetLabel (Label  & val) { return( loader->GetLabel(val) ); }
  int GetDV    (DV     & val) { return( loader->GetDV(val) ); }
  int GetDFlags(DFlags & val) { return( loader->GetDFlags(val) ); }
  int GetDSSet (DSSet  & val) { return( loader->GetDSSet(val) ); }
  int GetDVSet (DVSet  & val) { return( loader->GetDVSet(val) ); }
  int GetDVH   (DVH    & val) { return( loader->GetDVH(val) ); }
  int GetDD    (DD     & val) { return( loader->GetDD(val) ); }
  int GetVDD   (VDD    & val) { return( loader->GetVDD(val) ); }
  int GetIntV  (IntV   & val) { return( loader->GetIntV(val) ); }

  int GetNamedDouble(string name, double & val)
  { return( loader->GetNamedDouble(name, val) ); }

  int GetNamedInt(string name, int & val)
  { return( loader->GetNamedInt(name, val) ); }

  int GetNamedBool(string name, bool & val)
  { return( loader->GetNamedBool(name, val) ); }

  int GetNamedString(string name, string & val)
  { return( loader->GetNamedString(name, val) ); }

  int GetNamedLabel(string name, Label & val)
  { return( loader->GetNamedLabel(name, val) ); }

  int GetNamedDV(string name, DV & val)
  { return( loader->GetNamedDV(name, val) ); }

  int GetNamedDFlags(string name, DFlags & val)
  { return( loader->GetNamedDFlags(name, val) ); }

  int GetNamedDSSet(string name, DSSet & val)
  { return( loader->GetNamedDSSet(name, val) ); }

  int GetNamedDVSet(string name, DVSet & val)
  { return( loader->GetNamedDVSet(name, val) ); }

  int GetNamedDVH(string name, DVH & val)
  { return( loader->GetNamedDVH(name, val) ); }

  int GetNamedDD(string name, DD & val)
  { return( loader->GetNamedDD(name, val) ); }

  int GetNamedVDD(string name, VDD & val)
  { return( loader->GetNamedVDD(name, val) ); }

  int GetNamedIntV(string name, IntV & val)
  { return( loader->GetNamedIntV(name, val) ); }

private:
  ConcreteLoader *loader;
  vector<Node *> parents;
};

#endif // LOADER_H
