%  RUN_SGICA.M
%
%  run_sgica( datafile, initA, iter, nsave, diag_qs ) starts the
%  simulations for the ICA model with super-Gaussian sources.
%  The observed data are taken from datafile, initA ('ICA' or 'PCA')
%  specifes which initialisation for the mixing matrix is
%  used. Argument iter defines the total number of iterations of the
%  learning algorithm, nsave is the number of iterations between
%  saving intermediate results. The last argument diag_qs defines
%  wherther the fully factorial posterior approximation for the
%  sources is used (diag_qs=1) or the correlations between different
%  sources are modelled in the posterior (diag_qs=0, default).
%
%  The simulation results are saved in 'RESULTS/Full_Qs' or in
%  'RESULTS/Diag_qs' depending on the parameter diag_qs. The name of
%  the file is constructed automatically based on the passed
%  arguments.
%
%  Examples:
%    run_sgica( 'sg_1_0', 'PCA', 1000, 100, 0 )
%
%  To continue the simulations, use
%    cnt_sgica( 'sg_1_0', 'PCA', 2000, 100, 0 )
%
function run_sgica( datafile, initA, iter, nsave, diag_qs )

close all

if nargin < 1 | isempty(datafile)
    datafile = 'varfa-1';
end
if nargin < 2 | isempty(initA)
    initA = 'ICA';
end
if nargin < 3 | isempty(iter)
    iter = 2000;
end
if nargin < 4 | isempty(nsave)
    nsave = 100;
end
if nargin < 5 | isempty(diag_qs)
    diag_qs = 0;
end

if diag_qs
    netfile = [ package_dir '/RESULTS/Diag_Qs/' datafile '__' initA ];
else
    netfile = [ package_dir '/RESULTS/Full_Qs/' datafile '__' initA ];
end

% Check the existence of the file with the same results
D = dir( [ netfile '.mat' ] );
if ~isempty(D)
    R = '';
    while ~strcmp(R,'y')
        fprintf( '\nFile %s already exists\n', netfile )
        R = input( 'Do you want to overwrite the results? (y/n): ', 's' );
        R = lower(R);
        if strcmp(R,'n')
            return
        end
    end
end

    %
    % Data
    %
load( [ package_dir '/DATA/' datafile ], ...
      'A', 'Apca', 's', 'x' )
x = x';
s = s';

switch initA
case 'ICA'
    A_init = A;
case 'PCA'
    A_init = Apca;
case 'random'
    A_init = rand( size(A) ) - 0.5;
end


[ net{1}, status ] = sgica( x, 0,...
                             'searchsources', 2,...
                             'initA', A_init,...
                             'diag_qs', diag_qs );
[ net{2}, status ] = sgica( x, 10, net{1}, status );
status.A_clamped = 0;
nit = [ 0 10 ];

for i = 1:ceil( iter/nsave )
    
    if i*nsave > iter
        nit(i+2) = iter;
    else
        nit(i+2) = i*nsave;
    end
    [ net{i+2}, status ] = sgica( x, nit(i+2), net{i+1}, status );
    CF = status.CF;
    
    if i == 3
        save( netfile, 'net', 'nit', 'CF' )
    elseif i > 3
        save( netfile, 'net', 'nit', '-append' )
    end

end

disp( [ 'The results written to ' netfile ] )

