%  SGICA.M
%
%  VB independent component analysis with super-Gaussian sources. The
%  data must be centered. By default, the sources are fixed to the
%  principal components of the data.
%
%  Usage: To initialise the model and start learning:
%
%  net = sgica( data, iter, 'searchsources', 2,...
%                           'initsources', s_init,...
%                           'diag_qs', 0 )
%  or
%  net = sgica( data, iter, 'searchsources', 2,...
%                           'initA', A_init,...
%                           'diag_qs', 0 )
%
%  To continue learning of the existing model:
%  net = sgica( data, iter, net, status )
%
function [ net, status ] = sgica( data, iter, varargin )

[ xdim, tdim ] = size(data);

if isstruct(varargin{1})

    % The network is passed to the function
    [ A, s, vx, Vvx, Mvx, u, Mmu, Vmu, Mvu, Vvu, mu, vu, x ] = ...
        net2sgica( varargin{1}, data );
        
    if nargin < 4 | isempty(varargin{2})
        error( 'SGICA: Unspecifed status' )
    else
        status = varargin{2};
    end
    
    CF = status.CF;
    complete_iters = status.iter;
    diag_qs = status.diag_qs;
    
    if isfield( status, 'A_clamped' )
        A.clamped = status.A_clamped;
    else
        A.clamped = 0;
    end

    optional_flds = { 's_clamped', 'Mvx_clamped',...
                      'Vvx_clamped', 'vx_clamped' };
    for i = 1:length(optional_flds)
        if isfield( status, optional_flds{i} )
            eval( [ optional_flds{i} ...
                    '= status.' optional_flds{i} ';' ] )
        else
            eval( [ optional_flds{i} '= 0;' ] )
        end
    end

else
    
    % Default parameter values
    options = struct( 'searchsources', [], ...
                      'inita', [], ...
                      'initsources', [], ...
                      'diag_qs', 0 );
    
    if mod( length(varargin), 2 )
        error( 'SGICA: Not enough arguments' )
    end
    for i = 1:2:length(varargin)
        if ~isfield( options, lower(varargin{i}) )
            error( [ 'SGICA: Unknown parameter: ' varargin{i} ] )
        end
        options = setfield( options, lower(varargin{i}),...
                                     varargin{i+1} );
    end
    
    sdim = options.searchsources;
    if isempty( sdim )
        error( 'SGICA: Unspecified number of sources' )
    end
    diag_qs = options.diag_qs;

    %
    % Create the model
    %
    
    Mmu.mean = 0;
    Mmu.var  = 1;
    
    Vmu.mean = 0;
    Vmu.var  = 1;
    Vmu.mex  = meanexp( Vmu );
    
    Mvu.mean = 0;
    Mvu.var  = 1;
    
    Vvu.mean = 0;
    Vvu.var  = 1;
    Vvu.mex  = meanexp( Vvu );
    
    mu.mean = repmat( Mmu.mean, sdim, 1 );
    mu.var  = repmat( 1/Vmu.mex, sdim, 1 );

    vu.mean = repmat( Mvu.mean, sdim, 1 );
    vu.var  = repmat( 1/Vvu.mex, sdim, 1 );
    vu.mex  = meanexp( vu );
    
    for t = 1:tdim
        u(t).mean = mu.mean;
        u(t).var = 1./vu.mex;
        u(t).mex = meanexp(u(t));
    end
    
    for t = 1:tdim
        s(t).mean = zeros(sdim,1);
        s(t).covar = diag( 1./u(t).mex );
    end

    A.mean = zeros( xdim, sdim );
    A.var = ones( xdim, sdim );
    
    Mvx.mean = 0;
    Mvx.var  = 1;
    
    Vvx.mean = 0;
    Vvx.var  = 1;
    Vvx.mex  = meanexp( Vvx );
    
    vx.mean = repmat( Mvx.mean, xdim, 1 );
    vx.var  = repmat( 1/Vvx.mex, xdim, 1 );
    vx.mex  = meanexp( vx );
    
    for t = 1:tdim
        x(t).value = data(:,t);
    end

    %
    % Initialization
    %
    if isempty( options.initsources ) & isempty( options.inita )
        
        % Initialize sources with PCA
        C = data*data'/tdim;
        [ V, L ] = eig(C);
        
        [ L, id ] = sort( -diag(L) );
        V = V(:,id);
        options.initsources = V( :, 1:sdim )' * data;
        
    end
    if ~isempty( options.initsources )
        s_clamped = 1;
        for t = 1:tdim
            s(t).mean = options.initsources(:,t);
            s(t).covar = zeros(sdim,sdim);
        end
    else
        s_clamped = 0;
    end
    if ~isempty( options.inita )
        A.clamped = 1;
        A.mean = options.inita;
        A.var = zeros( xdim, sdim );
    else
        A.clamped = 0;
    end
    
    Mvx_clamped = 0;
    Vvx_clamped = 0;
    vx_clamped = 0;

    CF = [];
    
    complete_iters = 0;
    status.diag_qs = diag_qs;

end

    %
    % Learning
    %
for i = complete_iters+1:iter
    
    fprintf( '%d-', i );

    if ~vx_clamped
        vx = update_vx( vx, Mvx, Vvx, x, A, s, diag_qs );
    end
    if ~Vvx_clamped
        Vvx = update_Vvx( Vvx, Mvx, vx );
    end
    if ~Mvx_clamped
        Mvx = update_Mvx( Mvx, Vvx, vx );
    end
    
    if ~A.clamped
        A = update_A( A, s, vx, x );
    end

    if ~s_clamped
        s = update_sv( s, u, A, vx, x, diag_qs );
    end

    u = update_u( u, mu, vu, s );
    
    mu = update_mu( mu, Mmu, Vmu, u, vu );
    vu = update_vu( vu, Mvu, Vvu, u, mu );

    Mmu = update_Mvx( Mmu, Vmu, mu );
    Vmu = update_Vvx( Vmu, Mmu, mu );
    Mvu = update_Mvx( Mvu, Vvu, vu );
    Vvu = update_Vvx( Vvu, Mvu, vu );

end

net = sgica2net( A, s, vx, Vvx, Mvx, u, Mmu, Vmu, Mvu, Vvu, mu, vu );

status.iter = iter;
status.CF = CF;
status.A_clamped = A.clamped;
status.s_clamped = s_clamped;

status.Mvx_clamped = Mvx_clamped;
status.Vvx_clamped = Vvx_clamped;
status.vx_clamped = vx_clamped;

return
