%  NPLFIGURES.M
%
%  nplfigures(n) plots the n-th figure of the NPL paper.
%
%  Some extra results are available:
%    n = 11 - ICA with super-Gaussian sources,
%             full posterior for sources
%    n = 12 - ICA with mixture-of-Gaussians model for sources,
%             full posterior for sources
%    n = 13 - ICA with mixture-of-Gaussians model for sources,
%             diagonal MoG posterior for sources
%
function nplfigures(n)

global NPLFIG_4in1

NPLFIG_4in1 = 1;

%close all
switch n
case 1, Tradeoff
case 2, CloudDyn
case 3, diag_dyn
case 4, fullqs_dyn
case 5, CloudSG
case 6, diag_sg_pca, diag_sg_ica
case 7, diag_mog_pca, diag_mog_ica
    
    % Some extra figures
case 11
    fullqs_sg_pca, fullqs_sg_ica

case 12
    fullqs_mog_pca
    
case 13, diag_mogifa_pca, diag_mogifa_ica
    
end

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 3. ICA with temporally correlated sources: Diagonal approximation
function diag_dyn

Plot1Track( 'BBpython/dyn8_8__PCA', 'dyn8_8', [ 0:200:2000 ], 1 );
Plot1Track( 'BBpython/dyn8_6__PCA', 'dyn8_6', [ 0:200:2000 ], 2 );
Plot1Track( 'BBpython/dyn8_-2__PCA', 'dyn8_-2', [ 0:200:2000 ], 3 );
[ hPCA, hICA ] = Plot1Track( 'BBpython/dyn8_-8__PCA',...
                             'dyn8_-8', [ 0:200:2000 ], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );
    
return


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 4. ICA with temporally correlated sources: Full approximation
function fullqs_dyn

Plot1Track( 'Full_Qs/dyn8_8__ICA', 'dyn8_8', [ 0:5000:140000 ], 1 );
Plot1Track( 'Full_Qs/dyn8_6__PCA', 'dyn8_6', [ 0:1000:25000 ], 2 );
Plot1Track( 'Full_Qs/dyn8_-2__PCA', 'dyn8_-2', [ 0:100:2200 ], 3 );
[ hPCA, hICA ] = Plot1Track( 'Full_Qs/dyn8_-8__PCA',...
                             'dyn8_-8', [ 0:100:700 ], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );

return


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 6.1. ICA with super-Gaussian sources:
%      Diagonal approximation, PCA initialization
function diag_sg_pca

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'Super-Gaussian sources, diagonal, PCA' )
end

Plot1Track( 'BBpython/sg_0_6__PCA', 'sg_0_6', [ 10, 0:5000:30000 ], 1 );
Plot1Track( 'BBpython/sg_0_7__PCA', 'sg_0_7', [], 2 );
Plot1Track( 'BBpython/sg_0_9__PCA', 'sg_0_9', [], 3 );
[ hPCA, hICA ] = Plot1Track( 'BBpython/sg_1__PCA', 'sg_1_0', [], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 6.2. ICA with super-Gaussian sources:
%      Diagonal approximation, ICA initialization
function diag_sg_ica

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'Super-Gaussian sources, diagonal, ICA' )
end

Plot1Track( 'BBpython/sg_0_6__ICA', 'sg_0_6', [], 1 );
Plot1Track( 'BBpython/sg_0_7__ICA', 'sg_0_7', [ 10 0:10000:50000 ], 2 );
Plot1Track( 'BBpython/sg_0_9__ICA', 'sg_0_9', [ 10 0:10000:50000 ], 3 );
[ hPCA, hICA ] = Plot1Track( 'BBpython/sg_1__ICA',...
                             'sg_1_0', [ 10 0:10000:50000 ], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 7.1. ICA with mixture-of-Gaussians model for sources:
%      Diagonal approximation, PCA initialization
function diag_mog_pca

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'MoG sources, diagonal, PCA' )
end

Plot1Track( 'Diag_Qs/mog__sg_0_6__PCA',...
            'sg_0_6', [ 10 0:2000:10000 ], 1 );
Plot1Track( 'Diag_Qs/mog__sg_0_7__PCA',...
            'sg_0_7', [ 10 0:2000:10000 ], 2 );
Plot1Track( 'Diag_Qs/mog__sg_1__PCA',...
            'sg_1_0', [ 10 0:2000:25000 ], 3 );
[ hPCA, hICA ] = Plot1Track( 'Diag_Qs/mog__sg_1_2__PCA',...
                             'sg_1_2', [ 10 0:2000:30000 ], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );
return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 7.2. ICA with mixture-of-Gaussians model for sources:
%      Diagonal approximation, ICA initialization
function diag_mog_ica

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'MoG sources, diagonal, ICA' )
end

Plot1Track( 'Diag_Qs/mog__sg_0_6__ICA', 'sg_0_6', [], 1 );
Plot1Track( 'Diag_Qs/mog__sg_0_7__ICA', 'sg_0_7', [], 2 );
Plot1Track( 'Diag_Qs/mog__sg_1__ICA', 'sg_1_0', [], 3 );
[ hPCA, hICA ] = Plot1Track( 'Diag_Qs/mog__sg_1_2__ICA',...
                             'sg_1_2', [], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 2. The samples
%
function CloudDyn

datafile = 'dyn8_-8';
load( [ package_dir '/PAPER_DATA/' datafile ] )
x = x';

Q = orth(A);
prA = Q';

Apca_2 = prA*Apca;
A_2 = prA*A;
x_2 = prA*x;

figure
subplot(3,1,1)
plot( s(1,:), 'k-' )
set( gca, 'YLim', [ -3 3 ] )

subplot(3,1,2)
plot( s(2,:), 'k-' )
set( gca, 'YLim', [ -3 3 ] )

subplot(3,1,3)
set( gca, 'Box', 'On',...
          'NextPlot', 'Add',...
          'DrawMode', 'Fast',...
          'XLim', [ -10  10 ],...
          'YLim', [ -3 3 ] )
          %'XTick', [],...
          %'YTick', [] )

plot( x_2(1,:), x_2(2,:), 'k.', 'MarkerSize', 3 )

hPCA = plot_dir( Apca_2, [-20 20], [-20 20], ...
                 'Color', [ 0 0 0 ], 'LineStyle', '--' );
hICA = plot_dir( A_2, [-20 20], [-20 20], ...
                 'Color', [ 0 0 0 ], 'LineStyle', '-.' );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 )

axis equal

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 5. The samples
%
function CloudSG

global NPLFIG_4in1

files = { 'sg_0_6', 'sg_0_7', 'sg_0_9', 'sg_1_0' };

for i = 1:length(files)
    load( [ package_dir '/PAPER_DATA/' files{i} ] )
    x = x';
    
    Q = orth(A);
    prA = Q';
    
    Apca_2 = prA*Apca;
    A_2 = prA*A;
    x_2 = prA*x;
    
    if ~NPLFIG_4in1
        actfig( files{i} )
    else
        subplot(2,2,i)
    end
    set( gca, 'Box', 'On',...
              'NextPlot', 'Add',...
              'DrawMode', 'Fast',...
              'XLim', [ -10  10 ],...
              'YLim', [ -3 3 ] )

    set( gca, 'FontSize', 14 )

    if ~NPLFIG_4in1
        pos = [ 0.13  0.11  0.775  0.6 ];
    else
        x1 = 0.05;
        x2 = 0.52;
        axwidth = 0.775/1.8;
        axheight = 0.6/1.8;
        switch i
        case 1, pos = [ x1 0.6 axwidth axheight ];
        case 2, pos = [ x2 0.6 axwidth axheight ];
        case 3, pos = [ x1 0.1 axwidth axheight ];
        case 4, pos = [ x2 0.1 axwidth axheight ];
        end
    end
    set( gca, 'Position', pos )
    
    plot( x_2(1,:), x_2(2,:), '.', 'MarkerSize', 7 )
    
    hPCA = plot_dir( Apca_2, [-20 20], [-20 20], ...
                     'Color', [ 0 0 0 ], 'LineStyle', '--' );
    hICA = plot_dir( A_2, [-20 20], [-20 20], ...
                     'Color', [ 0 0 0 ], 'LineStyle', '-.' );
    if i == length(files)
        legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 )
    end
    
    axis equal

    set( gca, 'XTick', -10:5:10,...
              'YTick', -10:5:10 )
end

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 1.
function Tradeoff

figure

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
subplot(2,1,1)
%set( gca, 'Position', [ 0.1 0.54 0.39 0.39 ] )
y = 10;
fcontour( 'mvnpdf', -3:0.1:3, y+(-3:0.1:3), 3,...
          [ 0; y ], [ 1 0.8; 0.8 1 ] )
hold on

x = 7;
fcontour( 'mvnpdf', x+(-3:0.1:3), y+(-3:0.1:3), 3,...
          [ x; y ], [ 1.8 0; 0 0.3 ] )

set( gca',...
     'XLim',  [ -3 x+3 ],...
     'YTick', [ y ],...
     'YTickLabel', {},...
     'XTick', [ 0 x ],...
     'XTickLabel', {'ICA', 'PCA'} )
grid on
%axis equal
title('The form of the true posterior p(s(t) | A, x(t))')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
subplot(2,1,2)
%set( gca, 'Position', [ 0.51 0.07 0.39 0.39 ] )
xpts = 0:0.2:x;
%h1 = plot( xpts, exp(-0.7*xpts), '--' );
h1 = plot( xpts, 0.03*( exp(-0.5*(xpts-x)) - 1 ), 'k--' );

%text( 6.2, 0.85, '3' )
%text( 6.2, 0.25, '2' )
%text( 6.2, 0.05, '1' )

set( gca,...
     'XLim',  [ -3 x+3 ],...
     'YLim', [ -0.1 1.5 ],...
     'YTick', [ 0 ],...
     'YTickLabel', {},...
     'XTick', [ 0 x ],...
     'XTickLabel', {'ICA', 'PCA'} )
grid on
hold on
h2 = plot( xpts, (exp(xpts*0.5)-1)*0.03, 'k',...
           xpts, (exp(xpts*0.5)-1)*0.015, 'k',...
           xpts, (exp(xpts*0.5)-1)*0, 'k' );
title( [ 'The cost of the posterior '...
         'and source model misfit' ] )

text( 7.5, 0.05, 'b_2 = b_1' )
text( 7.5, 0.4, 'b_2 < b_1' )
text( 7.5, 0.9, 'b_2 << b_1' )

h = legend( [ h1 h2(1) ],...
            'Cost of posterior misfit',...
            'Cost of source model misfit', 2 );

pos = get(h,'Position');
pos(1) = 0.345;
%set( h, 'Position', pos )

return



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ hPCA, hICA ] = Plot1Track( netfile, datafile,...
                                      show_iter, snum, squareax )

global NPLFIG_4in1

if nargin < 5
    squareax = 0;
end
if nargin < 4 | ~NPLFIG_4in1
    snum = [];
end
if nargin < 3
    show_iter = [];
end

load( [ package_dir '/PAPER_DATA/' datafile ] )
prA = orth(A)';
Apca_2 = prA * Apca;
A_2 = prA * A;
%[ h, A, prA, A_2, Apca_2 ] = plotarfa( datafile, [ 0 0 0 ] );

load( [ package_dir '/PAPER_RES/' netfile ], 'net', 'nit', 'CF' )

fprintf( '\nAvailable iterations:' )
DisplayIterations(nit)

if ~isempty(show_iter)

    % C = nit(IA), C = show_iter(IB)
    [C,IA,IB] = intersect( nit, show_iter );
    
    % The displayed iterations:
    fprintf( 'Displayed iterations:' )
    DisplayIterations(C)
    
    % nit_show = C(I)
    [nit_show,I] = sort(C);
    %nit(IA(I))
    % nit_show = C(I) = nit(IA(I))
    
    net = { net{IA(I)} };
end

if isempty(snum)
    actfig( netfile )
else
    subplot(2,2,snum)
end
    

set( gca, 'Box', 'On',...
          'NextPlot', 'Add',...
          'XLim', [ -0.01 0.01 ],...
          'YLim', [ -0.01 0.01 ] )

set( gca, 'FontSize', 14 )

h = plot_track( net, prA, A_2, Apca_2, gca, 15 );

if squareax
    axis square

else
    xlims = get( gca, 'XLim' );
    ylims = get( gca, 'YLim' );
    axis normal

    xlims = [ min( xlims(1)-0.5, -0.2 ), xlims(2) + 0.2 ];
    ylims = [ min( ylims(1)-0.5, -0.2 ), ylims(2) + 0.2 ];

    if isempty(snum)
        pos = [ 0.13  0.11  0.775  0.6 ];
    else
        x1 = 0.05;
        x2 = 0.52;
        axwidth = 0.775/1.8;
        axheight = 0.6/1.8;
        switch snum
        case 1, pos = [ x1 0.6 axwidth axheight ];
        case 2, pos = [ x2 0.6 axwidth axheight ];
        case 3, pos = [ x1 0.1 axwidth axheight ];
        case 4, pos = [ x2 0.1 axwidth axheight ];
        end
    end
    set( gca, 'Position', pos )

    set( gca, 'XLim', xlims, 'YLim', ylims )
    axis equal
    
end


hPCA = plot_dir( Apca_2, [-10 10], [-10 10], ...
                 'Color', [ 0 0 0 ], 'LineStyle', '--' );
hICA = plot_dir( A_2, [-10 10], [-10 10], ...
                 'Color', [ 0 0 0 ], 'LineStyle', '-.' );


set( gca, 'XTick', -2:0.5:5,...
          'YTick', -2:0.5:5 )
set( gca, 'XTickLabel', [], 'YTickLabel', [] )


return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function DisplayIterations(C)

str = '';
n1 = [];
ndiff = [];

for i = 1:length(C)
    if isempty(n1)
        n1 = C(i);
        if i == length(C)
            str = sprintf( '%s %d', str, C(i) );
        end
    elseif isempty(ndiff)
        ndiff = C(i)-n1;
        if i == length(C)
            str = sprintf( '%s %d %d', str, n1, C(i) );
        end
    else
        if C(i) ~= C(i-1)+ndiff
            if C(i-1) == n1 + ndiff
                str = sprintf( '%s %d', str, n1 );
            elseif C(i-1) == n1 + 2*ndiff
                str = sprintf( '%s %d %d',...
                               str, n1, C(i-2) );
            else
                str = sprintf( '%s %d:%d:%d',...
                               str, n1, ndiff, C(i-2) );
            end
            n1 = C(i-1);
            ndiff = C(i)-C(i-1);
        elseif i == length(C)
            str = sprintf( '%s %d:%d:%d',...
                           str, n1, ndiff, C(i) );
        end
    end
end

disp(str)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 11.1. ICA with super-Gaussian sources:
%       Full approximation, PCA initialization
function fullqs_sg_pca

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'Super-Gaussian sources, full, PCA' )
end

Plot1Track( 'Full_Qs/sg_0_6__PCA', 'sg_0_6', [], 1 );
Plot1Track( 'Full_Qs/sg_0_7__PCA', 'sg_0_7', [], 2 );
Plot1Track( 'Full_Qs/sg_0_9__PCA', 'sg_0_9', [], 3 );
[ hPCA, hICA ] = Plot1Track( 'Full_Qs/sg_1__PCA', 'sg_1_0', [], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 11.2. ICA with super-Gaussian sources:
%       Full approximation, ICA initialization
function fullqs_sg_ica

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'Super-Gaussian sources, full, ICA' )
end

Plot1Track( 'Full_Qs/sg_0_6__ICA', 'sg_0_6', [], 1 );
Plot1Track( 'Full_Qs/sg_0_7__ICA', 'sg_0_7', [], 2 );
Plot1Track( 'Full_Qs/sg_0_9__ICA', 'sg_0_9', [], 3 );
[ hPCA, hICA ] = Plot1Track( 'Full_Qs/sg_1__ICA',...
                             'sg_1_0', [], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );

return


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 12.1. ICA with mixture-of-Gaussians model for sources:
%       Full approximation, PCA initialization
function fullqs_mog_pca

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'MoG sources, full, PCA' )
end

Plot1Track( 'Full_Qs/mog__sg_0_6__PCA', 'sg_0_6', [], 1 );
Plot1Track( 'Full_Qs/mog__sg_0_7__PCA', 'sg_0_7', [], 2 );
Plot1Track( 'Full_Qs/mog__sg_1__PCA', 'sg_1_0', [], 3 );
[ hPCA, hICA ] = Plot1Track( 'Full_Qs/mog__sg_1_2__PCA',...
                             'sg_1_2', [], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );
return


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 13.1. ICA with mixture-of-Gaussians model for sources:
%       Diagonal approximation with MoG posterior, PCA initialization
function diag_mogifa_pca

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'MoG sources, diagonal with MoG posterior, PCA' )
end

Plot1Track( 'BBpython/ifa__sg_0_5__PCA',...
            'sg_0_5', [ 10 0:2000:10000 ], 1 );
Plot1Track( 'BBpython/ifa__sg_0_7__PCA',...
            'sg_0_7', [ 10 0:2000:10000 ], 2 );
Plot1Track( 'BBpython/ifa__sg_1__PCA',...
            'sg_1_0', [ 10 0:2000:25000 ], 3 );
[ hPCA, hICA ] = Plot1Track( 'BBpython/ifa__sg_1_2__PCA',...
                             'sg_1_2', [ 10 0:2000:30000 ], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );
return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 13.2. ICA with mixture-of-Gaussians model for sources:
%       Diagonal approximation with MoG posterior, PCA initialization
function diag_mogifa_ica

global NPLFIG_4in1
if NPLFIG_4in1
   actfig( 'MoG sources, diagonal with MoG posterior, ICA' )
end

Plot1Track( 'BBpython/ifa__sg_0_5__ICA', 'sg_0_5', [], 1 );
Plot1Track( 'BBpython/ifa__sg_0_7__ICA', 'sg_0_7', [], 2 );
Plot1Track( 'BBpython/ifa__sg_1__ICA', 'sg_1_0', [], 3 );
[ hPCA, hICA ] = Plot1Track( 'BBpython/ifa__sg_1_2__ICA',...
                             'sg_1_2', [], 4 );
legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 );

return

