# Test Script for
# Elementwise Sparse Group Factor Analysis
# Suleiman Ali Khan, suleiman.khan@aalto.fi
# Seppo Virtanen, seppo.j.virtanen@aalto.fi

#
# Toy Data Generation and GFA run
#
# Toy data is generated for 2 views and 4 components
# 2 components are shared in both views (sine, cosine)
# 1 component is specific to 1st dataset/view (linear)
# 1 component is specific to 2nd dataset/view (random gaussian)
# Total 4 Components used to create the data
# This is followed by running the model on the data and plotting the model estimated
# latent components. The model finds the correct components.
#

#create byte code for speed up
library("compiler")

# Read in the code
source('gfa.R')

#load library and functions for plotting
library("gplots")
source("heatmap.2.flataxis.R")

#
# Generate the toy data with pre-specified
# latent components. 

N <- 100 # Number of samples (you can try also
	 #   smaller and larger values)
D <- c(15,20)             # Data dimensions
K <- 4
Z <- matrix(0,N,K)       # Latent components
Z[,1] <- sin((1:N)/(N/20)) #sin
Z[,2] <- cos((1:N)/(N/20)) #cos
Z[,3] <- rnorm(N,0,1)	     #random normal
Z[,4] <- as.matrix(c(2*((1:N)/N-0.5)),N,1) #linear
tau <- c(3,6)             # Noise precisions
alpha <- matrix(0,2,4)    # Component precisions for the two data sets
alpha[1,] <- c(1,1,1e6,1) # 1   = active
alpha[2,] <- c(1,1,1,1e6) # 1e6 = inactive

# Create some random projection vectors and sample data from the
# model.
Y <- vector("list",length=2)
W <- vector("list",length=2)
for(view in 1:2) {
  W[[view]] <- matrix(0,D[view],K)
  for(k in 1:K) {
    W[[view]][,k] <- rnorm(D[view],0,1/sqrt(alpha[view,k]))
  }
  Y[[view]] <- Z %*% t(W[[view]]) +
    matrix(rnorm(N*D[view],0,1/sqrt(tau[view])),N,D[view])
}

#
# Run GFA
#
opts <- getDefaultOpts(Y)
opts$verbose <- 1
print("Training the model")
print("==================")
model <- gfa(Y,K,opts)
model$res <- getPosteriorEV(model)
print("==================")
print("Model Run Complete")
print("==================")
Sys.sleep(1)
#
# Explore the results
#
print("Plot the True Components")
print("==================")
# Draw the latent components
#
# Note that the model is invariant to the sign of the latent
# components, so they might end up in a different direction.
#
# Also the component ordering could be different.
#
X11(); par(mfrow=c(K,1), oma=c(0,0,2,0))
for(i in 1:K) {
  plot(Z[,i],ylim=c(-2,2),xlab="Samples (N)",ylab="latent scores");
}
par(mfrow=c(1,1));
title(main="True latent components",outer=TRUE)

print("Press Enter to Proceed")
browser()
print("Plot the Estimated Components")
print("==================")
print("Note that the model is invariant to the sign of the latent components, so they might end up in a different direction. Also the component ordering can be differnt from that in the True latent components")
X11(); par(mfrow=c(K,1), oma=c(0,0,2,0))
for(i in 1:K) {
  plot(model$res$X[,i],ylim=c(-2,2),xlab="Samples (N)",ylab="latent scores",main=i);
}
par(mfrow=c(1,1));
title(main="Estimated latent components",outer=TRUE)

print("Press Enter to Proceed")
browser()
print("Plot the Activations")
print("==================")
X11(); par(oma=c(0,0,2,0))
rownames(model$res$Z) <- paste("Dataset",1:nrow(model$res$Z))
heatmap.2.flataxis(t(model$res$Z),col=c("white","black"),dendrogram="none",trace="none",main="Component Activation Plot",cexCol=1.5, cexRow=1.5, xlab="Views", ylab="Components");
print("=====DEMO END=====")

