sGFA <- function(Y, K, opts, W=NULL, rz=NULL, tau=NULL, beta=NULL) {

	## Sparse GFA
	# Author: Seppo Virtanen, seppo.j.virtanen@aalto.fi
	# Modifications: Tommi Suvitaival, tommi.suvitaival@aalto.fi
	#
	# Description:
	# Model is Y = XW^T + E, where W is sparse group sparse projection matrix of size D\times K 
	# and X is sparse matrix containing the latent variables, N\times K. Each column of Y has 
	# separate noise parameter. The view-specific W:s are obtained by model$W[opts$groups[[m]],]

	## License

	# This program is free software: you can redistribute it and/or modify
	# it under the terms of the GNU General Public License as published by
	# the Free Software Foundation, either version 3 of the License, or
	# (at your option) any later version.
	# 
	# This program is distributed in the hope that it will be useful,
	# but WITHOUT ANY WARRANTY; without even the implied warranty of
	# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	# GNU General Public License for more details.
	# 
	# You should have received a copy of the GNU General Public License
	# along with this program.  If not, see <http://www.gnu.org/licenses/>.

  gr <- opts$groups
  prior.betaW1 <- opts$prior.betaW1
  prior.betaW2 <- opts$prior.betaW2
  prior.betaX1 <- opts$prior.betaX1
  prior.betaX2 <- opts$prior.betaX2
  alpha_0 <- opts$prior.alpha_0
  beta_0 <- opts$prior.beta_0
  alphaZ <- opts$prior.alphaZ
  
  #
  # Store dimensionalities of data sets 
  #
  D <- ncol(Y)
  N <- nrow(Y)
    
  # Some constants for speeding up the computation
  id <- rep(1,K) # Vector of ones for fast matrix calculations
 
  # Latent variables
  X <- matrix(rnorm(N*K,0,1),N,K) 
  X <- scale(X)
  covZ <- diag(1,K) # The covariance
  XX <- crossprod(X) # The second moments
  
	W <- matrix(0,D,K)
	WtW <- matrix(0,K,K)
	WtWdiag <- rep(0,K)
	R <- Rinv <- rep(0,K)
	tau <- rep(opts$init.tau,D) # The mean noise precisions
	r <- matrix(0.9,D,K)
	rz <- matrix(0.9,N,K)

	if (opts$iter.saved.max>0) { # Save the Gibbs samples after burn-in.
		if (is.null(opts$iter.burnin)) {
			opts$iter.burnin = 0
		}
		mod.posterior = ceiling((opts$iter.max-opts$iter.burnin)/opts$iter.saved.max) # Save every 'mod.posterior'th Gibbs sample
		posterior <- list()
		posterior$W <- array(dim=c(floor((opts$iter.max-opts$iter.burnin)/mod.posterior), dim(W))) # iterations x dimensions of W -11.4.13
		colnames(posterior$W) = colnames(Y)
		posterior$rz <- array(dim=c(floor((opts$iter.max-opts$iter.burnin)/mod.posterior), ncol(rz))) # Same 'rz' for all samples. Thus, save only a vector per a Gibbs sample.
		posterior$r <- array(dim=c(floor((opts$iter.max-opts$iter.burnin)/mod.posterior), K, length(gr))) # Same 'r' for all variables of a view. Thus, save only a components x views matrix per a Gibbs sample.
		gr.start = vector(mode="integer", length=length(gr))
		for (m in 1:length(gr)) {
			gr.start[m] = gr[[m]][1]
		}
		posterior$tau <- array(dim=c(floor((opts$iter.max-opts$iter.burnin)/mod.posterior),length(tau)))
		colnames(posterior$tau) = colnames(Y)
	} else {
		posterior = NULL
	}

	Yconst <- colSums(Y^2)

  covW <- diag(1,K)
  
  alpha <- matrix(1,D,K)
  beta <- matrix(1,N,K)

  lambda <- mu <- zone <- rep(1,D)
  
  ##
  ## The main loop
  ##
  for(iter in 1:opts$iter.max){
		if (opts$verbose==2) {
			print(paste(iter, "/", opts$iter.max, "-", Sys.time()))
		}
		if (iter==1) {
			covW <- diag(1,K) + opts$init.tau*XX
			eS <- eigen(covW)
			covW <- tcrossprod(eS$vectors*outer(id,1/sqrt(eS$values)))
			W <- crossprod(Y,X)%*%covW*opts$init.tau + matrix(rnorm(D*K),D,K)%*%( eS$vectors*outer(id,1/sqrt(eS$values)) )
		} else {
			XXdiag <- diag(XX)
			diag(XX) <- 0
			for (k in 1:K) {
				lambda <- tau*XXdiag[k] + alpha[,k]
				mu <- tau/lambda*as.vector( crossprod(Y,X[,k]) - W%*%XX[k,])
				zone <- 0.5*( log(alpha[,k]) - log(lambda) + lambda*mu^2) + log(r[,k]) - log(1-r[,k])
				zone <- 1/(1+exp(-zone))
				zone <- as.double(runif(D) < zone)
				W[,k] <- mu + rnorm(D)*lambda^(-0.5)
				W[,k] <- W[,k]*zone
				for (m in 1:length(gr)) {
					zm <- sum(zone[ gr[[m]] ])
					r[gr[[m]],k] <- rbeta(1,shape1=zm+prior.betaW1,shape2=length(gr[[m]])-zm+prior.betaW2)
				}
				if (sum(zone)==1) {
					W[,k] <- 0
				}
			}
		}
               
    ## Update the latent variables

    if (iter>1) {
			WtW <- crossprod(W*sqrt(tau))
			WtWdiag <- diag(WtW)
			diag(WtW) <- 0
      for (k in 1:K) {
        lambda <- WtWdiag[k] + beta[,k]
        mu <- (Y%*%(W[,k]*tau) - X%*%WtW[k,])/lambda
        zone <- 0.5*( log(beta[,k]) - log(lambda) + lambda*mu^2) + log(rz[,k]) - log(1-rz[,k])
        zone <- 1/(1+exp(-zone))
        zone <- as.double(runif(N) < zone)
        X[,k] <- mu + rnorm(N)*lambda^(-0.5)
        X[,k] <- X[,k]*zone
				zm <- sum(zone)
				rz[,k] <- rbeta(1,shape1=zm+prior.betaX1,shape2=N-zm+prior.betaX2)
				R[k] <- sqrt(crossprod(X[zone==1,k])/zm)
				if (zm<=1) {
					R[k] <- 0
					Rinv[k] <- 0
				} else {
					Rinv[k] <- 1/R[k]
				}
      }
    } else {
      covZ <- diag(1,K) + crossprod(W*sqrt(tau))
      eS <- eigen(covZ)
      covZ <- tcrossprod(eS$vectors*outer(id,1/sqrt(eS$values)))
      X <- Y%*%(W*tau)
      X <- X%*%covZ + matrix(rnorm(N*K),N,K)%*%( eS$vectors*outer(id,1/sqrt(eS$values)) )
			R <- sqrt(colMeans(X^2))
			Rinv <- 1/R
    }

		X <- X*outer(rep(1,N),Rinv)
		W <- W*outer(rep(1,D),R)
		XX <- crossprod(X)

    ## Update alpha, the ARD parameters

    beta <- X^2/2
    keep <- which(beta!=0)
    for (n in keep) {
			beta[n] <- rgamma(1,shape=alphaZ+0.5,rate=alphaZ+beta[n])
		}
    beta[-keep] <- rgamma( length(beta[-keep]),shape=alphaZ,rate=alphaZ) + 1e-7
    alpha <- W^2/2
    keep <- which(alpha!=0)
    for (n in keep) {
			alpha[n] <- rgamma(1,shape=alpha_0+0.5,rate=beta_0+alpha[n])
		}
    alpha[-keep] <- 1e-7
    alpha[-keep] <- rgamma( length(alpha[-keep]),shape=alpha_0,rate=beta_0) + 1e-7
    
    ## Update tau, the noise precisions

		tau <- ( Yconst + rowSums((W%*%XX)*W) - 2*rowSums(crossprod(Y,X)*W) )/2
		for(d in 1:D) {
			tau[d] <- rgamma(1,shape=a_tau, rate= b_tau + tau[d])
		}

		if (opts$iter.saved.max>0) { ##
			if (iter>opts$iter.burnin & ((iter-opts$iter.burnin)%%mod.posterior)==0) { # Save the Gibbs sample. -11.4.13
				posterior$W[(iter-opts$iter.burnin)/mod.posterior,,] <- W
				posterior$tau[(iter-opts$iter.burnin)/mod.posterior,] <- tau
				posterior$corX = posterior$corX+cor(t(X))/(opts$iter.max-opts$iter.burnin)*mod.posterior
				posterior$rz[(iter-opts$iter.burnin)/mod.posterior,] <- rz[1,] # In the current model, 'rz' is identical for all samples (rows are identical). Thus, save only a vector.
				posterior$r[(iter-opts$iter.burnin)/mod.posterior,,] <- r[gr.start,] # In the current model, 'r' is identical for all variables within a view. Thus, save only a views x components vector.
			}
		}

  } ## The main loop of the algorithm ends.

  rem <- unique(c(which(colSums(X^2)==0),which(colSums(W^2)==0)))
  if (length(rem > 0)) {
    X[,rem] <- 0; W[,rem] <- 0
  }

  ## Return the output of the model as a list.

	return(list(W=W, X=X, tau=tau, alpha=alpha, beta=beta, rz=rz, D=D, K=K, posterior=posterior))

}

sGFA <- cmpfun(sGFA)


getDefaultOptsGFA <- function() {

	## Function for acquiring default settings for the Gibbs sampler for
	## group factor analysis with sparsity.

	# Author: Seppo Virtanen, seppo.j.virtanen@aalto.fi
	# Modifications: Tommi Suvitaival, tommi.suvitaival@aalto.fi

	## License

	# This program is free software: you can redistribute it and/or modify
	# it under the terms of the GNU General Public License as published by
	# the Free Software Foundation, either version 3 of the License, or
	# (at your option) any later version.
	# 
	# This program is distributed in the hope that it will be useful,
	# but WITHOUT ANY WARRANTY; without even the implied warranty of
	# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	# GNU General Public License for more details.
	# 
	# You should have received a copy of the GNU General Public License
	# along with this program.  If not, see <http://www.gnu.org/licenses/>.
  
	opts = list()

  ## Hyperparameters
  # - alpha_0, beta_0 for the ARD precisions
  # - alpha_0t, beta_0t for the residual noise predicions
  #
	opts$prior.alphaZ <- 1e-7
	opts$prior.alpha_0 <- 1e-7
	opts$prior.beta_0 <- 1e-7
	a <- 1e-2
	b <- 0.5
	opts$prior.betaW1 <- a*b
	opts$prior.betaW2 <- a*(1-b)
	a <- 1e-2
	b <- 0.5
	opts$prior.betaX1 <- b*a
	opts$prior.betaX2 <- (1-b)*a

	## Iterations

	opts$iter.max <- 1000 # Total number of iterations
	opts$iter.burnin <- 500 # Number of burn-in iterations out of all iterations.
	opts$iter.saved.max <- 0 # Maximum number of iterations after burn-in that are saved.

  ## Verbosity level
  #  0: Nothing
  #  1: Final cost function value for each run of gsCCAexperiment()
  #  2: Cost function values for each iteration
  #
  verbose <- 2
  
  return(opts)

}

