# Source the file and call demo() when rMTF is in your working directory
# Author: Eemeli Leppäaho, eemeli.leppaaho@aalto.fi
demo <- function() {
  if(!file.exists("rMTF.R"))
    stop("rMTF.R needed in the working directory")
  source("rMTF.R")
  
  N <- 300 #Number of samples in the data
  D <- 50 #Dimensionality of data views
  M <- 31 #Number of data views: 1 matrix + M-1 depth tensor
  K <- 15 #Number of components used to run the models (true=13)
  dd <- 1:25 #Which features are visualized
  
  #Data generation
  gendata <- genData(similarity=1,k=c(1,8,2),N=N,D=D,M=M,sine=T,verbose=F)
  X <- gendata$Y
  
  #Load options
  opts <- getDefaultOpts()
  opts$ARDW <- "grouped" #ARD (alpha) is MxK
  opts$normalLatents <- TRUE #N(0,1) distributed latent variables
  opts$iter.max <- 1200 # Total number of iterations
  opts$iter.burnin <- 1000 # Number of burn-in iterations before starting to save Gibbs samples
  opts$iter.saved <- 20 # Number of Gibbs samples saved after the burn-in
  opts$save.posterior <- list(X=T,W=T,tau=T) #Save X, W and tau posterior
  opts$prediction <- rep(FALSE, length(X)) #No prediction done
  opts$verbose <- 1
  
  #Running the model for 1 matrix + 1 tensor
  res <- rMTF(X,K=K,opts,sharing=c(0,rep(1,M-1)))
  print("rMTF run")
  #Common components between the matrix and the tensor
  k <- which(apply(res$W[1:D,],2,function(x){any(x!=0)}) & apply(res$W[-1:-D,],2,function(x){any(x!=0)}))
  if(length(k)>1) {
    print(paste(length(k),"shared components detected, visualizing the first"))
    k <- k[1]
  } else if(length(k)==0) {
    stop("No shared components detected")
  }
  #Compute the posterior mean for shared components in matrix and 1st tensor slab
  w1 <- w2 <- rep(0,D)
  Npost <- nrow(res$posterior$W)
  for(i in 1:Npost) {
    W <- res$posterior$W[i,,]
    X <- res$posterior$X[i,,]
    w1 <- w1 + W[(D*1+1):(D*2),k,drop=F]*sd(X[,k])/Npost
    w2 <- w2 + W[(D*2+1):(D*3),k,drop=F]*sd(X[,k])/Npost
  }
  sign <- 1
  if(mean((w1-gendata$w1)^2) > mean((w1+gendata$w1)^2))
    sign <- -1
  
  par(mfrow=c(2,1),mar=c(1,4.1,1,1),oma=c(4,0,0,0),xpd=F)
  Lwd <- 3
  lwd <- 1.5
  
  print("Weights with a sine shape")
  ### 1st plot ###
  ## Plot the true weights
  plot(gendata$w1[dd]*gendata$u[1,1],xlab="",ylab="Weight",type="l",xaxt="n",lwd=Lwd)
  lines(gendata$w2[dd]*gendata$u[2,1],lty=2,lwd=Lwd)
  ## Plot the rMTF weights
  lines(sign*w1[dd],col="blue",lwd=lwd)
  lines(sign*w2[dd],col="blue",lty=2,lwd=lwd)
  
  print("Added distrotion/relaxation to the weights")
  gendata <- genData(similarity=1,k=c(1,8,2),N=N,D=D,M=M,sine=T,sine2=T,verbose=F)
  X <- gendata$Y
  
  #Running the model for 1 matrix + 1 tensor
  res <- rMTF(X,K=K,opts,sharing=c(0,rep(1,M-1)))
  print("rMTF run")
  #Common components between the matrix and the tensor
  k <- which(apply(res$W[1:D,],2,function(x){any(x!=0)}) & apply(res$W[-1:-D,],2,function(x){any(x!=0)}))
  if(length(k)>1) {
    print("More than one shared component detected, visualizing the first")
    k <- k[1]
  } else if(length(k)==0) {
    stop("No shared components detected")
  }
  #Compute the posterior mean for shared components in matrix and 1st tensor slab
  w1 <- w2 <- rep(0,D)
  Npost <- nrow(res$posterior$W)
  for(i in 1:Npost) {
    W <- res$posterior$W[i,,]
    X <- res$posterior$X[i,,]
    w1 <- w1 + W[(D*1+1):(D*2),k,drop=F]*sd(X[,k])/Npost
    w2 <- w2 + W[(D*2+1):(D*3),k,drop=F]*sd(X[,k])/Npost
  }
  sign <- 1
  if(mean((w1-gendata$w1)^2) > mean((w1+gendata$w1)^2))
    sign <- -1
  
  ### 2nd plot ###
  ## Plot the true weights
  plot(gendata$w1[dd]*gendata$u[1,1],xlab="Feature",ylab="Weight",type="l",lwd=Lwd)
  lines(gendata$w2[dd]*gendata$u[2,1],lty=2,lwd=Lwd)
  ## Plot the rMTF weights
  lines(sign*w1[dd],col="blue",lwd=lwd)
  lines(sign*w2[dd],col="blue",lty=2,lwd=lwd)
  
  mtext("Feature",side=1,line=2,outer=T)
  par(xpd=NA)
  legend(2,1.45,c("True","rMTF"),lty=1,col=c("black","blue"),
         ncol=2,bty="n",text.width=3,lwd=c(Lwd,lwd))
  par(xpd=F)
  
  #TODO show also a prediction setup
}

genData <- function(similarity=1, k=c(1,8,2), N=500, M=10, D=20, tau=1, strength=1,
                    sine=F, sine2=F, verbose=T) {
  # Generate (r)MTF data
  D <- rep(D,M)
  kType <- c("fully shared","tensor-specific","matrix specific")
  kk <- vector("list",length=3)
  K <- 0
  for(i in 1:length(k)) {
    if(k[i]>0) {
      kk[[i]] <- 1:k[i]+K
      K <- max(kk[[i]])
    }
  }
  K <- sum(k)
  
  if(verbose) {
    print("Generating data with the following components:")
    for(i in 1:length(k))
      print(paste0(" ",k[i]," ",kType[i]))
  }
  
  
  alpha <- matrix(Inf,M,K)
  alpha[,kk[[1]]] <- 1/strength^2 #fully shared
  alpha[-1,kk[[2]]] <- 1 #tensor
  alpha[1,kk[[3]]] <- 1 #side-specific
  
  Z <- matrix(rnorm(K*N), K, N)
  Y <- noise <- mu <- vector("list",length=M)
  
  #Data generation for the matrix
  for(m in 1) {
    W <- matrix(NA, D[m], K)
    for(k in 1:K) {
      W[,k] <- rnorm(D[m], 0, 1/sqrt(alpha[m,k]))
    }
    noise[[m]] <- matrix(rnorm(N*D[m],0,1/sqrt(tau)), N, D[m])
    Y[[m]] <- t(W%*%Z) + noise[[m]]
  }
  
  #Data generation for the tensor
  w <- matrix(NA,D[1],K)
  u <- matrix(NA,M-1,K)
  for(k in 1:K) {
    w[,k] <- rnorm(D[2], 0, 1/sqrt(alpha[2,k]))
    u[,k] <- rnorm(M-1, 0, 1)
  }
  if(sine) {
    Z[1,] <- Z[1,]/sd(Z[1,]) #Shared component with exact 1 variance
    w[,1] <- sin((1:D[2]-1)/(D[2]-1)*4*pi)/sqrt(alpha[2,1]) #Sine wave
    u[1:2,1] <- c(1,-0.8) #3rd dimension weight for first two slabs
  }
  
  for(m in 2:M) {
    noise[[m]] <- matrix(rnorm(N*D[m],0,1/sqrt(tau)), N, D[m])
    Y[[m]] <- noise[[m]]
    for(k in 1:K) {
      ww <- similarity*w[,k] + (1-similarity)*rnorm(D[m], 0, 1/sqrt(alpha[2,k]))
      if(sine2 & k==1) { #Add deviance to the tensor model
        dev <- 1:D[2]
        const <- c(NA, 1.5, 0.5, seq(0.3,1.7,length=M-3))[m]
        ww[dev] <- sign(ww[dev])*abs(ww[dev])^const
      }
      if(m==2 & k==1) {w1 <- ww}
      if(m==3 & k==1) {w2 <- ww}
      Y[[m]] <- Y[[m]] + outer(Z[k,],ww)*u[m-1,k] #Proper tensor!
    }
  }
  
  foo <- list(Y=Y,D=D,w=w,u=u,w1=w1,w2=w2,Z=Z,alpha=alpha,noise=noise,tau=rep(tau,M))
  foo$similarity <- similarity
  
  return(foo)
}
