package jmprojection;

import java.util.Random;

/*
 * Created on 1.7.2005
 *
 * TODO To change the template for this generated file go to
 * Window - Preferences - Java - Code Style - Code Templates
 */
/**
 * Superclass of all mappings used in MegNet.
 * @author Jarkko Miettinen
 *
 */
public abstract class AbstractMapping{

	
	protected ProjectionStatus projectionStatus = new ProjectionStatus();
	
	protected final static Random R = new Random(System.currentTimeMillis());
	/**
	 * Field holding the number of iterations this mapping algorithm has
	 * performed so far. Meaningless if the algorithm isn't iterative.
	 */
	protected int iterations;
	/**
	 * Name of this mapping algorithm
	 */
	public final static String NAME_OF_MAPPING = "Generic AbstractMapping";

	/**
	 * Creates a new mapping. All subclasses should override this constructor
	 * and create new constructors for other parameters if needed.
	 */
	public AbstractMapping(double[][] data) {}

	/**
	 * Returns coordinates in projection space, which is in this case two
	 * dimensions.
	 * @return <code>double[2][N]</code> 2 x N matrix, where N is the number of
	 * data-vectors originally given to algorithm as the input.
	 */
	abstract public double[][] getState();

	/**
	 * Return some sort of mapping error which should inform, for example, how
	 * close distances in projection space and original space are to each other.
	 * @return the error
	 */
	abstract public double mappingError();

	/**
	 * Iterates mapping one step if the mapping is iterative method. If it is
	 * not, this should do nothing.
	 */
	abstract public void iterate();

	/**
	 * Calculates Sammon's stress by following formula:<br><pre>
	 * (1 / c) * Sum((d*(i,j) - d(i,j))^2/d*(i,j), i = 1..n-1, j = i + 1..n)
	 *
	 * where c = Sum(d*(i,j), i = 1..n-1, j = i + 1..n),
	 * d*(i,j) = Euclidean distance between points i and j in original space
	 * d(i,j)  = Euclidean distance between points i and j in projection space
	 * </pre>
	 * @return said Sammon's Stress
	 */
	public abstract double sammonsStress();

	/**
	 * Tells number of iteration steps taken so far is applicable.
	 * @return number of iteration-steps taken so far
	 */
	public int iterations(){
		return iterations;
	}

	/**
	 * Iterates n steps if mapping method is iterative. If it is not, this
	 * should do nothing.
	 * @param n number of times to iterate
	 */
	public void iterate(int n) {
		for (int i = 0; i < n;i++){
			projectionStatus.setFinishedPercentage( (float)i / (float)n );
			iterate();
		}
	}

	/**
	 * Returns name of this mapping method. All subclasses should override this
	 * if they don't want to identify themselves as "Generic AbstractMapping".
	 * @return name of this mapping method
	 */
	public String getName() {
		return NAME_OF_MAPPING;
	}

	/**
	 * Calculates euclidean distance between two vectors. If second vector (<B>
	 * y</B>) is shorter than first vector (<B>x</B>) then this will throw
	 * ArrayIndexOutOfBoundsException.
	 * @param x first vector
	 * @param y second vector
	 * @return euclidean distance between vectors x and y
	 */
	protected double dist(double[] x, double[] y){
		double dist = 0.0;
		double temp;
		for(int i = 0; i < x.length;i++){
			temp = x[i] - y[i];
			dist += temp*temp;
		}
		return Math.sqrt(dist);
	}

	public abstract String getParameters();

	/**
	 * Generates array containing random permutation with numbers between
	 * 0 ... len-1 So it is suitable for picking entries from array in random
	 * fashion.
	 * @param len first number not included in array, should be > 0
	 * @return
	 */
	protected int[] genRandPermutation(final int len)
	{
		int permArray[];
		int rnumber;
		int temp;

		permArray = new int[len];
		for (int i = 0; i < len; i++)
		{
			permArray[i] = i;
		}
		for (int i = 0; i < len; i++)
		{
			rnumber = R.nextInt(len);
			temp = permArray[rnumber];
			permArray[rnumber] = permArray[i];
			permArray[i] = temp;
		}
		return permArray;
	}

	public ProjectionStatus getProjectionStatus() {
		return projectionStatus;
	}
}
