package jmprojection;

import java.util.Iterator;

/**
 * This class implements Dijkstra's shortest path algorithm with <code>double
 * </code>
 * weights. It can be used with adjacencylists and with matrices.
 *
 * @author Jarkko Miettinen
 */
public class ShortestPathDijkstra
{
	private AdjacencyList adjacencyList;

	private boolean shortestPathsCalculated;

	private double maxDistance = 0.0;

	private FibHeap border;

	private boolean dijkstraInitialized;

	private FibNode[] fibNodes;

	private double[][] shortestPathsMatrix;

	private double[][] conMatrix;

	private boolean usingConnectivityMatrix;

	private int errorNumber;

	public static final int NO_CONNECTIONS_BETWEEN_NODES = 1;

	public ShortestPathDijkstra(double[][] connectivityMatrix)
	{
		this.conMatrix = connectivityMatrix;
		this.usingConnectivityMatrix = true;
	}

	public ShortestPathDijkstra(AdjacencyList adList)
	{
		this.adjacencyList = adList;
	}

	private boolean isSquare(double[][] matrix)
	{
		boolean isSquare = true;
		for (int i = 0, len = matrix.length; i < len; i++)
		{
			if (matrix[i].length != len)
			{
				isSquare = false;
				break;
			}
		}
		return isSquare;
	}

	/**
	 * @param includeInfinity
	 *            should we include Double.POSITIVE_INFINITY in the return
	 *            matrix or should we replace all it's instances with 20 *
	 *            maximum distance found
	 * @return double[][] matrix describing the smallest distances from a to b
	 *         and vice versa. The returned matrix isn't symmetric so distance
	 *         from a to b may differ of distance from b to a. Returned matrix
	 *         is the same as the one used by this object so if you modify it
	 *         and use allPairShortestPaths again without first calling
	 *         forceRecalculation(), you get the same modified one you already
	 *         have.
	 */
	public double[][] allPairShortestPaths(boolean includeInfinity)
	{
		return allPairShortestPaths(false, includeInfinity);
	}

	/**
	 * @param makeSymmetric
	 *            should we make this matrix symmetric by replacing larger of
	 *            distances d(a,b), d(b,a) with the smaller.
	 * @param includeInfinity
	 *            should we include Double.POSITIVE_INFINITY in the return
	 *            matrix or should we replace all it's instances with 20 *
	 *            maximum distance found
	 * @return double[][] matrix describing the smallest distances from a to b
	 *         and vice versa. The returned matrix may or may not be symmetric
	 *         so distance from a to b may differ of distance from b to a.
	 *         Returned matrix is the same as the one used by this object so if
	 *         you modify it and use allPairShortestPaths again without first
	 *         calling forceRecalculation(), you get the same modified one you
	 *         already have.
	 */
	public double[][] allPairShortestPaths(boolean makeSymmetric,
			boolean includeInfinity)
	{
		double[][] distanceMatrix;
		if (!this.shortestPathsCalculated)
		{
			this.maxDistance = 0.0;
			if (usingConnectivityMatrix)
			{
				int len = this.conMatrix.length;
				distanceMatrix = new double[len][];
				for (int i = 0; i < len; i++)
				{
					distanceMatrix[i] = shortestPathUsingConnectivityMatrix(i);
				}
			}
			else
			{
				distanceMatrix = new double[this.adjacencyList.size()][];
				FibNode[] nodes = this.adjacencyList.getNodes();
				for (int i = 0, n = this.adjacencyList.size(); i < n; i++)
				{
					 distanceMatrix[i] = calcShortestPath(nodes[i]);
//					distanceMatrix[i] = shortestPathUsingAdjacencyList(nodes[i]);
				}
			}
			this.maxDistance *= 20.0;
			if (!(this.maxDistance < Double.POSITIVE_INFINITY))
			{
				this.maxDistance = 500; // If all distances are >= 0.0 or
				// near infinity, we might as well put them to be 500 so nodes
				// are drawn better
			}
		}
		else
		{
			distanceMatrix = this.shortestPathsMatrix;
		}
		{
			boolean onlyIncludesInfinities = true;
			int len1 = distanceMatrix.length;
			int len2 = distanceMatrix[0].length;
			for (int i = 0; i < len1; i++)
			{
				for (int j = 0; j < len2; j++)
				{
					if (distanceMatrix[i][j] < Double.POSITIVE_INFINITY
							&& i != j)
					{
						onlyIncludesInfinities = false;
					}
				}
			}
			if (onlyIncludesInfinities)
				errorNumber = NO_CONNECTIONS_BETWEEN_NODES;
		}

		if (!includeInfinity)
		{
			for (int i = 0, len1 = distanceMatrix.length; i < len1; i++)
			{
				for (int j = 0, len2 = distanceMatrix[i].length; j < len2; j++)
				{
					if (distanceMatrix[i][j] > this.maxDistance)
					{
						distanceMatrix[i][j] = this.maxDistance;
					}
				}
			}
		}
		makeSymmetric = true;
		if (makeSymmetric)
		{
			/*
			 * Next we make the matrix symmetric by choosing lesser of two for
			 * every m[i][j], m[j][i]
			 */
			for (int i = 0; i < distanceMatrix.length; i++)
				for (int j = 0; j < distanceMatrix[i].length; j++)
				{
					if (distanceMatrix[i][j] > distanceMatrix[j][i])
					{
						distanceMatrix[i][j] = distanceMatrix[j][i];
					}
					else
					{
						distanceMatrix[j][i] = distanceMatrix[i][j];
					}
				}
		}
		this.shortestPathsCalculated = true;
		this.shortestPathsMatrix = distanceMatrix;
		return distanceMatrix;
		// this.shortestPathsMatrix = new double[distanceMatrix.length]
		// [distanceMatrix.length];
		// for (int i = 0; i < distanceMatrix.length;i++){
		// System.arraycopy(distanceMatrix[i], 0, this.shortestPathsMatrix[i],
		// 0, distanceMatrix.length);
		// }
		// return distanceMatrix;
	}

	/**
	 * Forces recalculation of all the shortest paths on the next call to
	 * allPairShortestPaths-method.
	 */
	public void forceRecalculation()
	{
		this.shortestPathsCalculated = false;
		this.errorNumber = 0;
	}

	/**
	 * Finds
	 *
	 * @param rootNode
	 * @return
	 * @throws IllegalArgumentException
	 */
	private double[] shortestPathUsingConnectivityMatrix(int rootNode)
			throws IllegalArgumentException
	{
		int matrixRows = conMatrix.length;

		if (!usingConnectivityMatrix) { throw new IllegalArgumentException(
				"Cannot calculate shortest paths"
						+ " using connectivity matrix because there is none."); }
		if (rootNode >= this.conMatrix.length)
		{
			throw new IllegalArgumentException("given number of root vertex"
					+ " is too large: it was " + rootNode + " when "
					+ "number of vertices is " + matrixRows);
		}
		else if (rootNode < 0) { throw new IllegalArgumentException(
				"given number of root vertex was" + " too small: " + rootNode); }
		if (!this.dijkstraInitialized)
		{
			this.border = new FibHeap();
			this.fibNodes = new FibNode[matrixRows];
			for (int i = 0; i < matrixRows; i++)
			{
				this.fibNodes[i] = new FibNode(i);
			}
			this.dijkstraInitialized = true;
		}

		double[] distanceArray = new double[matrixRows];
		FibNode extracted;
		double distance;
		int pointer;

		for (int i = 0; i < matrixRows; i++)
		{
			double dist;
			if (i != rootNode)
				dist = Double.POSITIVE_INFINITY;
			else
				dist = 0.0;

			this.fibNodes[i].reset();
			this.fibNodes[i].setKey(dist);
			distanceArray[i] = dist;
			border.insert(this.fibNodes[i]);
		}
		// Initialization ends here.

		while ((extracted = this.border.extractMin()) != null)
		{
			pointer = extracted.getId();
			distance = extracted.getKey();
			if ((distance > this.maxDistance) && (distance < Double.MAX_VALUE))
			{
				this.maxDistance = distance;
			}
			distanceArray[pointer] = distance;
			double[] localAdList = this.conMatrix[pointer];
			if (this.border.getNumberOfNodes() != 0 && localAdList != null)
			{
				for (int i = 0; i < matrixRows; i++)
				{
					FibNode currentNode = this.fibNodes[i];
					double distanceToNode = localAdList[i];
					// double dist = (distanceToNode >= 0) ? distanceToNode
					// + distance : distance;
					double dist = distanceToNode + distance;
					if (dist < currentNode.getKey())
					{
						this.border.decreaseKey(currentNode, dist);
					}
				}
			}
		}
		return distanceArray;
	}

	private double[] calcShortestPath(FibNode rootNode)
	{
		double distance;
		double distances[];
		FibNode extracted;

		border = new FibHeap();
		distances = new double[this.adjacencyList.size()];
		fibNodes = this.adjacencyList.getNodes();
		for (int i = 0; i < distances.length; i++)
		{
			double dist;
			if (fibNodes[i] != rootNode)
			{
				dist = Double.POSITIVE_INFINITY;
			}
			else
			{
				dist = 0.0;
			}
			fibNodes[i].reset();
			fibNodes[i].setKey(dist);
			border.insert(fibNodes[i]);
		}
		while ((extracted = border.extractMin()) != null)
		{
			Iterator<NodeKeyPair> iter;

			distance = extracted.getKey();

			// We keep account on largest distances found in the graph
			if ((distance > this.maxDistance) && (distance < Double.MAX_VALUE))
			{
				this.maxDistance = distance;
			}

			distances[extracted.getId()] = distance;
			iter = adjacencyList.getIterator(extracted);

			while (iter.hasNext())
			{
				NodeKeyPair curEdge = iter.next();
				double tempDist = distance + curEdge.getKey();
				if (tempDist < curEdge.getNode().getKey())
				{
					border.decreaseKey(curEdge.getNode(), tempDist);
				}
			}
		}
		return distances;
	}

	public double[] shortestPath(int rootNodeNumber)
	{
		double[] distanceArray;
		if (usingConnectivityMatrix)
		{
			distanceArray = shortestPathUsingConnectivityMatrix(rootNodeNumber);
		}
		else
		{
			FibNode rNode = this.adjacencyList.getNodes()[rootNodeNumber];
			distanceArray = calcShortestPath(rNode);
		}
		return distanceArray;
	}

	public int getErrorNumber()
	{
		return errorNumber;
	}
}