clusterPeakAnova = function(Q.dbeta.log=NULL, Q=NULL, param) {


	## PeakANOVA: Stronger findings from mass spectral data through multi-peak modeling
	## Clustering function

	## Copyright 2013 Tommi Suvitaival
	# Email: tommi.suvitaival@aalto.fi

	# This file is part of PeakANOVA.

	# PeakANOVA is free software: you can redistribute it and/or modify
	# it under the terms of the GNU Lesser General Public License as published by
	# the Free Software Foundation, either version 3 of the License, or
	# (at your option) any later version.

	# PeakANOVA is distributed in the hope that it will be useful,
	# but WITHOUT ANY WARRANTY; without even the implied warranty of
	# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	# GNU Lesser General Public License for more details.

	# You should have received a copy of the GNU Lesser General Public License
	# along with PeakANOVA.  If not, see <http://www.gnu.org/licenses/>.


	## Description
	# -A function for clustering variables (peaks) by their similarity in shapes.

	## Arguments
	# -Q: Peak shape correlations data. An array with of real values between -1 and 1 or missing values (NA). The array has dimensions NxPxP, where N is the total number of samples and P is the total number of variables (peaks). Provide either Q.dbeta.log or Q.
	# -Q.dbeta.log: Logarithmic likelihood of observed peak shape correlations data. A list of two matrices 'inside' and 'outside' containing the log-likelihoods of peak shape correlations in the same and in different clusters, respectively, summed over all samples. Both matrices are real-valued and are of dimensionality PxP, where P is the total number of variables (peaks).
	# param: A list of parameters created by the function generateSimulatedDataPeakAnova(). Provide either Q.dbeta.log or Q.

	## Value
	# -V.ls: The least-squares clustering computed over the Gibbs samples. A matrix with values 0 and 1 with one non-zero value on each row indicating the cluster assignment. The dimensionality of the matrix is PxK, where P is the total number of variables and K is the inferred number of clusters.
	# -V.vec: Gibbs samples of the clustering. A matrix with dimensionality SxP, where S is the number of Gibbs samples and P is the total number of variables. Each row 's' of the matrix is an indicator vector of cluster assignments of the P variables. Returned only if 'param$saveVvec=TRUE'.
	# -association.avg: Association matrix of the clustered variables computed as an average over the Gibbs samples. A matrix of real values between 0 and 1, indicating the probability of association of pairs of variables. The dimensionality of the matrix is PxP, where P is the total number of variables.


	## Source code

	if (!is.null(Q.dbeta.log)) {
		pre.computed.likelihood = TRUE
		N.variables = ncol(Q.dbeta.log$inside)
		diag(Q.dbeta.log$inside) = 0
		diag(Q.dbeta.log$outside) = 0
	} else {
		pre.computed.likelihood = FALSE
		N.variables = ncol(Q)
	}

	## Initialize the clustering.
	tmp = list()
	model = list()
	model$V = array(data=1, dim=c(N.variables,1))
	if (param$nonparametric.clustering) {
		association.avg = array(data=0, dim=c(1,1)*nrow(model$V))
		V.vec.saved = array(dim=c(param$Npsamples,nrow(model$V)))
	}

	## Sample the clustering.
	for (ni in 1:(param$Nburnin+param$Npsamples)) { # Go through all Gibbs samples.
		t = Sys.time()
		if (pre.computed.likelihood) {
			tmp = sampleV(q.dbeta.log=Q.dbeta.log, V=model$V, return.likelihood=TRUE, infinite=param$nonparametric.clustering, alpha.dp=param$alpha.dp)
		} else {
			tmp = sampleV(Q=Q, V=model$V, return.likelihood=TRUE, infinite=param$nonparametric.clustering, alpha.dp=param$alpha.dp)
		}
		model$V = tmp$V
		if (ni>param$Nburnin & param$nonparametric.clustering) { # Save clustering in vector format.
			for (ki in 1:ncol(model$V)) {
				variables.ki = which(model$V[,ki]==1)
				V.vec.saved[ni-param$Nburnin,variables.ki] = ki
			}
		}
		if (!param$silent) {
			print(paste("Clustering - ", ni, "/(", param$Nburnin, "+", param$Npsamples, ") - ", Sys.time(), " - ", signif(x=Sys.time()-t,digits=2), " sec/min - K=", ncol(model$V), sep=""))
		}
	}
	if (param$saveVadjacency) {
		tmp = findLSclustering(V.vec=V.vec.saved, return.association.avg=param$saveVadjacency)
		V.ls = tmp$V.ls
		association.avg = tmp$association.avg
	} else {
		V.ls = findLSclustering(V.vec=V.vec.saved, return.association.avg=param$saveVadjacency)
		association.avg = NULL
	}

	## Sort the clusters

	tmp = vector(mode="integer", length=ncol(V.ls))
	for (ki in 1:ncol(V.ls)) {
		tmp[ki] = median(which(V.ls[,ki]==1))
	}
	tmp = sort(x=tmp, index.return=TRUE)$ix
	V.ls = V.ls[,tmp]
	V.vec.sorted = array(dim=dim(V.vec.saved))
	if (param$saveVvec) {
		for (ki in 1:length(tmp)) {
			V.vec.sorted[which(V.vec.saved==tmp[ki])] = ki
		}
		tmp = which(is.na(V.vec.sorted))
		if (length(tmp)>0) {
			V.vec.sorted[tmp] = V.vec.saved[tmp]
		}
	} else {
		V.vec.sorted = NULL
	}

	return(list(V.ls=V.ls, V.vec=V.vec.sorted, association.avg=association.avg))

}