multiWayDR = function(data, param) {


	## PeakANOVA: Stronger findings from mass spectral data through multi-peak modeling
	## Main function of the sampler for covariate effects

	## Copyright 2013 Tommi Suvitaival
	# Email: tommi.suvitaival@aalto.fi

	# This file is part of PeakANOVA.

	# PeakANOVA is free software: you can redistribute it and/or modify
	# it under the terms of the GNU Lesser General Public License as published by
	# the Free Software Foundation, either version 3 of the License, or
	# (at your option) any later version.

	# PeakANOVA is distributed in the hope that it will be useful,
	# but WITHOUT ANY WARRANTY; without even the implied warranty of
	# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	# GNU Lesser General Public License for more details.

	# You should have received a copy of the GNU Lesser General Public License
	# along with PeakANOVA.  If not, see <http://www.gnu.org/licenses/>.


	## Description
	# -A function for inferring covariate effects from the intensity data given a clustering of variables (peaks).

	## Arguments
	# -data: A list of intensity data and covariates. 'X' is the intensity data matrix of real values with dimensionality PxN, where P is the total number of variables and N is the total number of samples. 'covariates' is a list of covariate indicator vectors 'a', 'b' and 'c', each with length of N. The vectors have positive integer values, and each value is matched to a column with the same index in 'X'.
	# -param: A list of parameters created by the function generateSimulatedDataPeakAnova().
 
	## Value
	# -posterior: A list of Gibbs samples from the model.
	#   -eff: A list of inferred covariate effects with arrays 'A', 'B' and 'C' corresponding to the covariates 'a', 'b' and 'c' (each returned if the corresponding covariate has values above 1 and if param$sampleEff${A,B,C}=TRUE). The arrays have dimensionality SxKxL{a,b,c}, where S is the number of Gibbs samples, K is the number of clusters and L{a,b,c} are the numbers of covariate levels of covariates 'a', 'b' and 'c', respectively. Additionally, interaction effects 'AB', 'AC', 'BC' and 'ABC' may be returned if the corresponding value in the list 'param$sampleEff' is 'TRUE'.
	#   -effects: Same covariate effects as in the list 'eff', but all saved into a single array. The array has dimensionality LxKxS, where L is the total number of covariate levels and their interaction levels, K is the number of clusters and S is the number of Gibbs samples. The correspondence of the levels of the covariates and the L rows of the array are mapped by the list 'design$ind$tab'.
	#   -sigma: A matrix of the inferred variance parameters. The matrix has dimensionality PxS, where P is the number of variables and S is the number of Gibbs samples.
	# -design:
	#   -mat: An indicator matrix of the covariate levels that match to the array 'posterior$effects'. The matrix has dimensionality NxL, where N is the total number of samples and L is the total number of covariate levels and their interaction levels.
	#   -ind$tab: A mapping from the covariate levels to the rows of the array 'posterior$effects'.


	## Source code

	n = ncol(data$X) # number of samples
	m = nrow(data$X) # number of metabolites (variables)

	prior = list()
	prior$theta = param$theta
	prior$sigma = param$sigma
	
	if (param$two.levels.clustering) {
		prior$psi = param$psi
	}
	
	## Initialization
	
	if (!is.null(param$fixed$V)) {
		param$initialization$V = param$fixed$V
	}

	if (is.null(data$Q) & is.null(data$Q.dbeta.log) & !param$two.levels.clustering) {
		param$K = ncol(param$initialization$V)
		param$L = param$K
		model = sampleFromPriorDR(data=data, priors=prior, xlatDim=param$K, clustering=param$clustering, initialization=param$initialization) # Initialize dimensionality reduction-related latent variables.
	} else {
		model = sampleFromPriorDR(data=data, priors=prior, xlatDim=c(param$L,param$K), clustering=param$clustering, initialization=param$initialization) # Initialize dimensionality reduction-related latent variables.
	}
	
	if (param$Npsamples>param$Npsaved) {
		mod = floor(param$Npsamples/param$Npsaved)
		Nps = param$Npsaved
	} else {
		mod = 1
		Nps = param$Npsamples
	}
	posterior = list()
	pcnt = 0
	
	tmp = NULL
	tmp.ind = vector(mode="integer")
	tmp.ind.V = vector(mode="integer")

	## Select covariate levels whose effects will be sampled.
	
	param$sampleEffects = list()

	param$sampleEffects$A = rep(x=FALSE, times=max(data$covariates$a))
	if (!is.null(param$sampleEff$A)) {
		if (max(data$covariates$a)>1) {
			if (length(param$sampleEff$A)==1) {
				if (param$sampleEff$A) {
					for (ai in 1:max(data$covariates$a)) {
						param$sampleEffects$A[ai] = (length(which(data$covariates$a==ai))>0)
					}
				}
			} else {
				for (ai in 1:max(data$covariates$a)) {
					param$sampleEffects$A[ai] = (length(which(data$covariates$a==ai))>0 & param$sampleEff$A[ai]==TRUE)
				}
			}
		}
	}

	param$sampleEffects$B = rep(x=FALSE, times=max(data$covariates$b))
	if (!is.null(param$sampleEff$B)) {
		if (max(data$covariates$b)>1) {
			if (length(param$sampleEff$B)==1) {
				if (param$sampleEff$B) {
					for (bi in 1:max(data$covariates$b)) {
						param$sampleEffects$B[bi] = (length(which(data$covariates$b==bi))>0)
					}
				}
			} else {
				for (bi in 1:max(data$covariates$b)) {
					param$sampleEffects$B[bi] = (length(which(data$covariates$b==bi))>0 & param$sampleEff$B[bi]==TRUE)
				}
			}
		}
	}

	param$sampleEffects$C = rep(x=FALSE, times=max(data$covariates$c))
	if (!is.null(param$sampleEff$C)) {
		if (max(data$covariates$c)>1) {
			if (length(param$sampleEff$C)==1) {
				if (param$sampleEff$C) {
					for (ci in 1:max(data$covariates$c)) {
						param$sampleEffects$C[ci] = (length(which(data$covariates$c==ci))>0)
					}
				}
			} else {
				for (ci in 1:max(data$covariates$c)) {
					param$sampleEffects$C[ci] = (length(which(data$covariates$c==ci))>0 & param$sampleEff$C[ci]==TRUE)
				}
			}
		}
	}

	param$sampleEffects$AB = array(data=FALSE, dim=c(max(data$covariates$a),max(data$covariates$b)))
	if (!is.null(param$sampleEff$AB)) {
		if (max(data$covariates$a)>1 & max(data$covariates$b)>1) {
			if (length(param$sampleEff$AB)==1) {
				if (param$sampleEff$AB==TRUE) {
					for (ai in 1:max(data$covariates$a)) {
						for (bi in 1:max(data$covariates$b)) {
							param$sampleEffects$AB[ai,bi] = (length(which(data$covariates$a==ai & data$covariates$b==bi))>0)
						}
					}
				}
			} else {
				for (ai in 1:max(data$covariates$a)) {
					for (bi in 1:max(data$covariates$b)) {
						param$sampleEffects$AB[ai,bi] = (length(which(data$covariates$a==ai & data$covariates$b==bi))>0 & param$sampleEff$AB[ai,bi]==TRUE)
					}
				}
			}
		}
	}

	param$sampleEffects$AC = array(data=FALSE, dim=c(max(data$covariates$a),max(data$covariates$c)))
	if (!is.null(param$sampleEff$AC)) {
		if (max(data$covariates$a)>1 & max(data$covariates$c)>1) {
			if (length(param$sampleEff$AC)==1) {
				if (param$sampleEff$AC==TRUE) {
					for (ai in 1:max(data$covariates$a)) {
						for (ci in 1:max(data$covariates$c)) {
							param$sampleEffects$AC[ai,ci] = (length(which(data$covariates$a==ai & data$covariates$c==ci))>0)
						}
					}
				}
			} else {
				for (ai in 1:max(data$covariates$a)) {
					for (ci in 1:max(data$covariates$c)) {
						param$sampleEffects$AC[ai,ci] = (length(which(data$covariates$a==ai & data$covariates$c==ci))>0 & param$sampleEff$AC[ai,ci]==TRUE)
					}
				}
			}
		}
	}

	param$sampleEffects$BC = array(data=FALSE, dim=c(max(data$covariates$b),max(data$covariates$c)))
	if (!is.null(param$sampleEff$BC)) {
		if (max(data$covariates$b)>1 & max(data$covariates$c)>1) {
			if (length(param$sampleEff$BC)==1) {
				if (param$sampleEff$BC==TRUE) {
					for (bi in 1:max(data$covariates$b)) {
						for (ci in 1:max(data$covariates$c)) {
							param$sampleEffects$BC[bi,ci] = (length(which(data$covariates$b==bi & data$covariates$c==ci))>0)
						}
					}
				}
			} else {
				for (bi in 1:max(data$covariates$b)) {
					for (ci in 1:max(data$covariates$c)) {
						param$sampleEffects$BC[bi,ci] = (length(which(data$covariates$b==bi & data$covariates$c==ci))>0 & param$sampleEff$BC[bi,ci]==TRUE)
					}
				}
			}
		}
	}

	param$sampleEffects$ABC = array(data=FALSE, dim=c(max(data$covariates$a),max(data$covariates$b),max(data$covariates$c)))
	if (!is.null(param$sampleEff$ABC)) {
		if (max(data$covariates$a)>1 & max(data$covariates$b)>1 & max(data$covariates$c)>1) {
			if (length(param$sampleEff$ABC)==1) {
				if (param$sampleEff$ABC==TRUE) {
					for (ai in 1:max(data$covariates$a)) {
						for (bi in 1:max(data$covariates$b)) {
							for (ci in 1:max(data$covariates$c)) {
								param$sampleEffects$ABC[ai,bi,ci] = (length(which(data$covariates$a==ai & data$covariates$b==bi & data$covariates$c==ci))>0)
							}
						}
					}
				}
			} else {
				for (ai in 1:max(data$covariates$a)) {
					for (bi in 1:max(data$covariates$b)) {
						for (ci in 1:max(data$covariates$c)) {
							param$sampleEffects$ABC[ai,bi,ci] = (length(which(data$covariates$a==ai & data$covariates$b==bi & data$covariates$c==ci))>0 & param$sampleEff$ABC[ai,bi,ci]==TRUE)
						}
					}
				}
			}
		}
	}
	
	## Pre-compute beta distribution densities for the correlation values.
	
	if (!is.null(data$Q) & is.null(data$Q.dbeta.log)) {
		data$Q.dbeta.log = list()
		data$Q.dbeta.log$inside = apply(X=dbeta(x=data$Q, shape1=param$shapes.beta$inside[1], shape2=param$shapes.beta$inside[2], log=TRUE), MAR=2:3, FUN=sum)
		diag(data$Q.dbeta.log$inside) = 0
		data$Q.dbeta.log$outside = apply(X=dbeta(x=data$Q, shape1=param$shapes.beta$outside[1], shape2=param$shapes.beta$outside[2], log=TRUE), MAR=2:3, FUN=sum)
		diag(data$Q.dbeta.log$outside) = 0
	}

	## Initialize covariate effects.
	
	data$design = create3wayDesignMatrixWithEffects(covA=data$covariates$a, covB=data$covariates$b, covC=data$covariates$c)
	model$effects = array(data=0, dim=c(length(c(data$design$ind$tab$A, data$design$ind$tab$B, data$design$ind$tab$C, data$design$ind$tab$AB, data$design$ind$tab$AC, data$design$ind$tab$BC, data$design$ind$tab$ABC)), param$K))
	posterior$eff = list()

	if (!is.null(param$fixed$eff$A)) {
		tmp.ind = which(!is.na(param$fixed$eff$A[1,]))
		tmp = data$design$ind$tab$A[tmp.ind]
		model$effects[tmp,] = t(param$fixed$eff$A[,tmp.ind])
		param$sampleEffects$A[tmp.ind] = FALSE
	}
	if (any(param$sampleEffects$A)) {
		tmp.ind = which(param$sampleEffects$A==TRUE)
		tmp = data$design$ind$tab$A[tmp.ind]
		model$effects[tmp,] = rnorm(n=length(tmp)*param$K)
		posterior$eff$A = array(dim=c(Nps,param$K,max(data$covariates$a)))
	}

	if (!is.null(param$fixed$eff$B)) {
		tmp.ind = which(!is.na(param$fixed$eff$B[1,]))
		tmp = data$design$ind$tab$B[tmp.ind]
		model$effects[tmp,] = t(param$fixed$eff$B[,tmp.ind])
		param$sampleEffects$B[tmp.ind] = FALSE
	}
	if (any(param$sampleEffects$B)) {
		tmp.ind = which(param$sampleEffects$B==TRUE)
		tmp = data$design$ind$tab$B[tmp.ind]
		model$effects[tmp,] = rnorm(n=length(tmp)*param$K)
		posterior$eff$B = array(dim=c(Nps,param$K,max(data$covariates$b)))
	}

	if (!is.null(param$fixed$eff$C)) {
		tmp.ind = which(!is.na(param$fixed$eff$C[1,]))
		tmp = data$design$ind$tab$C[tmp.ind]
		model$effects[tmp,] = t(param$fixed$eff$C[,tmp.ind])
		param$sampleEffects$C[tmp.ind] = FALSE
	}
	if (any(param$sampleEffects$C)) {
		tmp.ind = which(param$sampleEffects$C==TRUE)
		tmp = data$design$ind$tab$C[tmp.ind]
		model$effects[tmp,] = rnorm(n=length(tmp)*param$K)
		posterior$eff$C = array(dim=c(Nps,param$K,max(data$covariates$c)))
	}

	if (!is.null(param$fixed$eff$AB)) {
		tmp.ind = which(!is.na(param$fixed$eff$AB[1,,])) # Fixed levels are the same for all clusters.
		tmp = data$design$ind$tab$AB[tmp.ind]
		for (ki in 1:ncol(model$effects)) { # Go through all clusters.
			model$effects[tmp,ki] = param$fixed$eff$AB[ki,,][tmp.ind]
		}
		param$sampleEffects$AB[tmp.ind] = FALSE
	}
	if (any(param$sampleEffects$AB)) {
		tmp = data$design$ind$tab$AB[which(param$sampleEffects$AB==TRUE)]
		model$effects[tmp,] = rnorm(n=length(tmp)*param$K)
		posterior$eff$AB = array(dim=c(Nps,param$K,max(data$covariates$a),max(data$covariates$b)))
	}

	if (!is.null(param$fixed$eff$AC)) {
		tmp.ind = which(!is.na(param$fixed$eff$AC[1,,])) # Fixed levels are the same for all clusters.
		tmp = data$design$ind$tab$AC[tmp.ind]
		for (ki in 1:ncol(model$effects)) { # Go through all clusters.
			model$effects[tmp,ki] = param$fixed$eff$AC[ki,,][tmp.ind]
		}
		param$sampleEffects$AC[tmp.ind] = FALSE
	}
	if (any(param$sampleEffects$AC)) {
		tmp = data$design$ind$tab$AC[which(param$sampleEffects$AC==TRUE)]
		model$effects[tmp,] = rnorm(n=length(tmp)*param$K)
		posterior$eff$AC = array(dim=c(Nps,param$K,max(data$covariates$a),max(data$covariates$c)))
	}

	if (!is.null(param$fixed$eff$BC)) {
		tmp.ind = which(!is.na(param$fixed$eff$BC[1,,])) # Fixed levels are the same for all clusters.
		tmp = data$design$ind$tab$BC[tmp.ind]
		for (ki in 1:ncol(model$effects)) { # Go through all clusters.
			model$effects[tmp,ki] = param$fixed$eff$BC[ki,,][tmp.ind]
		}
		param$sampleEffects$BC[tmp.ind] = FALSE
	}
	if (any(param$sampleEffects$BC)) {
		tmp = data$design$ind$tab$BC[which(param$sampleEffects$BC==TRUE)]
		model$effects[tmp,] = rnorm(n=length(tmp)*param$K)
		posterior$eff$BC = array(dim=c(Nps,param$K,max(data$covariates$b),max(data$covariates$c)))
	}

	if (!is.null(param$fixed$eff$ABC)) {
		tmp.ind = which(!is.na(param$fixed$eff$ABC[1,,,])) # Fixed levels are the same for all clusters.
		tmp = data$design$ind$tab$ABC[tmp.ind]
		for (ki in 1:ncol(model$effects)) { # Go through all clusters.
			model$effects[tmp,ki] = param$fixed$eff$ABC[ki,,,][tmp.ind]
		}
		param$sampleEffects$ABC[tmp.ind] = FALSE
	}
	if (any(param$sampleEffects$ABC)) {
		tmp = data$design$ind$tab$ABC[which(param$sampleEffects$ABC==TRUE)]
		model$effects[tmp,] = rnorm(n=length(tmp)*param$K)
		posterior$eff$ABC = array(dim=c(Nps,param$K,max(data$covariates$a),max(data$covariates$b),max(data$covariates$c)))
	}

	## Initialize containers for clustering-related variables' Gibbs samples.
	
	if (param$clustering) {
		if (param$saveXlat) {
			posterior$xLat <- array(dim=c(param$L,n,Nps)) # 'xlats'
		}
		if (!is.null(param$fixed$sigma)) {
			model$sigma = param$fixed$sigma
			param$sampleSigma = FALSE
		} else if (param$sampleSigma) {
			posterior$sigma <- array(dim=c(m,Nps)) # residual variance
		}
		if (is.null(param$fixed$V)) {
			if (param$saveV) {
				posterior$V = array(dim=c(m,param$L,Nps)) # clusterings
			} else {
				posterior$Vsum = array(data=0, dim=c(m,param$L)) # cumulative clusterings
			}
			if (param$saveVvec) {
				posterior$V.vec = array(dim=c(m,Nps))
			}
			if (param$saveVadjacency) {
				posterior$V.adjacency = array(data=0, dim=c(1,1)*m)
			}
		} else {
			model$V = param$fixed$V
		}
		if (!is.null(param$fixed$psi)) {
			model$psi = param$fixed$psi
			param$samplePsi = FALSE
		}
		if (param$two.levels.clustering) {
			if (param$samplePsi) {
				posterior$psi <- array(dim=c(param$L,Nps)) # residual variance
			}
			if (param$saveZ) {
				posterior$z <- array(dim=c(param$K,n,Nps)) # 'xlats'
			}
			if (param$saveW) {
				posterior$W = array(dim=c(param$L,param$K,Nps)) # second level of clusterings
			} else {
				posterior$Wsum = array(data=0, dim=c(param$L,param$K)) # cumulative clusterings
			}
			if (param$saveWvec) {
				posterior$W.vec = array(dim=c(param$L,Nps))
			}
			if (!is.null(param$fixed$W)) {
				model$W = param$fixed$W
			}
			if (param$saveVWadjacency) {
				posterior$VW.adjacency = array(data=0, dim=c(1,1)*m)
			}
			if (param$saveWadjacency) {
				posterior$W.adjacency = array(data=0, dim=c(1,1)*param$L)
			}
			if (!is.null(param$fixed$ksi)) {
				model$ksi = param$fixed$ksi
			} else {
				model$ksi = rep(x=1, times=param$K)
			}
		}
	} else { # If sigma is sampled for the model without clustering, the variable is stored as psi to have consistency in the sampling of the effects. -8.1.13
		if (!is.null(param$fixed$sigma)) {
			model$psi = param$fixed$sigma
			param$sampleSigma = FALSE
		} else if (param$sampleSigma) {
			model$psi = model$sigma
			model$sigma = NULL
			model$V = diag(nrow=m)
			posterior$sigma <- array(dim=c(m,Nps)) # residual variance
		}
	}
	
	posterior$effects = array(dim=c(dim(model$effects),Nps))

	## Sampling

	if (!param$silent) {
		print(paste("Dimension reduction with",n,"samples and",m,"variables into",param$K,"clusters."))
	}
	t = Sys.time()

	for (i in 1:(param$Nburnin+param$Npsamples)) { # Go through all Gibbs samples.
		if (i%%mod==0) {
			if (!param$silent) {
				print(paste("Multi-way - ", i, "/(", param$Nburnin, "+", param$Npsamples, ") - ", Sys.time(), " - ", signif(x=Sys.time()-t,digits=2), " sec/min", sep=""))
			}
			t = Sys.time()
		}
		## Update covariate effects.
		if (any(param$sampleEffects$A)) {
			tmp = data$design$ind$tab$A[which(param$sampleEffects$A)]
			if (!param$two.levels.clustering) {
				model$effects[tmp,] = sampleEffMat(z=model$xLat, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$psi)
			} else {
				model$effects[tmp,] = sampleEffMat(z=model$z, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$ksi)
			}
		}
		if (any(param$sampleEffects$B)) {
			tmp = data$design$ind$tab$B[which(param$sampleEffects$B)]
			if (!param$two.levels.clustering) {
				model$effects[tmp,] = sampleEffMat(z=model$xLat, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$psi)
			} else {
				model$effects[tmp,] = sampleEffMat(z=model$z, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$ksi)
			}
		}
		if (any(param$sampleEffects$C)) {
			tmp = data$design$ind$tab$C[which(param$sampleEffects$C)]
			if (!param$two.levels.clustering) {
				model$effects[tmp,] = sampleEffMat(z=model$xLat, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$psi)
			} else {
				model$effects[tmp,] = sampleEffMat(z=model$z, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$ksi)
			}
		}
		if (any(param$sampleEffects$AB)) {
			tmp = data$design$ind$tab$AB[which(param$sampleEffects$AB==TRUE)]
			if (!param$two.levels.clustering) {
				model$effects[tmp,] = sampleEffMat(z=model$xLat, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$psi)
			} else {
				model$effects[tmp,] = sampleEffMat(z=model$z, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$ksi)
			}
		}
		if (any(param$sampleEffects$AC)) {
			tmp = data$design$ind$tab$AC[which(param$sampleEffects$AC==TRUE)]
			if (!param$two.levels.clustering) {
				model$effects[tmp,] = sampleEffMat(z=model$xLat, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$psi)
			} else {
				model$effects[tmp,] = sampleEffMat(z=model$z, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$ksi)
			}
		}
		if (any(param$sampleEffects$BC)) {
			tmp = data$design$ind$tab$BC[which(param$sampleEffects$BC==TRUE)]
			if (!param$two.levels.clustering) {
				model$effects[tmp,] = sampleEffMat(z=model$xLat, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$psi)
			} else {
				model$effects[tmp,] = sampleEffMat(z=model$z, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$ksi)
			}
		}
		if (any(param$sampleEffects$ABC)) {
			tmp = data$design$ind$tab$ABC[which(param$sampleEffects$ABC==TRUE)]
			if (!param$two.levels.clustering) {
				model$effects[tmp,] = sampleEffMat(z=model$xLat, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$psi)
			} else {
				model$effects[tmp,] = sampleEffMat(z=model$z, design=data$design$mat, effects=model$effects, sampleLevels=tmp, multiWay=TRUE, prior.var=model$ksi)
			}
		}
		## Update dimensionality reduction-related latent variables.
		if (param$clustering) {
			if (is.null(param$fixed$V)) {
				if (is.null(data$Q) & is.null(data$Q.dbeta.log)) {
					model$V = sampleV(x=data$X, V=model$V, z=model$xLat, mu=model$mu, Sigma=model$sigma, theta=prior$theta$K)
				} else {
					model$V = sampleV(x=data$X, q.dbeta.log=data$Q.dbeta.log, V=model$V, z=model$xLat, mu=model$mu, Sigma=model$sigma, theta=prior$theta$L)
				}
			}
			if (param$sampleSigma) {
				model$sigma = sampleSigma(x=data$X, V=model$V, z=model$xLat, prior.sigma0=prior$sigma$sigma0, prior.N0=prior$sigma$N0)
			}
			if (!param$two.levels.clustering) { ## One-level model
				model$xLat = sampleXlats2mat(x=data$X, Sigma=model$sigma, V=model$V, design=data$design$mat, effects=model$effects, Psi=model$psi)
			} else { ## Two-level model
				model$xLat = sampleXlats2mat(x=data$X, Sigma=model$sigma, V=model$V, z=model$z, Psi=model$psi, W=model$W)
				model$W = sampleV(x=model$xLat, z=model$z, Sigma=model$psi, theta=prior$theta$K)
				if (param$samplePsi) {
					model$psi = sampleSigma(x=model$xLat, V=model$W, z=model$z, prior.sigma0=prior$psi$psi0, prior.N0=prior$psi$N0)
				}
				model$z = sampleXlats2mat(x=model$xLat, Sigma=model$psi, V=model$W, design=data$design$mat, effects=model$effects, Psi=model$ksi)
			}
		} else {
			if (param$sampleSigma) {
				model$psi = sampleSigma(x=data$X, V=model$V, z=t(data$design$mat%*%model$effects), prior.sigma0=prior$sigma$sigma0, prior.N0=prior$sigma$N0) # Sample sigma when no clustering: The variable is named psi in order to have consistency at sampling of the effects. -8.1.12
			}
		}
		
		## Save the posterior sample.
		if (i>param$Nburnin & i%%mod==0) {
			pcnt = pcnt+1
			posterior$effects[,,pcnt] = model$effects
			if (param$clustering) {
				if (param$sampleSigma) {
					posterior$sigma[,pcnt] <- model$sigma
				}
				if (param$saveXlat)
					posterior$xLat[,,pcnt] = model$xLat
				if (is.null(param$fixed$V)) {
					if (param$saveV) {
						posterior$V[,,pcnt] = model$V
					}
					if (param$saveVadjacency | param$saveVvec) {
						for (ki in 1:ncol(model$V)) {
							tmp.ind = which(model$V[,ki]==1)
							if (length(tmp.ind)>0) {
								if (param$saveVadjacency) {
									posterior$V.adjacency[tmp.ind,tmp.ind] = posterior$V.adjacency[tmp.ind,tmp.ind]+1
								}
								if (param$saveVvec) {
									posterior$V.vec[tmp.ind,pcnt] = ki
								}
							}
						}
					}
				}
				if (param$two.levels.clustering) {
					if (param$samplePsi)
						posterior$psi[,pcnt] <- model$psi
					if (param$saveZ)
						posterior$z[,,pcnt] = model$z
					if (is.null(param$fixed$W)) {
						if (param$saveW) {
							posterior$W[,,pcnt] = model$W
						}
						if (param$saveWadjacency | param$saveVWadjacency | param$saveWvec) {
							for (ki in 1:ncol(model$W)) {
								tmp.ind = which(model$W[,ki]==1)
								if (length(tmp.ind)>0) {
									if (param$saveWadjacency) {
										posterior$W.adjacency[tmp.ind,tmp.ind] = posterior$W.adjacency[tmp.ind,tmp.ind]+1
									}
									if (param$saveVWadjacency) {
										tmp.ind.V = which(model$V[,tmp.ind,drop=FALSE]==1, arr.ind=TRUE)[,1]
										if (length(tmp.ind.V)>0) {
											posterior$VW.adjacency[tmp.ind.V,tmp.ind.V] = posterior$VW.adjacency[tmp.ind.V,tmp.ind.V]+1
										}
									}
									if (param$saveWvec) {
										posterior$W.vec[tmp.ind,pcnt] = ki
									}
								}
							}
						}
					}
				}
			} else {
				if (param$sampleSigma) {
					posterior$sigma[,pcnt] <- model$psi
				}
			}
		}
	}

	if (!param$silent) {
		print("Sampling finished.")
	}

	# Convert covariate effects to a more useful format.
	if (any(param$sampleEffects$A) | !is.null(param$fixed$eff$A)) { # 1
		posterior$eff$A = aperm(a=posterior$effects[data$design$ind$tab$A,,,drop=FALSE], perm=c(3,2,1))
	}
	if (any(param$sampleEffects$B) | !is.null(param$fixed$eff$B)) { # 2
		posterior$eff$B = aperm(a=posterior$effects[data$design$ind$tab$B,,,drop=FALSE], perm=c(3,2,1))
	}
	if (any(param$sampleEffects$C) | !is.null(param$fixed$eff$C)) # 3
		posterior$eff$C = aperm(a=posterior$effects[data$design$ind$tab$C,,,drop=FALSE], perm=c(3,2,1))
	for (ai in 1:max(data$covariates$a)) {
		if (any(param$sampleEffects$AB) | !is.null(param$fixed$eff$AB)) # 5
			posterior$eff$AB[,,ai,] = aperm(a=posterior$effects[data$design$ind$tab$AB[ai,],,,drop=FALSE], perm=c(3,2,1))
		if (any(param$sampleEffects$AC) | !is.null(param$fixed$eff$AC)) # 6
			posterior$eff$AC[,,ai,] = aperm(a=posterior$effects[data$design$ind$tab$AC[ai,],,,drop=FALSE], perm=c(3,2,1))
		for (bi in 1:max(data$covariates$b)) {
			if (any(param$sampleEffects$ABC) | !is.null(param$fixed$eff$ABC)) # 11
				posterior$eff$ABC[,,ai,bi,] = aperm(a=posterior$effects[data$design$ind$tab$ABC[ai,bi,],,,drop=FALSE], perm=c(3,2,1))
			if (ai==1) {
				if (any(param$sampleEffects$BC) | !is.null(param$fixed$eff$BC)) # 8
					posterior$eff$BC[,,bi,] = aperm(a=posterior$effects[data$design$ind$tab$BC[bi,],,,drop=FALSE], perm=c(3,2,1))
			}
		}
	}

	return(list(posterior=posterior, design=data$design))

}