normalizeDataByControlPopulation = function(X, covariates, log.transform=FALSE, zero.mean=TRUE, unit.scale=FALSE) {


	## PeakANOVA: Stronger findings from mass spectral data through multi-peak modeling
	## Function for normalizing the intensity data

	## Copyright 2013 Tommi Suvitaival
	# Email: tommi.suvitaival@aalto.fi

	# This file is part of PeakANOVA.

	# PeakANOVA is free software: you can redistribute it and/or modify
	# it under the terms of the GNU Lesser General Public License as published by
	# the Free Software Foundation, either version 3 of the License, or
	# (at your option) any later version.

	# PeakANOVA is distributed in the hope that it will be useful,
	# but WITHOUT ANY WARRANTY; without even the implied warranty of
	# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	# GNU Lesser General Public License for more details.

	# You should have received a copy of the GNU Lesser General Public License
	# along with PeakANOVA.  If not, see <http://www.gnu.org/licenses/>.


	## Description
	# -A function for normalizing (i.e., Z-transforming) the intensity data based on the control samples

	## Arguments
	# -X: The intensity (peak height) data. A matrix of real values or missing values with dimensionality PxN, where P is the total number of variables and N is the total number of samples.
	# -covariates: Covariate levels of samples. A list of vectors 'a', 'b' and 'c' corresponding to covariates with the same names. Each vector is of length N with positive integer values, where N is the total number of samples. Control samples are defined as the samples fulfilling the condition 'a==1', 'b==1' and 'c==1'.
	# -log.transform: Indicator on whether X will be transformed into logarithmic space before the normalization. A value 'FALSE', or a positive real value indicating the base of the logarithm (as in log(x,base)).
	# -zero.mean: Indicator on whether the variable-specific mean of the control population will or will not be subtracted from the data or not ('TRUE' or 'FALSE', respectively).
	# -unit.scale: Indicator on whether the data will or will not be divided with the variable-specific scale (variance) of the control population ('TRUE' or 'FALSE', respectively).
 
	## Value
	# -normalization: Normalization parameter vectors mean and scale. Both vectors are of length P, where P is the total number of variables.
	# -X: The normalized intensity data. A matrix with same dimensionality as the argument X.


	## Source code

	base.population = rep(x=TRUE, length=length(covariates[[1]]))
	for (ci in 1:length(covariates)) { # Go through all covariates.
		base.population = (base.population & covariates[[ci]]==1) # Find samples that have level 1 for all covariates
	}

	if (log.transform) {
		if (log.transform==1) {
			X = log(X)
		} else {
			X = log(X, base=log.transform)
		}
	}

	normalization = list()

	if (zero.mean) {
		normalization$mean = rowMeans(x=X[,base.population], na.rm=TRUE)
		X = X-normalization$mean
	}

	if (unit.scale) {
		normalization$scale = apply(X=X[,base.population], MAR=1, FUN=sd, na.rm=TRUE)
		X = X/normalization$scale
	}

	return(list(X=X, normalization=normalization))

}