preComputeQlikeSum = function(Q, p.spike.inside=0.01, p.spike.outside=0.99, shapes.beta.inside=c(2,1), shapes.beta.outside=c(1,1)) {


	## PeakANOVA: Stronger findings from mass spectral data through multi-peak modeling
	## Function for pre-computing the likelihood values of the peak shape correlations

	## Copyright 2013 Tommi Suvitaival
	# Email: tommi.suvitaival@aalto.fi

	# This file is part of PeakANOVA.

	# PeakANOVA is free software: you can redistribute it and/or modify
	# it under the terms of the GNU Lesser General Public License as published by
	# the Free Software Foundation, either version 3 of the License, or
	# (at your option) any later version.

	# PeakANOVA is distributed in the hope that it will be useful,
	# but WITHOUT ANY WARRANTY; without even the implied warranty of
	# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	# GNU Lesser General Public License for more details.

	# You should have received a copy of the GNU Lesser General Public License
	# along with PeakANOVA.  If not, see <http://www.gnu.org/licenses/>.


	## Description
	# -A function for computing the likelihood of peak shape correlations.
	# -Useful for pre-computing the likelihoods to make the clustering step significantly faster.

	## Arguments
	# -Q: Peak shape correlations data. An array with of real values between -1 and 1 or missing values (NA). The array has dimensions NxPxP, where N is the total number of samples and P is the total number of variables (peaks).
	# -p.spike.inside: Likelihood of a missing value in a peak shape correlation matrix, for a pair of peaks in the same cluster. A real value between 0 and 1.
	# -p.spike.outside: Likelihood of a missing value in a peak shape correlation matrix, for a pair of peaks in different clusters. A real value between 0 and 1.
	# -shapes.beta.inside: Parameters of the beta distribution that defines the likelihood of an observed value in a peak shape correlation matrix, for a pair of peaks in the same cluster. A vector of length 2 with non-negative real values.
	# -shapes.beta.outside: Parameters of the beta distribution that defines the likelihood of an observed value in a peak shape correlation matrix, for a pair of peaks in different clusters. A vector of length 2 with non-negative real values.

	## Value
	# -Q.dbeta.log:  Logarithmic likelihood of observed peak shape correlations data. A list of two matrices 'inside' and 'outside' containing the log-likelihoods of peak shape correlations in the same and in different clusters, respectively, summed over all samples. Both matrices are real-valued and are of dimensionality PxP, where P is the total number of variables (peaks).

	
	## Source code

	# Initialize the likelihood arrays (samples x variables x variables).
	Q.dbeta.log.inside = array(dim=dim(Q))
	Q.dbeta.log.outside = array(dim=dim(Q))
	
	# Compute the likelihood of missing data.
	Q.dbeta.log.inside[which(is.na(Q))] = log(p.spike$inside)
	Q.dbeta.log.outside[which(is.na(Q))] = log(p.spike$outside)
	
	# Compute the likelihood of observed data.
	Q.dbeta.log.inside[which(!is.na(Q))] = dbeta(x=Q[which(!is.na(Q))], shape1=shapes.beta.inside[1], shape2=shapes.beta.inside[2], log=TRUE) + log(1-p.spike$inside)
	Q.dbeta.log.outside[which(!is.na(Q))] = dbeta(x=Q[which(!is.na(Q))], shape1=shapes.beta.outside[1], shape2=shapes.beta.outside[2], log=TRUE) + log(1-p.spike$outside)
	
	# Set diagonal entries to zero.
	diag(Q.dbeta.log.inside) = 0
	diag(Q.dbeta.log.outside) = 0
	
	# Compute sum over samples. => A matrix with dimensions variables x variables.
	Q.dbeta.log.sum = list()
	Q.dbeta.log.sum$inside = apply(X=Q.dbeta.log.inside, MAR=2:3, FUN=sum)
	Q.dbeta.log.sum$outside = apply(X=Q.dbeta.log.outside, MAR=2:3, FUN=sum)
	
	return(Q.dbeta.log.sum)
	
}