function args = lineSearch_m_gamma_update_all(args)
%Take a step of length a in the search direction searchDir. All model
%parameters are updated according to this step and components satisfying the removal criteria are removed from the model.
%This function is used to perform the line search in conjuction with the
%function quadraticSearch.

%All the input variables are assembled to cell args. args{1} is the step
%size and args{2} is reserved for the cost to be evaluated by this
%function. See the first line of code for the detailed contents of this
%cell. The output of the function is a similar cell of values
%corresponding to the step taken by the function.

%Copyright (C) 2008-2010 Mikael Kuusela.

[a, cost, searchDir, alpha_0, beta_0, v_0, W_0, invW_0, mean_0, res, N_eff, x_avg, S, alpha, beta, mean, v, W, pi_eff, lambda_eff, K, N, X, D, removalCriteria] = args{:};

%Update mean using step size a
mean = mean + a*reshape(searchDir(1:D*K),D,[]);

%Calculate delta_gamma with step size a
delta_gamma = a*reshape(searchDir(D*K+1:end),K-1,[])';

%Calculate new responsibilities
for n=1:N
    for k=1:K
        if k ~= K
            res(n,k) = res(n,k)*exp(delta_gamma(n,k));
        end
    end
    res(n,:) = res(n,:) / sum(res(n,:)); %Normalization
    
    normalization = false;
    for k=1:K
        if res(n,k) < 1e-10
            res(n,k) = 1e-10; %Avoid domain errors due to limited accuracy of floating point numbers
            normalization = true;
        end
    end
    if normalization == true
        res(n,:) = res(n,:) / sum(res(n,:)); %Normalization
    end
end

%Calculate helper values  
N_eff = zeros(1,K);
for k=1:K
    N_eff(k) = sum(res(:,k));
end

%Remove components for which N_eff < removalCriteria
remove = find(N_eff < removalCriteria);
removeN = size(remove,2);
if removeN > 0
    K = K - removeN;
    
    res(:,remove) = [];
    mean(:,remove) = [];

    for n=1:N
        res(n,:) = res(n,:) / sum(res(n,:)); %Normalization
    end

    N_eff = zeros(1,K);
    for k=1:K
        N_eff(k) = sum(res(:,k));
    end
end
    
x_avg = zeros(D,K);
S = zeros(D,D,K);
for k=1:K
    for n=1:N
        x_avg(:,k) = x_avg(:,k) + res(n,k)*X(:,n);
    end    
    x_avg(:,k) = x_avg(:,k) / N_eff(k);

    for n=1:N
        S(:,:,k) = S(:,:,k) + res(n,k)*(X(:,n) - x_avg(:,k))*(X(:,n) - x_avg(:,k))';
    end
    S(:,:,k) = S(:,:,k) / N_eff(k);
end

%Update distribution parameters for each Gaussian    
alpha = zeros(1,K);
beta = zeros(1,K);
v = zeros(1,K);
W = zeros(D,D,K);

for k=1:K
    alpha(k) = alpha_0 + N_eff(k);  
end

for k=1:K    
    beta(k) = beta_0 + N_eff(k);
end

for k=1:K    
    v(k) = v_0 + N_eff(k);
end

for k=1:K    
    W(:,:,k) = inv(invW_0 + N_eff(k) * S(:,:,k) + (beta_0 * N_eff(k)) / (beta_0 + N_eff(k)) * (x_avg(:,k) - mean_0) * (x_avg(:,k) - mean_0)');
end

%Calculate pi_eff and lambda_eff
pi_eff = exp(psi(alpha)-psi(sum(alpha)));
lambda_eff = zeros(1,K);
for k=1:K
    temp = 0;
    for i=1:D
        temp = temp + psi((v(k) + 1 - i)/2);
    end
    lambda_eff(k) = exp(temp + D*log(2) + log(det(W(:,:,k))));
end

%Evaluate the cost
cost = costFunction(alpha_0, beta_0, v_0, W_0, invW_0, mean_0, res, N_eff, x_avg, S, alpha, beta, mean, v, W, pi_eff, lambda_eff, K, N, D);

args = {a, cost, searchDir, alpha_0, beta_0, v_0, W_0, invW_0, mean_0, res, N_eff, x_avg, S, alpha, beta, mean, v, W, pi_eff, lambda_eff, K, N, X, D, removalCriteria};