function [args evals] = quadraticSearch(fun, args, t1, t3, fun_t1, fun_t3, maxEvals, usePolynomialExtrap)
%Perform line search using quadratic polynomial interpolation. The cost
%function values are calculated by fun which takes the cell args
%corresponding to the current model as input and returns a similar cell
%corresponding to the model after the line search step. This function
%adjusts the step size args{1} based on the cost function values args{2}
%calculated by fun. args{3} contains the search direction. The args
%corresponding to the optimal step is returned along with the number of
%cost function evaluations required. See lineSearch_m_gamma.m for detailed
%contents of the cell args.

%Parameter t1 specifies the minimum step length (usually 0) and t3 an
%initial estimate for the magnitude of the optimal step (should be set to
%a value which is approximately twice the optimal step size). fun_t1 and
%fun_t3 must be set to NaN when calling the function. These are used in
%recursive function calls. maxEvals specifies the maximum number of cost
%function evaluations allowed. When usePolynomialExtrap is set to true,
%extrapolation with a quadratic polynomial is performed to obtain a
%configuration where interpolation can be performed. Otherwise, the search
%interval is either doubled or halved.

%The function includes a number of safeguard measures to ensure proper
%behaviour when the cost function is not convex in the seach direction or
%when an optimal step length is not found using the maximum allowed number
%of cost function evaluations.

%Copyright (C) 2008-2010 Mikael Kuusela.

t2 = t3/2;
evals = 0;

if ~iscell(fun_t1)
    args{1} = t1;
    fun_t1 = fun(args); %Evaluate fun at t1
    evals = evals + 1;
end

if evals >= maxEvals
    disp('Warning: Maximum number of cost function evaluations reached.');
    if(fun_t1{2} > fun_t3{2})
        args = fun_t3;
        return;
    else
        args = fun_t1;
        return;
    end
end
args{1} = t2;
fun_t2 = fun(args); %Evaluate fun at t2
evals = evals + 1;

if ~iscell(fun_t3)
    args{1} = t3;
    fun_t3 = fun(args); %Evaluate fun at t3
    evals = evals + 1;
end

if fun_t1{2} == fun_t2{2} && fun_t2{2} == fun_t3{2} %Cost function constant in search direction
    disp('Warning: cost function seems to be constant in search direction. Returning t1.');
    args = fun_t1;
    return;
end

%Obtain a configuration with fun_t1{2} > fun_t2{2} and fun_t2{2} < fun_t3{2}
if usePolynomialExtrap == true
    
    %Handle NaNs and Infs
    while(~isfinite(fun_t2{2}) || ~isfinite(fun_t3{2}))
        if evals >= maxEvals
            disp('Warning: Maximum number of cost function evaluations reached.');
            if(fun_t1{2} > fun_t3{2})
                args = fun_t3;
                return;
            elseif(fun_t1{2} > fun_t2{2})
                args = fun_t2;
                return;
            else
                args = fun_t1;
                return;
            end
        end
        t3 = t2;
        t2 = t2/2;
        fun_t3 = fun_t2;
        args{1} = t2;
        fun_t2 = fun(args);
        evals = evals + 1;
    end

    while(fun_t1{2} > fun_t2{2} && fun_t2{2} > fun_t3{2})
        if evals >= maxEvals
            disp('Warning: Maximum number of cost function evaluations reached. Returning t3.');
            args = fun_t3;
            return;
        end
        %Quadratic extrapolation/interpolation
        t_min = 0.5*((t2^2-t3^2)*fun_t1{2} + (t3^2-t1^2)*fun_t2{2} + (t1^2-t2^2)*fun_t3{2})/((t2-t3)*fun_t1{2} + (t3-t1)*fun_t2{2} + (t1-t2)*fun_t3{2});
        if t_min >= t3
            t2 = t3;
            t3 = 2*t_min;
            fun_t2 = fun_t3;
            args{1} = t3;
            fun_t3 = fun(args);
            evals = evals + 1;
            while(isnan(fun_t3{2}))
                t3 = 0.8*t3;
                if(t3 <= t2 | evals >= maxEvals)
                    disp('Warning: Returning t2.');
                    args = fun_t2;
                    return;
                end
                args{1} = t3;
                fun_t3 = fun(args);
                evals = evals + 1;
            end
        elseif (t_min > t2 && t_min < t3) % t2 < t_min < t3
            t2 = t_min;
            args{1} = t2;
            fun_t2 = fun(args);
            evals = evals + 1;
        else % t_min < t2
            t2 = t3;
            t3 = 2*t3;
            fun_t2 = fun_t3;
            args{1} = t3;
            fun_t3 = fun(args);
            evals = evals + 1;
            while(isnan(fun_t3{2}))
                t3 = 0.8*t3;
                if(t3 <= t2 | evals >= maxEvals)
                    disp('Warning: Returning t2.');
                    args = fun_t2;
                    return;
                end
                args{1} = t3;
                fun_t3 = fun(args);
                evals = evals + 1;
            end
        end

    end

    while(fun_t1{2} < fun_t2{2} && fun_t2{2} < fun_t3{2})
        if evals >= maxEvals
            disp('Warning: Maximum number of cost function evaluations reached. Returning t1.');
            args = fun_t1;
            return;
        end
        %Quadratic extrapolation/interpolation        
        t_min = 0.5*((t2^2-t3^2)*fun_t1{2} + (t3^2-t1^2)*fun_t2{2} + (t1^2-t2^2)*fun_t3{2})/((t2-t3)*fun_t1{2} + (t3-t1)*fun_t2{2} + (t1-t2)*fun_t3{2});
        if(t_min > 0 && t_min < t2)
            t3 = t2;
            t2 = t_min;
        else
            t3 = t2;
            t2 = t2/2;
        end
        fun_t3 = fun_t2;
        args{1} = t2;
        fun_t2 = fun(args);
        evals = evals + 1;
    end

else %usePolynomialExtrap == false
    
    while(fun_t1{2} > fun_t2{2} && fun_t2{2} > fun_t3{2})
        if evals >= maxEvals
            disp('Warning: Maximum number of cost function evaluations reached. Returning t3.');
            args = fun_t3;
            return;
        end
        t2 = t3;
        t3 = 2*t3;
        fun_t2 = fun_t3;
        args{1} = t3;
        fun_t3 = fun(args);
        evals = evals + 1;
    end

    while((fun_t1{2} < fun_t2{2} && fun_t2{2} < fun_t3{2}) | ~isfinite(fun_t2{2}) | ~isfinite(fun_t3{2}))
        if evals >= maxEvals
            disp('Warning: Maximum number of cost function evaluations reached. Returning t1.');
            args = fun_t1;
            return;
        end
        t3 = t2;
        t2 = t2/2;
        fun_t3 = fun_t2;
        args{1} = t2;
        fun_t2 = fun(args);
        evals = evals + 1;
    end
    
end

clear t_min;

%In case of non-convex configuration call recursively with smaller search range
if(fun_t1{2} < fun_t2{2} && fun_t2{2} > fun_t3{2})
    disp('Warning: Non-convex configuration.');
    [args evals2] = quadraticSearch(fun, args, t1, t2, fun_t1, fun_t2, maxEvals-evals, usePolynomialExtrap);
    evals = evals + evals2;
    return;
end

if(fun_t1{2} > fun_t2{2} && fun_t2{2} < fun_t3{2})
    %When a configuration with fun_t1{2} > fun_t2{2} and
    %fun_t2{2} < fun_t3{2} is obtained perform quadratic interpolation
    t_min = 0.5*((t2^2-t3^2)*fun_t1{2} + (t3^2-t1^2)*fun_t2{2} + (t1^2-t2^2)*fun_t3{2})/((t2-t3)*fun_t1{2} + (t3-t1)*fun_t2{2} + (t1-t2)*fun_t3{2});
    args{1} = t_min;
    fun_t_min = fun(args);
    if fun_t_min{2} < fun_t2{2}
        args = fun_t_min;
    else
        args = fun_t2;
    end
    return;
else
    %Desired point configuration was not obtained with maxEvals
    disp('Warning: Unexpected final configuration.');
    if(fun_t1{2} > fun_t2{2} && fun_t2{2} > fun_t3{2})
        args = fun_t3;
        return;
    elseif(fun_t1{2} > fun_t2{2})
        args = fun_t2;
        return;
    else
        args = fun_t1;
        return;
    end
end