function X = generateData(type, N, D)
%Generate artificial dataset. Type determines dataset type (explained in
%the comments below), N the number of data points to generate and D the
%dimensionality of the data. Netlab library is required to be installed.
%Copyright (C) 2008-2010 Mikael Kuusela.

path(path, '../netlab');

switch type
    case 1 %Two well separated clusters
        if D == 2
            mix = gmm(2,2,'full');
            rot = [cos(pi/4) -sin(pi/4); sin(pi/4) cos(pi/4)];
            mix.covars = repmat(rot' * [0.003 0; 0 0.03] * rot,[1 1 2]);
            mix.centres = [0.5 0.5; -0.5 -0.5];
            X = gmmsamp(mix,N)';
        else
            error('D must be 2');
        end
    case 2 %3 dimensional spiral
        if D == 3
            phi = 0/N:4*pi/(N-1):4*pi;
            r = random('Normal',0.7,0.1,1,N);
            z = 0:1/(N-1):1;
            [x,y,z] = pol2cart(phi,r,z);
            X=[x;y;z];
        else
            error('D must be 3');
        end
    case 3 %Five overlapping clusters
        if D >= 2
            mix = gmm(D,5,'spherical');
            mix.covars = [0.03 0.03 0.03 0.03 0.03];
            mix.priors = [0.2 0.2 0.2 0.2 0.2];
            mix.centres(1,:) = repmat(0, [1 D]);
            mix.centres(2,:) = repmat(0.3, [1 D]);
            mix.centres(3,:) = repmat(-0.3, [1 D]);        
            mix.centres(4,:) = repmat(0.3, [1 D]);
            mix.centres(5,:) = repmat(-0.3, [1 D]);
            mix.centres(4,2) = -mix.centres(4,2);
            mix.centres(5,2) = -mix.centres(5,2);        
            X = gmmsamp(mix,N)';
        else
            error('D must be at least 2');
        end            
    case 4 %3 overlapping components    
        if D == 2
            mix = gmm(2,3,'full');
            rot = [cos(pi/4) -sin(pi/4); sin(pi/4) cos(pi/4)];
            mix.covars(:,:,1) = rot' * [0.003 0; 0 0.3] * rot;
            mix.covars(:,:,2) = [0.03 0; 0 0.03];
            mix.covars(:,:,3) = [0.3 0; 0 0.3];            
            mix.centres = zeros(3,2);
            mix.priors = [0.5 0.3 0.2];
            X = gmmsamp(mix,N)';
        else
            error('D must be 2');
        end
    case 5 %Two overlapping one dimensional Gaussians
        if D == 1
            mix = gmm(1,2,'full');
            mix.covars(:,:,1) = 0.05;
            mix.covars(:,:,2) = 0.05;
            mix.centres(1) = -0.5;
            mix.centres(2) = 0.5;
            mix.priors = [0.5 0.5];
            X = gmmsamp(mix,N)';
        else
            error('D must be 1');
        end
    case 12 %3 dimensional spiral z=-1...1
        if D == 3
            phi = 0:4*pi/(N-1):4*pi;
            r = random('Normal',0.7,0.1,1,N);
            z = -1:2/(N-1):1;
            [x,y,z] = pol2cart(phi,r,z);
            X=[x;y;z];
        else
            error('D must be 3');
        end
    case 13 %3 dimensional spiral z=-1/2...1/2
    if D == 3
        phi = 0:4*pi/(N-1):4*pi;
        r = random('Normal',0.7,0.1,1,N);
        z = -1/2:1/(N-1):1/2;
        [x,y,z] = pol2cart(phi,r,z);
        X=[x;y;z];
    else
        error('D must be 3');
    end
    case 14 %3 dimensional spiral z=-1...1, 4 rounds
    if D == 3
        phi = 0:8*pi/(N-1):8*pi;
        r = random('Normal',0.7,0.1,1,N);
        z = -1:2/(N-1):1;
        [x,y,z] = pol2cart(phi,r,z);
        X=[x;y;z];
    else
        error('D must be 3');
    end    
    case 15 %3 dimensional spiral z=-1...1, weighted
    if D == 3
        phi = 0:1/(N-1):1;
        phi = 4*pi*phi.^(1/2);
        r = random('Normal',0.7,0.1,1,N);
        z = 0:1/(N-1):1;
        z = 2*z.^(1/2)-1;
        [x,y,z] = pol2cart(phi,r,z);
        X=[x;y;z];
    else
        error('D must be 3');
    end     
    case 16 %3 dimensional spiral z=-1...1, heavily weighted
    if D == 3
        phi = 0:1/(N-1):1;
        phi = 4*pi*phi.^(1/4);
        r = random('Normal',0.7,0.1,1,N);
        z = 0:1/(N-1):1;
        z = 2*z.^(1/4)-1;
        [x,y,z] = pol2cart(phi,r,z);
        X=[x;y;z];
    else
        error('D must be 3');
    end
    case 17 %3 dimensional spiral z=-1...1 spiral cone
    if D == 3
        phi = 0:8*pi/(N-1):8*pi;
        r = 0:0.7/(N-1):0.7;
        r = r + random('Normal',0,0.1,1,N);
        z = -1:2/(N-1):1;
        [x,y,z] = pol2cart(phi,r,z);
        X=[x;y;z];
    else
        error('D must be 3');
    end    
    otherwise
        error('Unknow data type');
end
