# -*- coding: iso-8859-1 -*-
#
# Copyright (C) 2001-2006 Markus Harva, Antti Honkela, Alexander
# Ilin, Tapani Raiko, Harri Valpola and Tomas stman.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License (included in file License.txt in the
# program package) for more details.
#
# $Id: run_varmod.py,v 1.1.1.1 2006/11/23 09:42:07 mha Exp $
#
""" Script for doing simulations with various variance models.

Usage:

  python run_varmod.py [options] base data
  python run_varmod.py --file=file data

Options are given in the long format as in --sdim=10. These are
shortly described in config.py. The base argument selects a base
configuration from config.py, consult it for the different
possibilities. The data should be a pickled Numeric array. If it is
given without a path, the data is tried to be loaded from DATADIR. After the
run is complete three files are saved to RESULTDIR. The first contains
the actual net, which can be examined for the results. The second
contains some history information i.e. the cost functions values
during the learning and the possible prunings and additions of weights
etc. The third file contains the parameters of the simulation so that it can
be repeated easily later on by calling this script with the file-switch.
"""


import sys
import time
import os.path
import getopt
import string, re

from bblocks import PickleHelpers

import util
import config
import blocks


# The directory data is fetched from
DATADIR = "."

# The resulting nets are saved in this directory
RESULTDIR = "results"


PARAMNAMES = ["sdim", "rdim", "sblock", "smap", "rblock", "rmap",
              "useux", "stox", "stoux", "rtos", "rtous", "rtox", "rtoux",
              "iters", "srclayer", "varlayer", "printcost",
              "prune", "prunefreq", "prunecost", "rotater",
              "add", "addfreq", "addcost",
              "sdecay", "rdecay",
              "clampwithvar",
              "file"]

PSEUDOPARAMS = ["base", "file"]

def usage():
    print """
    %s [options] base data
    %s --file=file data
    """ % (sys.argv[0], sys.argv[0])


def merge_configs(base, derived):
    conf = {}
    for key in base.keys():
        conf[key] = base[key]
    for key in derived.keys():
        conf[key] = derived[key]
    return conf

def create_config(base, opts):
    conf = config.configs[base]
    if conf.has_key("base"):
        baseconf = create_config(conf["base"], [])
        conf = merge_configs(baseconf, conf)
    for opt in opts:
        key = opt[0][2:]
        sval = opt[1]
        conf[key] = eval(sval)
    for param in PARAMNAMES:
        if not conf.has_key(param):
            conf[param] = None
    return conf

def read_config(filename):
    # use defaults
    conf = create_config("default", [])
    f = open(filename, "r")
    rexp = re.compile(r"\s*([^\s=]+)\s*=\s*([^\s]+)")
    line = f.readline()
    while line:
        m = rexp.match(line)
        if m:
            key = m.group(1)
            if key in PSEUDOPARAMS:
                val = m.group(2)
            else:
                val = eval(m.group(2))
            conf[key] = val
        else:
            raise "InvalidConfigFileException"
        line = f.readline()
    return conf

def write_config(conf, filename):
    f = open(filename, "w")
    for key in conf.keys():
        f.write("%s = %s\n" % (key, str(conf[key])))
    f.close

if __name__ == "__main__":
    optnames = map(lambda s: s + "=", PARAMNAMES)
    try:
        opts, args = getopt.getopt(sys.argv[1:], "", optnames)
    except getopt.GetoptError:
        usage()
        sys.exit(1)

    if len(args) == 1 and opts[0][0] == "--file":
        base = "file"
        dataname = args[0]
        optstr = "fromfile"
        conf = read_config(opts[0][1])
    elif len(args) == 2:
        base = args[0]
        dataname = args[1]
        myopts = map(lambda x: x[0][2]+x[1], opts)
        optstr = string.join(myopts, '_')
        conf = create_config(base, opts)
    else:
        usage()
        sys.exit()
        
    if dataname.find("/") == -1:
        datafile = DATADIR + "/" + dataname + ".pickle"
    else:
        datafile = dataname
        dataname = os.path.basename(datafile)

    timestamp = time.time()
    ident = "%s_%s_%s_%d" % (dataname, base, optstr, timestamp)
    netfilename = ident + ".net"
    historyfilename = ident + ".history"
    configfilename = ident + ".config"
 
    data = PickleHelpers.LoadWithPickle(datafile)
    model = util.do_model(conf, data)

    PickleHelpers.SaveWithPickle(model.net, RESULTDIR + "/" + netfilename)
    PickleHelpers.SaveWithPickle(model.history,
                                 RESULTDIR + "/" + historyfilename)
    write_config(conf, RESULTDIR + "/" + configfilename)
