function net = getnet( pythonnet, varargin )
%  GETNET.M
%
%  net = getnet( pythonnet ) converts the python network pythonnet
%   into the matlab structure with the network variables as its
%   fields. All the nodes corresponding to the same variable are
%   collected in one field. For example, python nodes with labels
%   'a(1)',..., 'a(n)' would be stored in matlab.a which would be a
%   vector of size n.
%
%  net = getnet( datafile ) does the same but loads the python network
%   from datafile.
%
%  The additional arguments can be given as a list as in
%  getnet( pythonnet, 'name1', val1, 'name2', val2, ...).
%
%  The recognised arguments are as follows:
%
%  'skipvars'           The variables that should not be saved in the
%                       matlab structure (for example, observed
%                       variables); netname - the name of
%                       the python network in datafile.
%  'netname'            The variables that should not be saved in the
%                       matlab structure (for example, observed
%                       variables); netname - the name of
%                       the python network in datafile.
%  'expectations'       1 - save only the expected value of each
%                           variable (default);
%                       0 - save all the relevant fields of variable
%                           nodes such as mean, var, ex.
%
%  Note: When saved to mat-files, indices of the python node labels
%   are changed so that they start from 1.
%

%
% This file is a part of the Bayes Blocks library
%
% Copyright (C) 2001-2003 Markus Harva, Antti Honkela, Alexander
% Ilin, Tapani Raiko, Harri Valpola and Tomas stman.
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2, or (at your option)
% any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License (included in file License.txt in the
% program package) for more details.
%
% $Id: getnet.m 4 2006-10-26 07:23:55Z ah $
%

% Default values of the arguments
args = struct( 'skipvars', {{}},...
               'netname', 'pythonnet',...
               'expectations', 1 );
if mod( length(varargin), 2 )
    error('GETNET: Keyword arguments should appear in pairs')
end
for ix = 1:2:length(varargin)
    if ~isfield( args, lower(varargin{ix}) )
        error( [ 'GETNET: Unknown parameter ' varargin{ix} ] )
    end
    args = setfield( args, lower(varargin{ix}), varargin{ix+1} );
end

if isstr(pythonnet)
    % The first argument is the mat-file containing a python network
    
    mydata = load(pythonnet);
    pythonnet = getfield(mydata, args.netname);
end

%  python{1} contains the information concerning the entire network,
%   e.g. pythonnet{1}.variables is the list of variable nodes.
%
%  python{2:end} contains information for different nodes
%   and therefore length(pythonnet) = 1 + pythonnet{1}.node_num


% Construct the list of variables
variables = {};
nvar = 0;
for i = 1:length(pythonnet{1}.variables)
    varname = pythonnet{1}.variables{i};
    fpr = find( varname == '(' );
    if ~isempty( fpr )
        varname = varname( 1:fpr-1 );
    end
    
    if  isempty( strmatch( varname, variables, 'exact' ) ) & ...
        isempty( strmatch( varname, args.skipvars, 'exact' ) )
        nvar = nvar + 1;
        variables{nvar} = varname;
    end
end

% Assign the variable values
net = getparm( pythonnet, args.expectations, variables{:} );

return
