//
// This file is a part of the Bayes Blocks library
//
// Copyright (C) 2001-2006 Markus Harva, Antti Honkela, Alexander
// Ilin, Tapani Raiko, Harri Valpola and Tomas stman.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License (included in file License.txt in the
// program package) for more details.
//
// $Id: Net.cc 7 2006-10-26 10:26:41Z ah $

#include "config.h"
#ifdef WITH_PYTHON
#include <Python.h>
#define __PYTHON_H_INCLUDED__
#endif

#include "Templates.h"
#include "Net.h"
#include "Node.h"
#include "Saver.h"
#include "XMLSaver.h"
#include "Loader.h"
#include "DecayCounter.h"
#include <iostream>
#include <sstream>
#include <stdlib.h>
#include <math.h>
#include <set>
#include <stack>
#include <utility>
#include <algorithm>
#include <functional>



Net::Net(size_t ti)
{
  t = ti;
  labelconst = 2;
  oldcost = 0;
  debuglevel = 0;
  dc = new TraditionalDecayCounter();

  activetimeindexgroup = NULL;
}

Net::~Net()
{
  for (int i = nodes.size() - 1; i >= 0; i--) {
    delete nodes[i];
  }

  delete dc;

  map<Label, VariableVector *>::iterator iter;
  for (iter = variablegroups.begin(); iter != variablegroups.end(); iter++) {
    VariableVector *vars = iter->second;
    delete vars;
  }

  map<Label, IntV *>::iterator iter2;
  for (iter2 = timeindexgroups.begin(); 
       iter2 != timeindexgroups.end(); iter2++) {
    IntV *group = iter2->second;
    delete group;
  }
}

double Net::Cost()
{
  double c = oldcost;

  for (size_t i = 0; i < variables.size(); i++) {
    c += variables[i]->Cost();
  }

  return c;
}

void Net::CleanUp()
{
  Node *ptr;

  while (!deadnodes.empty()) {
    ptr = deadnodes.back();
    RemovePtr(ptr);
    // ptr->~Node();
    delete ptr;
  }
}

void Net::AddVariable(Variable *ptr, Label label) 
{
  variables.push_back(ptr);
  variableindex[label] = ptr;

  AddVariableToGroups(ptr);
}

void Net::RemovePtr(Node *ptr)
{
  nodes.erase(remove(nodes.begin(), nodes.end(), ptr),
	      nodes.end());
  variables.erase(remove(variables.begin(), variables.end(), ptr),
		  variables.end());
  deadnodes.erase(remove(deadnodes.begin(), deadnodes.end(), ptr),
		  deadnodes.end());
  nodeindex.erase(ptr->GetLabel());
  variableindex.erase(ptr->GetLabel());

  Decayer *d = dynamic_cast<Decayer *>(ptr);
  if (d) {
    UnregisterDecay(d);
  }

  Variable *v = dynamic_cast<Variable *>(ptr);
  if (v) {
    RemoveVariableFromGroups(v);
  }
}

void Net::UpdateAll()
{
  CleanUp();

  if (activetimeindexgroup == NULL) {
    VariableRIterator i = variables.rbegin();
    while (i != variables.rend()) {
      (*i++)->Update();
    }
  } else {
    VariableRIterator i = variables.rbegin();
    while (i != variables.rend()) {
      (*i++)->PartialUpdate(activetimeindexgroup);
    }
  }

  ProcessDecayHook("UpdateAll");
}

void Net::UpdateTimeInd()
{
  CleanUp();
  for (size_t i = variables.size(); i > 0; i--)
    if (variables[i-1]->TimeType() == 0)
      variables[i-1]->Update();
  ProcessDecayHook("UpdateTimeInd");
}

bool Net::HasVariableGroup(Label group)
{
  return variablegroups.find(group) != variablegroups.end();
}

void Net::DefineVariableGroup(Label group)
{
  if (HasVariableGroup(group)) {
    throw StructureException("Group allready exists");
  }

  VariableVector *vars = new VariableVector();
  VariableIterator iter;

  for (iter = variables.begin(); iter != variables.end(); iter++) {
#ifdef BROKEN_STRING
    if ((*iter)->GetLabel().compare(group, 0, group.size()) == 0) {
      vars->push_back(*iter);
    }
#else
    if ((*iter)->GetLabel().compare(0, group.size(), group) == 0) {
      vars->push_back(*iter);
    }
#endif
  }

  variablegroups[group] = vars;
}

void Net::RemoveVariableFromGroups(Variable *ptr)
{
  map<Label, VariableVector *>::iterator iter;
  for (iter = variablegroups.begin(); iter != variablegroups.end(); iter++) {
    VariableVector *vars = iter->second;
    vars->erase(remove(vars->begin(), vars->end(), ptr), vars->end());
  }
}

void Net::AddVariableToGroups(Variable *ptr)
{
  map<Label, VariableVector *>::iterator iter;
  for (iter = variablegroups.begin(); iter != variablegroups.end(); iter++) {
    const Label &group = iter->first;
    VariableVector *vars = iter->second;
#ifdef BROKEN_STRING
    if (ptr->GetLabel().compare(group, 0, group.length()) == 0) {
      vars->push_back(ptr);
    }
#else
    if (ptr->GetLabel().compare(0, group.length(), group) == 0) {
      vars->push_back(ptr);
    }
#endif
  }
}

size_t Net::NumGroupVariables(Label group)
{
  if (HasVariableGroup(group)) {
    return variablegroups[group]->size();
  } else {
    throw StructureException("No such group.");
  }
}

Variable *Net::GetGroupVariable(Label group, size_t index)
{
  if (index < NumGroupVariables(group)) {
    return (*variablegroups[group])[index];
  } else {
    throw StructureException("Index out of range.");
  }
}

/* Updates the variables in the group. */

void Net::UpdateGroup(Label group)
{
  if (variablegroups.find(group) == variablegroups.end()) {
    throw StructureException("Group doesn't exist");
  }

  CleanUp();
  VariableVector *g = variablegroups[group];
  VariableIterator iter;

  if (activetimeindexgroup == NULL) {
    for (iter = g->begin(); iter != g->end(); iter++) {
      (*iter)->Update();
    }
  } else {
    for (iter = g->begin(); iter != g->end(); iter++) {
      (*iter)->PartialUpdate(activetimeindexgroup);
    }
  }
}

/* Calculates the cost arising from the group. */

double Net::CostGroup(Label group)
{
  if (variablegroups.find(group) == variablegroups.end()) {
    throw StructureException("Group doesn't exist");
  }

  VariableVector *g = variablegroups[group];
  double c = oldcost;

  for (VariableIterator iter = g->begin(); 
       iter != g->end(); iter++) {
    c += (*iter)->Cost();
  }

  return c;
}

bool Net::HasTimeIndexGroup(Label group)
{
  return timeindexgroups.find(group) != timeindexgroups.end();

}

void Net::DefineTimeIndexGroup(Label group, IntV &indices)
{
  if (HasTimeIndexGroup(group)) {
    throw StructureException("Time index group allready exists");
  }

  // We have to copy indices because it might get freed by Python's GC
  IntV *ind = new IntV(indices.size());
  copy(indices.begin(), indices.end(), ind->begin());

  timeindexgroups[group] = ind;
}


void Net::EnableTimeIndexGroup(Label group)
{
  if (activetimeindexgroup != NULL) {
    throw StructureException("Some time index group is allready active");
  }

  if (!HasTimeIndexGroup(group)) {
    throw StructureException("No such group");
  }
  
  activetimeindexgroup = timeindexgroups[group];
}


void Net::DisableTimeIndexGroup(Label group)
{
  if (activetimeindexgroup == NULL 
      || !HasTimeIndexGroup(group)
      || timeindexgroups[group] != activetimeindexgroup) {
    throw StructureException("Time index group is not active");
  }

  activetimeindexgroup = NULL;
}

void Net::UpdateTimeDep()
{
  CleanUp();
  for (size_t i = variables.size(); i > 0; i--)
    if (variables[i-1]->TimeType() == 1)
      variables[i-1]->Update();
  ProcessDecayHook("UpdateTimeDep");
}

void Net::SaveAllStates() {
  for (size_t i = variables.size(); i > 0; i--)
    variables[i-1]->SaveState();
}

void Net::SaveAllSteps() {
  for (size_t i = variables.size(); i > 0; i--)
    variables[i-1]->SaveStep();
}

void Net::RepeatAllSteps(double alpha) {
  for (size_t i = variables.size(); i > 0; i--)
    variables[i-1]->RepeatStep(alpha);
}

void Net::ClearAllStatesAndSteps() {
  for (size_t i = variables.size(); i > 0; i--)
    variables[i-1]->ClearStateAndStep();
}

void Net::StepTime()
{
  CleanUp();
  dc->StepTime();
  for (size_t i = variables.size(); i > 0; i--)
    switch (variables[i-1]->TimeType()) {
    case 0:
      break;
    case 1:
      oldcost += variables[i-1]->Cost();
      break;
    case 2:
      variables[i-1]->Update();
    }
  oldcost *= Decay();
  for (size_t i = oldelays.size(); i > 0; i--)
    oldelays[i-1]->StepTime();
  ProcessDecayHook("StepTime");
}

void Net::ResetTime()
{
  dc->StepTime();
  for (size_t i = variables.size(); i > 0; i--)
    switch (variables[i-1]->TimeType()) {
    case 0:
      break;
    case 1:
      oldcost += variables[i-1]->Cost();
      break;
    case 2:
      variables[i-1]->Update();
    }
  oldcost *= Decay();
  for (size_t i = oldelays.size(); i > 0; i--)
    oldelays[i-1]->ResetTime();
  ProcessDecayHook("ResetTime");
}

void Net::SetDecayCounter(DecayCounter *d)
{
  if (dc)
    free(dc);
  dc = d;
}

double Net::Decay()
{
  return dc->GetDecay();
}

// Compare the first element of a pair against a fixed value
template <class _Pair>
struct Compare1st : public unary_function<_Pair, bool>
{
  Compare1st(typename _Pair::first_type __x): myval(__x) { }

  bool operator()(const _Pair& __x) const {
    return myval == __x.first;
  };
private:
  typename _Pair::first_type myval;
};

// Topologically sort the nodes
void Net::SortNodes()
{
  vector<Node *> sortednodes;
  std::set<Node *> addednodes;
  vector< pair<Node *, size_t> > dfs_stack;
  Node *curnode;
  Node *parent = 0;
  size_t parind;

  // Reserve space for sorted sequence of nodes
  sortednodes.reserve(nodes.size());

  // Go through the list of nodes in the order they are in the vector
  // (in most cases they should already be sorted...)
  for (size_t i = 0; i < nodes.size(); i++) {
    curnode = nodes[i];
    // This node has already found its place
    if (addednodes.count(curnode))
      continue;
    // Add the node to the stack of things to process
    dfs_stack.push_back(pair<Node *, size_t>(curnode, 0));
    // While there are nodes in the depth first search stack...
    while (!dfs_stack.empty()) {
      // Check the current node and index of the parent to process
      curnode = dfs_stack.back().first;
      parind = dfs_stack.back().second;
      // Proxies have no real parents and so we skip the checking of parents
      // While there are more parents
      if ((curnode->GetType() != "Proxy") &&
	  (parent = curnode->GetParent(parind)) != 0) {
	// Increment the index of parent to process next
	dfs_stack.back().second++;
	// If parent has not already been processed
	if (!(addednodes.count(parent))) {
	  // Check that we haven't already passed through the parent
	  // and push it to the stack to be processed next
	  if ((find_if(dfs_stack.begin(), dfs_stack.end(),
		       Compare1st< pair<Node *, size_t> >(parent))
	       == dfs_stack.end())) {
	    dfs_stack.push_back(pair<Node *, size_t>(parent, 0));
	  }
	  // We've been here before.  This really shouldn't happen
	  // if the graph is a DAG.
	  else {
	    throw StructureException();
	  }
	}
      }
      // All parents have been accounted for, so this is the place for
      // our brave little node.
      else {
	sortednodes.push_back(curnode);
	(void)addednodes.insert(curnode);
	dfs_stack.pop_back();
      }
    }
  }
  // And we're done.
  nodes = sortednodes;
}

void Net::CheckStructure()
{
  std::set<Node *> checkednodes;
  std::set<Node *> tocheck;
  std::set<Node *>::iterator curnode;
  vector<Node *> stack;
  Node *node;

  CleanUp();
  SortNodes();

  for (size_t i = 0; i < variables.size(); i++) {
    tocheck.insert((Node *)variables[i]);
  }
  for (size_t i = 0; i < nodes.size(); i++) {
    if (nodes[i]->GetType() == "DelayV") {
      tocheck.insert(nodes[i]);
    }
  }

  for (curnode = tocheck.begin(); curnode != tocheck.end(); ++curnode) {
    stack.clear();
    stack.push_back(*curnode);
    checkednodes.clear();
    checkednodes.insert(*curnode);
    while (!stack.empty()) {
      node = stack.back();
      stack.pop_back();
      for (size_t i = 0; i < node->NumChildren(); i++) {
	if (checkednodes.find(node->GetChild(i)) != checkednodes.end()) {
	  throw StructureException("Invalid structure of the network. More than one route from " +
				   (*curnode)->GetLabel() + " to " +
				   node->GetChild(i)->GetLabel() + ".");

	}
	checkednodes.insert(node->GetChild(i));
	if (tocheck.find(node->GetChild(i)) == tocheck.end()) {
	  stack.push_back(node->GetChild(i));
	}
      }
    }
  }
}

bool Net::RegisterDecay(Decayer *d, string hook) {
  multimap<string, Decayer *>::iterator i;

  for (i = decay_hooks.lower_bound(hook);
       i != decay_hooks.upper_bound(hook); i++) {
    if ((*i).second == d)
      return false;
  }
  if (GetDebugLevel() > 10)
    cout << "Registering decay " << d << " for hook " << hook << endl;
  decay_hooks.insert(multimap<string, Decayer *>::value_type(hook, d));
  d->RegisterToHook(hook);
  return true;
}

bool Net::UnregisterDecay(Decayer *d) {
  multimap<string, Decayer *>::iterator i;
  bool retval = false;

  i = decay_hooks.begin();
  while (i != decay_hooks.end()) {
    if ((*i).second == d) {
      if (GetDebugLevel() > 10)
	cout << "Removing decay " << d << " from hook " << (*i).first << endl;
      d->EraseHook((*i).first);
      decay_hooks.erase(i);
      retval = true;
      i = decay_hooks.begin();
    }
    else
      i++;
  }

  return retval;
}

bool Net::UnregisterDecayFromHook(Decayer *d, string hook) {
  multimap<string, Decayer *>::iterator i;

  for (i = decay_hooks.lower_bound(hook);
       i != decay_hooks.upper_bound(hook); i++) {
    if ((*i).second == d) {
      if (GetDebugLevel() > 10)
	cout << "Removing decay " << d << " from hook " << hook << endl;
      d->EraseHook(hook);
      decay_hooks.erase(i);
      return true;
    }
  }
  return false;
}

bool Net::ProcessDecayHook(string hook) {
  multimap<string, Decayer *>::iterator i;
  std::set<Decayer *> killlist;
  bool val, rval = true;

  if (GetDebugLevel() > 10)
    cout << "Processing hook " << hook << endl;
  for (i = decay_hooks.lower_bound(hook);
       i != decay_hooks.upper_bound(hook); i++) {
    if (GetDebugLevel() > 10)
      cout << "Calling it for " << (*i).second << endl;
    val = ((*i).second)->DoDecay(hook);
    if (!val) {
      killlist.insert((*i).second);
      rval = false;
    }
  }

  if (!rval) {
    std::set<Decayer *>::iterator i;

    for (i = killlist.begin(); i != killlist.end(); i++) {
      UnregisterDecay(*i);
    }
  }
  CleanUp();
  return rval;
}


void Net::SetSumNKeepUpdated(bool keepupdated)
{
  string type;

  sumnkeepupdated=keepupdated;
  for (size_t i=0; i<nodes.size(); i++) {
    type = nodes[i]->GetType();
    //if type.compare("SumNV",0,5) ((SumNV *)nodes[i])->SetKeepUpdated(keepupdated);
    //else if type.compare("SumN",0,4) ((SumN *)nodes[i])->SetKeepUpdated(keepupdated);
    if (type=="SumN") ((SumN *)nodes[i])->SetKeepUpdated(keepupdated);
    if (type=="SumNV") ((SumNV *)nodes[i])->SetKeepUpdated(keepupdated);
  }
}


void Net::Save(NetSaver *saver)
{
  CleanUp();
  SortNodes();

  saver->StartNet(nodes.size() + 1, "net");

  saver->StartNamedCont("header");

  saver->StartEnumCont(variables.size(), "variables");
  for (size_t i=0; i<variables.size(); i++)
    saver->SetLabel(variables[i]->GetLabel());
  saver->FinishEnumCont("variables");

  saver->SetNamedInt("t", (int)t);
  saver->SetNamedInt("labelconst", labelconst);
  saver->SetNamedInt("debuglevel", debuglevel);
  saver->SetNamedInt("node_num", (int)nodes.size());
  saver->SetNamedDouble("oldcost", oldcost);
  saver->StartNamedCont("decaycounter");

  dc->Save(saver);
  saver->FinishNamedCont("decaycounter");
  saver->FinishNamedCont("header");

  for (size_t i=0; i<nodes.size(); i++) {
    //cout << i << ": " << nodes[i]->GetIdent() << endl;

    saver->StartNode(nodes[i]->GetType());
    nodes[i]->Save(saver);
    saver->FinishNode(nodes[i]->GetType());
  }

  saver->FinishNet("net");
}

#ifdef WITH_MATLAB
#include "MatlabSaver.h"

void Net::SaveToMatFile(string fname, string varname, bool debugsave)
{
  NetSaver *saver = new NetSaver( new MatlabSaver(fname, varname), debugsave);
  this->Save(saver);
  saver->SaveIt();
  delete saver;
}
#else
void Net::SaveToMatFile(string fname, string varname, bool debugsave)
{
  throw MatlabException("No Matlab support in this library version");
}
#endif  // WITH_MATLAB

void Net::SaveToXMLFile(string fname, bool debugsave)
{
  NetSaver *saver = new NetSaver( new XMLSaver(fname), debugsave );
  this->Save(saver);
  saver->SaveIt();
  delete saver;
}

void Net::SaveNodeToXMLFile(string fname, Node *node, bool debugsave)
{
  NetSaver *saver = new NetSaver( new XMLSaver(fname), debugsave );
  node->Save(saver);
  saver->SaveIt();
  delete saver;
}

#ifdef WITH_PYTHON
#include "PythonSaver.h"
PyObject *Net::SaveToPyObject(bool debugsave)
{
  PyObject *res;
  PythonSaver *psaver = new PythonSaver();

  NetSaver *saver = new NetSaver(psaver, debugsave);
  this->Save(saver);
  res = psaver->GetTheNet();

  delete saver;
  //delete psaver;

  return res;
}
#endif  // WITH_PYTHON


Label Net::GetNextLabel(Label label)
{
  map<Label, Node *>::iterator it = nodeindex.find(label);

  if (it == nodeindex.end())
    return label;

  int i=0;
  Label newlabel = label;
  ostringstream ss;
  ss << label << '_' << i;
  newlabel = ss.str();

  while (nodeindex.find(newlabel) != nodeindex.end()) {
    i = labelconst*i + 1 + (int)(((double)labelconst)*rand()/(RAND_MAX+1.0));
    ostringstream ss;
    ss << label << '_' << i;
    newlabel = ss.str();
  }

  return newlabel;
}

bool Net::ConnectProxies()
{
  bool retval = true;
  for (size_t i=proxies.size(); i>0; i--)
    retval &= proxies[i-1]->CheckRef();
  return retval;
}

#ifdef WITH_MATLAB

#include "MatlabLoader.h"

Net * LoadFromMatFile( string fname, string varname )
{
  NetLoader * loader = 0;
  try {
    loader = new NetLoader(new MatlabLoader( fname, varname ) );

    loader->LoadIt();
  }
  catch (...) {
    if (loader)
      delete loader;
    throw;
  }

  Net       * mynet = new Net( loader );

  delete loader;

  return mynet;
}
#else
Net * LoadFromMatFile( string fname, string varname )
{
  throw MatlabException("No Matlab support in this library version");
}
#endif // WITH_MATLAB


/**********************************
**  Load from file
*/
Net::Net(NetLoader *loader)
{
  size_t            node_num = 0;
  string            type;
  map<string,int>   typeNum;          // Map of node types
  Node            * newnode;
  int temp;
  ostringstream msg;

  // Initialize the node type map
  typeNum[ "Constant" ]     = cn_Constant;
  typeNum[ "ConstantV" ]    = cn_ConstantV;

  typeNum[ "Prod" ]         = cn_Prod;
  typeNum[ "ProdV" ]        = cn_ProdV;
  typeNum[ "Sum2" ]         = cn_Sum2;
  typeNum[ "Sum2V" ]        = cn_Sum2V;
  typeNum[ "SumN" ]         = cn_SumN;
  typeNum[ "SumNV" ]        = cn_SumNV;
  typeNum[ "Rectification" ] = cn_Rectification;
  typeNum[ "RectificationV" ] = cn_RectificationV;
  typeNum[ "DelayV" ]       = cn_DelayV;

  typeNum[ "Gaussian" ]     = cn_Gaussian;
  typeNum[ "GaussianV" ]    = cn_GaussianV;
  typeNum[ "DelayGaussV" ]  = cn_DelayGaussV;
  typeNum[ "SparseGaussV" ] = cn_SparseGaussV;
  typeNum[ "RectifiedGaussian" ] = cn_RectifiedGaussian;
  typeNum[ "RectifiedGaussianV" ] = cn_RectifiedGaussianV;
  typeNum[ "GaussRect" ] = cn_GaussRect;
  typeNum[ "GaussRectV" ] = cn_GaussRectV;
  typeNum[ "GaussNonlin" ]  = cn_GaussNonlin;
  typeNum[ "GaussNonlinV" ] = cn_GaussNonlinV;
  typeNum[ "MoG" ] = cn_MoG;
  typeNum[ "MoGV" ] = cn_MoGV;
  typeNum[ "Discrete" ]     = cn_Discrete;
  typeNum[ "DiscreteV" ]    = cn_DiscreteV;
  typeNum[ "DiscreteDirichlet" ] = cn_DiscreteDirichlet;
  typeNum[ "DiscreteDirichletV" ] = cn_DiscreteDirichletV;
  typeNum[ "Dirichlet" ] = cn_Dirichlet;

  typeNum[ "Proxy" ]        = cn_Proxy;
  typeNum[ "Relay" ]        = cn_Relay;
  typeNum[ "Evidence" ]     = cn_Evidence;
  typeNum[ "EvidenceV" ]    = cn_EvidenceV;

  typeNum[ "Memory" ]       = cn_Memory;
  typeNum[ "OLDelayS" ]     = cn_OLDelayS;
  typeNum[ "OLDelayD" ]     = cn_OLDelayD;


  loader->StartNet("net");

  loader->StartNamedCont("header");

  loader->StartEnumCont("variables");
  loader->FinishEnumCont("variables");

  loader->GetNamedInt("t", temp);
  t = temp;
  loader->GetNamedInt("labelconst", labelconst);
  loader->GetNamedInt("debuglevel", debuglevel);
  loader->GetNamedInt("node_num", temp);
  node_num = temp;
  loader->GetNamedDouble("oldcost", oldcost);

  loader->StartNamedCont("decaycounter");
  dc = DecayCounter::GlobalLoader(loader);
  loader->FinishNamedCont("decaycounter");

  loader->FinishNamedCont("header");

  activetimeindexgroup = NULL;

  for (size_t i=0; i<node_num; i++) {
    loader->StartNode(type);

    switch( typeNum[ type ] ) {
    case  0: // Doesn't exist
      msg << "Unknown node in save file, claims to be " << type;
      throw TypeException(msg.str());
      break;

    case cn_Constant:
      newnode = new Constant( this, loader ); break;
    case cn_ConstantV:
      newnode = new ConstantV( this, loader ); break;

    case cn_Prod:
      newnode = new Prod( this, loader ); break;
    case cn_ProdV:
      newnode = new ProdV( this, loader ); break;
    case cn_Sum2:
      newnode = new Sum2( this, loader ); break;
    case cn_Sum2V:
      newnode = new Sum2V( this, loader ); break;
    case cn_SumN:
      newnode = new SumN( this, loader ); break;
    case cn_SumNV:
      newnode = new SumNV( this, loader ); break;
    case cn_Rectification:
      newnode = new Rectification( this, loader ); break;
    case cn_RectificationV:
      newnode = new RectificationV( this, loader ); break;
    case cn_DelayV:
      newnode = new DelayV( this, loader ); break;

    case cn_Gaussian:
      newnode = new Gaussian( this, loader ); break;
    case cn_GaussianV:
      newnode = new GaussianV( this, loader ); break;
    case cn_DelayGaussV:
      newnode = new DelayGaussV( this, loader ); break;
    case cn_SparseGaussV:
      newnode = new SparseGaussV( this, loader ); break;
    case cn_RectifiedGaussian:
      newnode = new RectifiedGaussian( this, loader ); break;
    case cn_RectifiedGaussianV:
      newnode = new RectifiedGaussianV( this, loader ); break;
    case cn_GaussRect:
      newnode = new GaussRect( this, loader ); break;
    case cn_GaussRectV:
      newnode = new GaussRectV( this, loader ); break;
    case cn_GaussNonlin:
      newnode = new GaussNonlin( this, loader ); break;
    case cn_GaussNonlinV:
      newnode = new GaussNonlinV( this, loader ); break;
    case cn_MoG:
      newnode = new MoG( this, loader ); break;
    case cn_MoGV:
      newnode = new MoGV( this, loader ); break;
    case cn_Discrete:
      newnode = new Discrete( this, loader ); break;
    case cn_DiscreteV:
      newnode = new DiscreteV( this, loader ); break;
    case cn_DiscreteDirichlet:
      newnode = new DiscreteDirichlet( this, loader); break;
    case cn_DiscreteDirichletV:
      newnode = new DiscreteDirichletV( this, loader); break;
    case cn_Dirichlet:
      newnode = new Dirichlet( this, loader); break;

    case cn_Proxy:
      newnode = new Proxy( this, loader ); break;
    case cn_Relay:
      newnode = new Relay( this, loader ); break;
    case cn_Evidence:
      newnode = new Evidence( this, loader ); break;
    case cn_EvidenceV:
      newnode = new EvidenceV( this, loader ); break;

    case cn_Memory:
      newnode = new Memory( this, loader ); break;
    case cn_OLDelayS:
      newnode = new OLDelayS( this, loader ); break;
    case cn_OLDelayD:
      newnode = new OLDelayD( this, loader ); break;

    default:
      msg << "Net::Net(loader): No handler for node type " << type;
      throw TypeException(msg.str());
    }
    
    loader->FinishNode(type, bind1st(mem_fun(&Node::ReallyAddParent), newnode));
    //cout << i << ": " << newnode->GetIdent() << endl;
  }

  loader->FinishNet("net");

  this->ConnectProxies();

} // End of "Net::Net"

#ifdef WITH_PYTHON
#include "PythonLoader.h"

Net *CreateNetFromPyObject(PyObject *obj)
{
  NetLoader *loader = new NetLoader(new PythonLoader(obj));
  Net *mynet = new Net(loader);
  delete loader;
  return mynet;
}
#endif  // WITH_PYTHON
