// -*- C++ -*-

//
// This file is a part of the Bayes Blocks library
//
// Copyright (C) 2001-2006 Markus Harva, Antti Honkela, Alexander
// Ilin, Tapani Raiko, Harri Valpola and Tomas stman.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License (included in file License.txt in the
// program package) for more details.
//
// $Id: Templates.h 7 2006-10-26 10:26:41Z ah $

#ifndef TEMPLATES_H
#define TEMPLATES_H

#ifndef BUILDING_SWIG_INTERFACE
#include "config.h"
#endif
#include <vector>
#include <string>
#include <math.h>
#include <stdexcept>
#include <sstream>
#include <iostream>

using namespace std;

typedef string Label;

#ifndef BUILDING_SWIG_INTERFACE
#ifndef BB_NDEBUG
#ifdef WITH_PYTHON
#define BBASSERT(expr) \
do { \
  if (!(expr)) { \
    ostringstream message; \
    message << "Assertion failed in " << __FILE__ << " at line " \
            << __LINE__ << ": " << #expr; \
    throw AssertionFailedException(message.str()); \
  } \
} while (false)
#define BBASSERT2(expr) \
do { \
  if (!(expr)) { \
    ostringstream message; \
    message << GetLabel() << ": Assertion failed in " << __FILE__ \
            << " at line " << __LINE__ << ": " << #expr; \
    throw AssertionFailedException(message.str()); \
  } \
} while (false)
#else // defined(WITH_PYTHON)
#define BBASSERT(expr) \
do { \
  if (!(expr)) { \
    cerr << "Assertion failed in " << __FILE__ << " at line " \
         << __LINE__ << ": " << #expr << endl; \
    exit(1); \
  } \
} while (false)
#define BBASSERT2(expr)
#endif // defined(WITH_PYTHON)
#else
#define BBASSERT(expr)
#define BBASSERT2(expr)
#endif // !defined(DDEBUG)

#endif // BUILDING_SWIG_INTERFACE

typedef vector<double> DV;
typedef vector<int> IntV;

class AssertionFailedException : public runtime_error
{
public:
  AssertionFailedException(string msg) : runtime_error(msg) { ; }
};

class TypeException : public std::invalid_argument
{
public:
  TypeException(string _msg) : std::invalid_argument(_msg) { }
};

class MatlabException : public std::runtime_error
{
public:
  MatlabException(string _msg) : std::runtime_error(_msg) { }
};

class StructureException : public runtime_error
{
public:
 StructureException() :
   std::runtime_error("Invalid structure of the network") { }
 StructureException(string _msg) : std::runtime_error(_msg) { }
};

class DFlags
{
public:
  DFlags(bool m=false, bool v=false, bool e=false) {mean=m; var=v; ex=e;}
  bool AllFalse() {return !mean && !var && !ex;}
  void Add(DFlags f) {mean |= f.mean; var |= f.var; ex |= f.ex;}
  bool mean, var, ex;
};

class DSSet
{
public:
  DSSet(const DSSet &d) : mean(d.mean), var(d.var), ex(d.ex) { }
  DSSet(double m=0, double v=0, double e=0) : mean(m), var(v), ex(e) { }
#ifndef BUILDING_SWIG_INTERFACE
  DSSet& operator=(const DSSet &d) {
    mean=d.mean; var=d.var; ex=d.ex;
    return *this;
  }
#endif

  double mean, var, ex;
};

class DVSet
{
public:
  DVSet() {}
  DVSet(const DVSet &val) {
    mean.resize(val.mean.size());
    for (size_t i = 0; i < val.mean.size(); i++) {
      mean[i] = val.mean[i];
    }
    var.resize(val.var.size());
    for (size_t i = 0; i < val.var.size(); i++) {
      var[i] = val.var[i];
    }
    ex.resize(val.ex.size());
    for (size_t i = 0; i < val.ex.size(); i++) {
      ex[i] = val.ex[i];
    }
  }

#ifndef BUILDING_SWIG_INTERFACE
  DVSet & operator=(const DVSet &val) {
    mean.resize(val.mean.size());
    for (size_t i = 0; i < val.mean.size(); i++) {
      mean[i] = val.mean[i];
    }
    var.resize(val.var.size());
    for (size_t i = 0; i < val.var.size(); i++) {
      var[i] = val.var[i];
    }
    ex.resize(val.ex.size());
    for (size_t i = 0; i < val.ex.size(); i++) {
      ex[i] = val.ex[i];
    }
    return *this;
  }
#endif

  DV mean, var, ex;
};

class DVH
{
public:
  DVH() {vec = 0;}
  DSSet scalar;
  DVSet *vec;
  double &Mean(int i) {return vec ? vec->mean[i] : scalar.mean;}
  double &Var(int i) {return vec ? vec->var[i] : scalar.var;}
  double &Exp(int i) {return vec ? vec->ex[i] : scalar.ex;}
};

class DD
{
public:
  DD(int i=0) { val = new DV(i); }
  ~DD() { if (val) delete val; }
  DD(const DD &d) { 
    val = new DV(d.val->size());
    for (size_t i=0; i<d.val->size(); i++) (*val)[i] = (*d.val)[i];
  }
#ifndef BUILDING_SWIG_INTERFACE
  DD &operator=(const DD &d) {
    if (this != &d) {
      Resize(d.val->size());
      for (size_t i=0; i<d.val->size(); i++) (*val)[i] = (*d.val)[i];
    }
    return *this;
  }
  double &operator[](int i) { return Get(i); }
#endif
  DV *GetDV() const { return val; }
  double &Get(int i) { return (*val)[i]; }
  void Set(int i, double d) { (*val)[i] = d; }
  //void Resize(int i) { if (val) delete val; val = new DV(i); }
  void Resize(int i) {
    if (val) {
      val->resize(i);
    } else {
      cerr << "val is null, should never happen!" << endl;
      throw StructureException();
    }
  }
  size_t size() const { return val->size(); }

  double Minimum() const {
    double minval = (*val)[0];
    for (size_t i = 1; i < val->size(); i++) {
      if ((*val)[i] < minval) {
	minval = (*val)[i];
      }
    }
    return minval;
  }

  friend class VDD;

private:
  DV *val;
};

class VDD
{
public:
  VDD(size_t i=0, size_t j=0) {
    val = new vector<DD *>(i);
    for (size_t k=0; k<i; k++)
      (*val)[k] = new DD(j);
  }
  ~VDD() {
    if (val)
      for (size_t i=0; i<val->size(); i++)
	if((*val)[i]) delete (*val)[i];
    delete val;
  }
  VDD(const VDD &d) { 
    val = new vector<DD *>(d.val->size());
    for (size_t i=0; i<d.val->size(); i++) {
      (*val)[i] = new DD(*((*d.val)[i]));
    }
  }
#ifndef BUILDING_SWIG_INTERFACE
  VDD &operator=(const VDD &d) {
    if (this != &d) {
      Resize(d.val->size());
      for (size_t i=0; i<d.val->size(); i++) {
	(*val)[i] = new DD(*((*d.val)[i]));
      }
    }
    return *this;
  }
  DD &operator[](int i) { return *(*val)[i]; }
  const DD &operator[](int i) const { return *(*val)[i]; }
#endif
  double &Get(int i, int j) { return (*((*val)[i]))[j]; }
  void Set(int i, int j, double d) { (*((*val)[i]))[j] = d; }
  DD *&GetDDp(int i) { return (*val)[i]; }
  size_t size() const { return val->size(); }
  size_t DDsize() const { 
    if (val->size() > 0) {
      return (*val)[0]->size();
    } else {
      return 0;
    }
  }
//   void Resize(int i) { 
//     if (val) delete val; val = new vector<DD *>(i); }
  void Resize(int newsize) {
    if (val)
      val->resize(newsize);
    else
      val = new vector<DD *>(newsize);

    for (int i=0; i < newsize; i++) {
      if (!(*val)[i])
	(*val)[i] = new DD();
    }
  }
  void ResizeDD(size_t newsize) {
    for (size_t i = 0; i < val->size(); i++) {
      if (!(*val)[i]) {
	(*val)[i] = new DD(newsize);
      } else {
	(*val)[i]->Resize(newsize);
      }
    }
  }

private:
  vector<DD *> *val;
};

class VDDH {
public:
  VDDH() {scalar = 0; vec = 0;}
  DD *scalar;
  VDD *vec;
#ifndef BUILDING_SWIG_INTERFACE
  DD &operator[](int i) { return vec ? (*vec)[i] : *scalar; }
#endif
};

#endif // TEMPLATES_H
