function [x, t, c] = linesearch(varargin),
% LINESEARCH  Perform a line search to find optimal step length

% Copyright (C) 1999-2006 Antti Honkela, Harri Valpola,
% Xavier Giannakopoulos, and Matti Tornio.
%
% This package comes with ABSOLUTELY NO WARRANTY; for details
% see License.txt in the program package.  This is free software,
% and you are welcome to redistribute it under certain conditions;
% see License.txt for details.

[x, t, c] = cubic_search(varargin{:});


function [x, t2, c2] = cubic_search(x0, step, t0, f, args, c0)
% CUBIC_SEARCH  Line search by method of cubic approximation

% Copyright (C) 1999-2004 Antti Honkela, Harri Valpola,
% Xavier Giannakopoulos, and Matti Tornio.
%
% This package comes with ABSOLUTELY NO WARRANTY; for details
% see License.txt in the program package.  This is free software,
% and you are welcome to redistribute it under certain conditions;
% see License.txt for details.


epsilon = 1e-6;  % Stopping criterion for the step size
epsilon2 = 1e-8; % Stopping criterion for the cost function
maxiters = 30;   % Maximum number of line search iterations

t0 = max(epsilon, t0);
t1 = 0;
t2 = .5 * t0;
t4 = nan;

if nargin > 5,
  c1 = c0;
  x1 = x0;
else
  [x1, c1] = feval(f, x0, step, t1, args);
  c0 = c1;
end
[x2, c2] = feval(f, x0, step, t2, args);
t3 = t0;
[x3, c3] = feval(f, x0, step, t3, args);
c4 = nan;

while ((c1 > c2) & (c2 > c3)),
  tnew = (t1.^2 .* (c2-c3) + t2.^2 .* (c3-c1) + t3.^2 .* (c1-c2)) ./ ...
	 (2*(t1 .* (c2-c3) + t2 .* (c3-c1) + t3 .* (c1-c2)));

  tnew = min([max([tnew .75*t3]) 5*t3]);
  t4 = c2;
  c4 = c2;
  c2 = c3;
  x2 = x3;
  t2 = t3;
  t3 = 2*tnew;
  [x3, c3] = feval(f, x0, step, t3, args);
  if args.status.debug,
    fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, c2, t2, c3, t3);
  end
end

while (((c1 < c2) & (c2 < c3)) | ((c1 < c2) & (c2 > c3)) | ...
       ~isfinite(c2) | ~isfinite(c3)),
  tnew = (t1.^2 .* (c2-c3) + t2.^2 .* (c3-c1) + t3.^2 .* (c1-c2)) ./ ...
	 (2*(t1 .* (c2-c3) + t2 .* (c3-c1) + t3 .* (c1-c2)));
  tnew = min([max([tnew .1*t2]) .5*t2]);
  t4 = t3;
  c4 = c3;
  c3 = c2;
  x3 = x2;
  t3 = t2;
  t2 = tnew;
  [x2, c2] = feval(f, x0, step, t2, args);
  if args.status.debug,
    fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, c2, t2, c3, t3);
  end
end

if ((c1 < c2) & (c2 > c3)),
  warning('Non-convex point configuration for line search');
  keyboard;
end

itercount = 0;
while (((t3 - t1) > epsilon) & ...
       ((abs(c2-c3) + abs(c3-c1) + abs(c1-c2)) > epsilon2)),

  tnew = (t1.^2 .* (c2-c3) + t2.^2 .* (c3-c1) + t3.^2 .* (c1-c2)) ./ ...
         (2*(t1 .* (c2-c3) + t2 .* (c3-c1) + t3 .* (c1-c2)));

  % Use cubic interpolation if valid
  if ~isnan(t4),
    tnew2 = cubic_minimum([t1 t2 t3 t4], [c1 c2 c3 c4]);
    % Check if the cubic approximation is valid, if not, revert to quadratic
    if tnew2 > t1 & tnew2 < t3 & isreal(tnew2),
      tnew = tnew2;
    end
  end



  if (tnew == t2),
    warning('NLFA:LINESEARCH:neweqold', 'LINESEARCH: proposed new point equal to old midpoint -> trouble...');
    if (t3 - t2) > (t2 - t1),
      tnew = t2 + .1 * (t3 - t2);
    else
      tnew = t2 - .1 * (t3 - t2);
    end
  end
  
  [xnew, cnew] = feval(f, x0, step, tnew, args);
  
  if (tnew > t2),
    if (cnew > c2),
      t4 = t3;
      c4 = c3;
      t3 = tnew;
      x3 = xnew;
      c3 = cnew;
    else
      t4 = t1;
      c4 = c1;
      t1 = t2;
      x1 = x2;
      c1 = c2;
      t2 = tnew;
      x2 = xnew;
      c2 = cnew;
    end
  else    % tnew < t2
    if (cnew > c2),
      t4 = t1;
      c4 = c1;
      t1 = tnew;
      x1 = xnew;
      c1 = cnew;
    else
      t4 = t3;
      c4 = c3;
      t3 = t2;
      x3 = x2;
      c3 = c2;
      t2 = tnew;
      x2 = xnew;
      c2 = cnew;
    end
  end
  if args.status.debug,
    fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, c2, t2, c3, t3);
  end

  itercount = itercount + 1;
  if itercount > maxiters,
    [x, t2] = goldsect_search(x0, step, t2, f, args, c0);
    return;
  end
end

x = x2;



function [x, t2, c2] = quadratic_search(x0, step, t0, f, args, c0)
% QUADRATIC_SEARCH  Line search by method of quadratic approximation

% Copyright (C) 1999-2004 Antti Honkela, Harri Valpola,
% and Xavier Giannakopoulos.
%
% This package comes with ABSOLUTELY NO WARRANTY; for details
% see License.txt in the program package.  This is free software,
% and you are welcome to redistribute it under certain conditions;
% see License.txt for details.

epsilon = 1e-6;
epsilon2 = 1e-8;
maxiters = 30;

t1 = 0;
if nargin > 5,
  c1 = c0;
  x1 = x0;
else
  [x1, c1] = feval(f, x0, step, t1, args);
  c0 = c1;
end
t2 = .5 * t0;
[x2, c2] = feval(f, x0, step, t2, args);
t3 = t0;
[x3, c3] = feval(f, x0, step, t3, args);

while ((c1 > c2) & (c2 > c3)),
  c2 = c3;
  x2 = x3;
  t2 = t3;
  t3 = 2*t3;
  [x3, c3] = feval(f, x0, step, t3, args);
  fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, c2, t2, c3, t3);
end

while (((c1 < c2) & (c2 < c3)) | ((c1 < c2) & (c2 > c3)) | ~isfinite(c2) | ~isfinite(c3)),
  c3 = c2;
  x3 = x2;
  t3 = t2;
  t2 = .5*t2;
  [x2, c2] = feval(f, x0, step, t2, args);
  fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, c2, t2, c3, t3);
end

if ((c1 < c2) & (c2 > c3)),
  warning('Non-convex point configuration for line search');
  keyboard;
end

itercount = 0;
while (((t3 - t1) > epsilon) & ...
       ((abs(c2-c3) + abs(c3-c1) + abs(c1-c2)) > epsilon2)),
  tnew = (t1.^2 .* (c2-c3) + t2.^2 .* (c3-c1) + t3.^2 .* (c1-c2)) ./ ...
	 (2*(t1 .* (c2-c3) + t2 .* (c3-c1) + t3 .* (c1-c2)));

  if (tnew == t2),
    warning('NLFA:LINESEARCH:neweqold', 'LINESEARCH: proposed new point equal to old midpoint -> trouble...');
    if (t3 - t2) > (t2 - t1),
      tnew = t2 + .1 * (t3 - t2);
    else
      tnew = t2 - .1 * (t3 - t2);
    end
  end
  
  [xnew, cnew] = feval(f, x0, step, tnew, args);
  
  if (tnew > t2),
    if (cnew > c2),
      t3 = tnew;
      x3 = xnew;
      c3 = cnew;
    else
      t1 = t2;
      x1 = x2;
      c1 = c2;
      t2 = tnew;
      x2 = xnew;
      c2 = cnew;
    end
  else    % tnew < t2
    if (cnew > c2),
      t1 = tnew;
      x1 = xnew;
      c1 = cnew;
    else
      t3 = t2;
      x3 = x2;
      c3 = c2;
      t2 = tnew;
      x2 = xnew;
      c2 = cnew;
    end
  end
  fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, c2, t2, c3, t3);

  itercount = itercount + 1;
  if itercount > maxiters,
    [x, t2] = goldsect_search(x0, step, t2, f, args, c0);
    return;
  end
end

x = x2;



function [x, t2, c2] = goldsect_search(x0, step, t0, f, args, c0)
% GOLDSECT_SEARCH  Line search by golden section method

% Copyright (C) 1999-2004 Antti Honkela, Harri Valpola,
% and Xavier Giannakopoulos.
%
% This package comes with ABSOLUTELY NO WARRANTY; for details
% see License.txt in the program package.  This is free software,
% and you are welcome to redistribute it under certain conditions;
% see License.txt for details.

epsilon = 1e-6;
alpha = 0.6180339887498949;

t1 = 0;
if nargin > 5,
  c1 = c0;
  x1 = x0;
else
  [x1, c1] = feval(f, x0, step, t1, args);
end
t2 = (1 - alpha) * t0;
[x2, c2] = feval(f, x0, step, t2, args);
t3 = t0;
[x3, c3] = feval(f, x0, step, t3, args);

while ((c1 > c2) & (c2 > c3)),
  c2 = c3;
  x2 = x3;
  t2 = t3;
  t3 = t3 / (1 - alpha);
  [x3, c3] = feval(f, x0, step, t3, args);
end

while (((c1 < c2) & (c2 < c3)) | ((c1 < c2) & (c2 > c3)) | ~isfinite(c2) | ~isfinite(c3)),
  c3 = c2;
  x3 = x2;
  t3 = t2;
  t2 = t2 * (1 - alpha);
  [x2, c2] = feval(f, x0, step, t2, args);
end

if ((c1 < c2) & (c2 > c3)),
  warning('Non-convex point configuration for line search');
  keyboard;
end

%l = t1 + (1-alpha) * (t3-t1);
l = t2;
m = t1 + alpha     * (t3-t1);

xl = x2;
cl = c2;
[xm, cm] = feval(f, x0, step, m, args);

%fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, cl, l, cm, m, c3, t3);

while (((t3 - t1) > epsilon)),

  if cl > cm,
    t1 = l;
    c1 = cl;
    l = m;
    xl = xm;
    m = t1 + alpha     * (t3-t1);
    cl = cm;
    [xm, cm] = feval(f, x0, step, m, args);
  else
    t3 = m;
    c3 = cm;
    m = l;
    xm = xl;
    l = t1 + (1-alpha) * (t3-t1);
    cm = cl;
    [xl, cl] = feval(f, x0, step, l, args);
  end
  %fprintf('%.2f (%.4g) %.2f (%.4g) %.2f (%.4g) %.2f (%.4g)\n', c1, t1, cl, l, cm, m, c3, t3);
end

if cl < cm,
  t0 = l;
  x = xl;
else
  t0 = m;
  x = xm;
end
