%  DYNICA.M
%
%  VB independent component analysis with temporally correlated
%  sources. The data must be centered. By default, the sources are
%  fixed to the principal components of the data.
%
%  Usage: To initialise the model and start learning:
%
%  net = dynica( data, iter, 'searchsources', 2,...
%                            'initsources', s_init,...
%                            'diag_qs', 0 )
%  or
%  net = dynica( data, iter, 'searchsources', 2,...
%                            'initA', A_init,...
%                            'diag_qs', 0 )
%
%  To continue learning of the existing model:
%  net = dynica( data, iter, net, status )
%
function [ net, status ] = dynica( data, iter, varargin )

[ xdim, tdim ] = size(data);

if isstruct(varargin{1})

    % The network is passed to the function
    [ A, s, vx, Vvx, Mvx, B, vs, Mvs, Vvs, x ] = ...
        net2dynica( varargin{1}, data );
        
    if nargin < 4 | isempty(varargin{2})
        error( 'DYNICA: Unspecifed status' )
    else
        status = varargin{2};
    end
    
    CF = status.CF;
    complete_iters = status.iter;
    diag_qs = status.diag_qs;
    
    if isfield( status, 'A_clamped' )
        A.clamped = status.A_clamped;
    else
        A.clamped = 0;
    end

    optional_flds = { 's_clamped', 'B_clamped', 'Mvx_clamped',...
                      'Vvx_clamped', 'vx_clamped', 'Mvs_clamped',...
                      'Vvs_clamped', 'vs_clamped' };
    for i = 1:length(optional_flds)
        if isfield( status, optional_flds{i} )
            eval( [ optional_flds{i} ...
                    '= status.' optional_flds{i} ';' ] )
        else
            eval( [ optional_flds{i} '= 0;' ] )
        end
    end

else
    
    % Default parameter values
    options = struct( 'searchsources', [], ...
                      'inita', [], ...
                      'initsources', [], ...
                      'diag_qs', 0 );
    
    if mod( length(varargin), 2 )
        error( 'DYNICA: Not enough arguments' )
    end
    for i = 1:2:length(varargin)
        if ~isfield( options, lower(varargin{i}) )
            error( [ 'DYNICA: Unknown parameter: ' varargin{i} ] )
        end
        options = setfield( options, lower(varargin{i}),...
                                     varargin{i+1} );
    end
    
    sdim = options.searchsources;
    if isempty( sdim )
        error( 'DYNICA: Unspecified number of sources' )
    end
    diag_qs = options.diag_qs;

    %
    % Create the model
    %
    
    % Diagonal dynamics matrix
    B.mean = zeros( sdim, 1 );
    B.var = ones( sdim, 1 );

    % Mvs, Vvs, vs
    Mvs.mean = 0;
    Mvs.var  = 1;

    Vvs.mean = 0;
    Vvs.var  = 1;
    Vvs.mex  = meanexp( Vvs );

    vs.mean = repmat( Mvs.mean, sdim, 1 );
    vs.var  = repmat( 1/Vvs.mex, sdim, 1 );
    vs.mex  = meanexp( vs );

    % Sources
    s(1).mean = zeros(sdim,1);
    s(1).covar = diag( 1./( exp(0.5) + vs.mex .* ( B.mean.^2 + B.var ) ) );
    for t = 2:tdim-1
        s(t).mean = zeros(sdim,1);
        s(t).covar = diag( 1./( vs.mex .* ( 1 + B.mean.^2 + B.var ) ) );
    end
    s(tdim).mean = zeros(sdim,1);
    s(tdim).covar = diag( 1./vs.mex );

    % Mixing matrix
    A.mean = zeros( xdim, sdim );
    A.var = ones( xdim, sdim );

    % vx, Mvx, Vvx
    Mvx.mean = 0;
    Mvx.var  = 1;

    Vvx.mean = 0;
    Vvx.var  = 1;
    Vvx.mex  = meanexp( Vvx );

    vx.mean = repmat( Mvx.mean, xdim, 1 );
    vx.var  = repmat( 1/Vvx.mex, xdim, 1 );
    vx.mex  = meanexp( vx );

    % Data
    for t = 1:tdim
        x(t).value = data(:,t);
    end

    %
    % Initialization
    %
    if isempty( options.initsources ) & isempty( options.inita )
        
        % Initialize sources with PCA
        C = data*data'/tdim;
        [ V, L ] = eig(C);
        
        [ L, id ] = sort( -diag(L) );
        V = V(:,id);
        options.initsources = V( :, 1:sdim )' * data;
        
    end
    if ~isempty( options.initsources )
        s_clamped = 1;
        for t = 1:tdim
            s(t).mean = options.initsources(:,t);
            s(t).covar = zeros(sdim,sdim);
        end
    else
        s_clamped = 0;
    end
    if ~isempty( options.inita )
        A.clamped = 1;
        A.mean = options.inita;
        A.var = zeros( xdim, sdim );
    else
        A.clamped = 0;
    end
    
    B_clamped = 0;

    Mvx_clamped = 0;
    Vvx_clamped = 0;
    vx_clamped = 0;

    Mvs_clamped = 0;
    Vvs_clamped = 0;
    vs_clamped = 0;
    
    CF = [];
    
    complete_iters = 0;
    status.diag_qs = diag_qs;

end

    %
    % Learning
    %
for i = complete_iters+1:iter
    
    fprintf( '%d-', i );

    if ~vx_clamped
        vx = update_vx( vx, Mvx, Vvx, x, A, s, diag_qs );
    end
    if ~Vvx_clamped
        Vvx = update_Vvx( Vvx, Mvx, vx );
    end
    if ~Mvx_clamped
        Mvx = update_Mvx( Mvx, Vvx, vx );
    end
    
    if ~A.clamped
        A = update_A( A, s, vx, x );
    end

    if ~s_clamped
        s = update_sd( s, B, vs, A, vx, x, diag_qs );
    end

    if ~B_clamped
        B = update_B( B, s, vs );
    end
    
    if ~vs_clamped
        vs = update_vsd( vs, Mvs, Vvs, s, B );
    end
    if ~Vvs_clamped
        Vvs = update_Vvx( Vvs, Mvs, vs );
    end
    if ~Mvs_clamped
        Mvs = update_Mvx( Mvs, Vvs, vs );
    end

end

net = dynica2net( A, s, B, vs, Vvs, Mvs, vx, Vvx, Mvx );

status.iter = iter;
status.CF = CF;
status.A_clamped = A.clamped;
status.s_clamped = s_clamped;
status.B_clamped = B_clamped;

status.Mvx_clamped = Mvx_clamped;
status.Vvx_clamped = Vvx_clamped;
status.vx_clamped = vx_clamped;

status.Mvs_clamped = Mvs_clamped;
status.Vvs_clamped = Vvs_clamped;
status.vs_clamped = vs_clamped;

return
