%  RES_TRACK.M
%
%  res_track( netfile, initA ) plots the trajectory of the mixing
%  matrix A in the subspace of the true A during learning. The results
%  are taken from files starting with 'netfile', the initialisations
%  of A for which the results should be displayed are listed in the
%  cell array initA. By default, both 'ICA' and 'PCA' initialisations
%  are displayed. The function works only for two sources.
%
%  Examples of usage:
%   res_track( 'RESULTS/Full_Qs/dyn8_-8', 'PCA' )
%   res_track( 'RESULTS/Full_Qs/sg_1_0', { 'PCA', 'ICA' } )
%   res_track( 'RESULTS/Full_Qs/mog__sg_1_0', 'PCA' )
%
%   res_track( 'PAPER_RES/Full_Qs/dyn8_8' )
%   res_track( 'PAPER_RES/Full_Qs/sg_1_0' )
%   res_track( 'PAPER_RES/Full_Qs/mog__sg_1_0' )
%
function res_track( netfile, initA )

if nargin < 2
    initA = { 'PCA', 'ICA' };
else
    if ~iscell(initA)
        initA = { initA };
    end
end

pos_slash = findstr( netfile, '/' );
datafile = netfile( pos_slash(end)+1 : end );
pos__ = findstr( datafile, '__' );
if ~isempty(pos__)
    datafile = datafile( pos__(1)+2:end );
end

    %
    % The data
    %
load( [ package_dir '/DATA/' datafile ], 'A', 'Apca' )
prA = orth(A)';
Apca_2 = prA*Apca;
A_2 = prA*A;

    %
    % The results
    %
track_ax = [];
for i = 1:length(initA)

    nf = [ netfile '__' initA{i} ];
    
    [ net, nit{i}, CF{i} ] = MakeTrack( nf, datafile );
    CFcalculated(i) = ~isempty(CF{i});
    
    if ~isempty( net );

        if isempty(CF{i})
            fprintf( '\n%s: %d iterations\n', nf, nit{i}(end) )
        else
            fprintf( '\n%s: %d iterations, cost: %f\n',...
                     nf, nit{i}(end), CF{i}(end) )
        end
        
        if 0
            Ishow = [1,2:2:length(nit{i})];
            nit{i} = nit{i}(Ishow);
            nit{i}
            net = { net{Ishow} }
        else
            disp( nit{i} )
        end
        
        track_ax = plot_track( net, prA, A_2, Apca_2, track_ax );
        
        hPCA = plot_dir( Apca_2, [-10 10], [-10 10], ...
                         'Color', [ 0 0 0 ], 'LineStyle', '--' );
        hICA = plot_dir( A_2, [-10 10], [-10 10], ...
                         'Color', [ 0 0 0 ], 'LineStyle', '-.' );
    end
    
end

if isempty( track_ax )
    disp( 'No files found' )
else

    axes( track_ax )
    legend( [ hPCA(1) hICA(1) ], 'PCA', 'ICA', 4 )
    xlims = get( track_ax, 'XLim' ) + [ -0.5 0.5 ];
    ylims = get( track_ax, 'YLim' ) + [ -0.5 0.5 ];
    set( track_ax, 'XLim', xlims, 'YLim', ylims )
    set( gcf, 'Name', netfile )

    % Plot the cost function
    if any( CFcalculated )
        actfig( [ netfile ': Cost function' ] );
        for i = 1:length(CF)
            if ~isempty(CF{i})
                subplot( length(CF), 1, i )
                
                if nit{i}(end) == length(CF{i})
                    % CF saved at each iterations
                    nit2plot = nit{i}(2)-1:nit{i}(end);
                    plot( nit2plot, CF{i}( nit2plot ) )
                else
                    % CF saved at nit(j) only
                    plot( nit{i}(2:end), CF{i}(2:end) )
                end
                title( [ initA{i} ': the cost function' ] )
                %set( gca, 'xtick', [ 0:100:500, 1000:500:nit(end) ] )
                grid on
            end
        end
    end
    
end

return

% Final solution
net = net{end};

net.a

plot_ts( net.s )

se = net.s;
Ae = net.a;
xe = Ae * se;

%plot_ts(xe-x)

figure( h(3) )
plot_vect( prA*Ae, 'k*-' )

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The function has been changed and not tested for the previuos
% versions: python, ddynfa_qs, ddynfa_qs_cont, varfa_qs, varfa_qs_cont
function [ net, nit, CF ] = MakeTrack( netfile, datafile )

if netfile(1:3) == 'QS/'
    python_res = 0;
    diag_qs = 0;
    
elseif netfile(1:5) == 'DGQS/'
    python_res = 0;
    diag_qs = 1;

else
    python_res = 1;
    diag_qs = 1;
end


netfile = [ package_dir '/' netfile ]

% Check for the file with packed results
D = dir( [ netfile '.mat' ] );
if isempty(D)
    net = {}; nit = []; CF = [];
else
    load( netfile, 'net', 'nit', 'CF' )
end


% Pack the intermediate results into one file
D = dir( [ netfile '_i*.mat' ] );

if isempty(D)
    
    if ~python_res & length(CF) < length(net)
        CF = CostTrack( net, diag_qs, datafile, CF );
        save( netfile, 'CF', '-append' )
    end
    
else

    fprintf( '%d net files found ... ', length(D) )
    
    for ix = 1:length(D)
        filename = D(ix).name;
        pos_i = findstr( filename, '_i' );
        pos_dot = findstr( filename, '.' );
        nit_new(ix) = str2num( filename( pos_i+2 : pos_dot-1 ) );
    end
    [ nit_new, I ] = sort( nit_new );

    if ~isempty(nit) & nit_new(1) <= nit(end)
        %R = ''; % Uncomment for asking
        R = 'y';
        while ~strcmp(R,'y')
            fprintf( '\nOld simulations found in %s\n', netfile )
            R = input( 'Do you want to overwrite the results? (y/n): ',...
                       's' );
            R = lower(R);
            if strcmp(R,'n')
                return
            end
        end
        net = {}; nit = []; CF = [];
    end

    packed = length(net);
    if python_res
        % Python net results
        
        for ix = 1:length(I)
            filename = [ netfile '_i' int2str(nit_new(ix)) ];
            %fprintf( '%d: %s\n', ix, filename );
            
            net{packed+ix} =...
                getnet( filename, 'skipvars', {'x'}, 'expectations', 1 );
            if ix == length(I)
                net_end = getnet( filename,...
                                  'skipvars', {'x'}, 'expectations', 0 );
            end
            delete( [ filename '.mat' ] )
        end
        fprintf( 'deleted\n' )

        % Scan for the cost function values
        filename = [ netfile '_CF.txt' ];
        if ~isempty( dir( filename ) )
            fid = fopen( filename, 'r' );
            [ new_CF, cnt ] = fscanf( fid, '%f' );
            fclose(fid);
            delete( filename )
            CF = [ CF new_CF ];
        end
    
    else
        % Matlab results
        
        for ix = 1:length(I)
            filename = [ netfile '_i' int2str(nit_new(ix)) ];
            results = load( filename );
            if isfield( results, 'net' )
                net{packed+ix} = results.net;
            end
            delete( [ filename '.mat' ] )
        end
        fprintf( 'deleted\n' )
    
        if length(CF) < length(net)
            CF = CostTrack( net, diag_qs, datafile, CF );
        end
    
    end
    
    nit = [ nit nit_new ];
    
    if python_res
        save( netfile, 'net', 'net_end', 'nit', 'CF' )
    else
        save( netfile, 'net', 'nit', 'CF' )
    end        
    disp( [ 'The results written to ' netfile ] )

end

return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function CF = CostTrack( net, diag_qs, datafile, CF )

fprintf( 'Calculating %d cost values: ', length(net) )
if isfield( net{1}, 'bs' )
    % DDYNFA
    for i = length(CF)+1:length(net)
        fprintf( '-%d', i )
        [ A, s, vx, Vvx, Mvx,...
           B, vs, Mvs, Vvs, x ] = net2ddynfa( net{i}, datafile );
        if any(A.var == 0)
            CF(i) = -inf;
        else
            CF(i) = cost_ddynfa( x, A, s, vx, Vvx, Mvx,...
                                 B, vs, Mvs, Vvs, diag_qs );
        end
    end
    
elseif isfield( net{1}, 'u' )
    % VARFA
    for i = length(CF)+1:length(net)
        fprintf( '-%d', i )
        [ A, s, vx, Vvx, Mvx, u, Mmu, Vmu, Mvu, Vvu, mu, vu, x ] = ...
            net2varfa( net{i}, datafile );
        if any(A.var == 0)
            CF(i) = -inf;
        else
            CF(i) = cost_varfa( x, A, s, vx, Vvx, Mvx, u,...
                                Mmu, Vmu, Mvu, Vvu, mu, vu, diag_qs );
        end
    end
    
end
fprintf( ' done\n' )

return

