`ICMg.combined.sampler` <-
function(L, X, C, alpha=10, beta=0.01, pm0=0, V0=1, V=0.1, B.num=8, B.size=100, S.num=20, S.size=10, C.boost=1) {

  cat("Sampling ICMg2...\n")
  ## L is Nx2 matrix of link endpoints
  ## X is MXD matrix of node data (real value vectors)
  ## C is the number of latent components
  ## alpha and beta are the hyperparameters
  ## pm0, V0 and V are the node data parameters

  ## B.size is the size of one burnin round,
  ## after which convergence and component counts are reported
  ## B.num is the number of these burnin rounds
  ## S size is likewise the size of one sample round, after which samples are taken
  ## S.num is the number of these sample rounds
  ## C.boost = 1 uses faster C functions for sampling

  N <- dim(L)[1]                # The number of links
  M <- dim(X)[1]                # The number of nodes
  D <- dim(X)[2]                # The number of observations in the expression data

  cat("nodes:",N,"links:",M,"observations:",D,"components:",C,"alpha:",alpha,"beta:",beta,"\n")
  
  out <- list()
  out$z <- vector("integer", N)    # link l is generated by component z[l]
  out$w <- vector("integer", M)    # node k is generated by component w[k]
  out$n <- vector("integer", C)    # n[z] links generated by component z
  out$m <- vector("integer", C)    # m[w] nodes generated by component w
  out$q <- matrix(0, C, M)       # q[z,i] links generated by z end at i, {{and + 1, if node k generated by z}}
  out$E <- matrix(0, C, D) # \sum_k^{m_w} x_k Sum of datas generated by component w

  ## Initialize vectors randomly
  out$z <- sample(1:C, N, replace=TRUE)
  out$w <- sample(1:C, M, replace=TRUE)
  out$n <- as.vector(table(factor(out$z, levels=1:C)))
  out$m <- as.vector(table(factor(out$w, levels=1:C)))

  ## Initialize k and E according to z and w
  for (l in 1:N) {
    out$q[out$z[l], L[l,1]] <- out$q[out$z[l], L[l,1]]+1
    out$q[out$z[l], L[l,2]] <- out$q[out$z[l], L[l,2]]+1
  }
  for (k in 1:M) {
    out$q[out$w[k],k] <- out$q[out$w[k],k] +1
    out$E[out$w[k],] <- out$E[out$w[k], ] + X[k, ]
  }

  ## Initialize C random number generator
  ICMg.randominit()

  ## Initialize some necessary data structures
  samples <- list(z=matrix(0, S.num, N), w=matrix(0, S.num, M),
                  convl=vector("numeric", B.num+S.num),
                  convn=vector("numeric", B.num+S.num),
                  countsl=matrix(0,B.num+S.num,C),
                  countsn=matrix(0,B.num+S.num,C))
  convl <- vector("numeric", N)
  convn <- vector("numeric", M)
  
  cat("Sampling", B.num*B.size + S.num*S.size,"iterationcs\n")
  
  ## Burnin
  cat("Burnin iterations:",B.num*B.size,"\n")
  cat("I: 0\n")
  cat("n(z):",out$n,"\n")
  cat("m(z):",out$m,"\n")

  for (b in 1:B.num) {

    ## Randomize links and nodes
    Lindices <- sample(N)
    Nindices <- sample(M)   
    ## Sample components for links and nodes
    out <- ICMg.combined.wrapper(L, X, B.size, N, M, Lindices, Nindices,
                         D, C, out$z, out$w, out$n, out$m, out$q, out$E,
                         alpha, beta, pm0, V0, V, convl, convn, C.boost)
    ## Save and print convergence monitoring and component counts
    samples$convl[b] <- mean(log(out$convl))
    samples$countsl[b,] <- out$n
    samples$convn[b] <- mean(log(out$convn))
    samples$countsn[b,] <- out$m
    cat("I:",b*B.size,"\n")
    cat("convL:",samples$convl[b],"n(z):",samples$countsl[b,],"\n")
    cat("convN:",samples$convn[b],"m(z):",samples$countsn[b,],"\n")
  }

  ## Samples
  cat("Sample iterations:",S.num*S.size,"\n")
  for (s in 1:S.num) {

    ## Randomize links and nodes
    Lindices <- sample(N)
    Nindices <- sample(M) 
    ## Sample components for links and nodes
    out <- ICMg.combined.wrapper(L, X, B.size, N, M, Lindices, Nindices,
                         D, C, out$z, out$w, out$n, out$m, out$q, out$E,
                         alpha, beta, pm0, V0, V, convl, convn, C.boost)
    ## Save and print convergence monitoring and component counts
    samples$convl[B.num+s] <- mean(log(out$convl))
    samples$countsl[B.num+s,] <- out$n
    samples$convn[B.num+s] <- mean(log(out$convn))
    samples$countsn[B.num+s,] <- out$m
    cat("I:",B.num*B.size+s*S.size,"\n")
    cat("convL:",samples$convl[B.num+s],"n(z):",samples$countsl[B.num+s,],"\n")
    cat("convN:",samples$convn[B.num+s],"m(z):",samples$countsn[B.num+s,],"\n")
    ## Take samples
    samples$z[s,] <- out$z
    samples$w[s,] <- out$w
  }

  ## Save parameters
  samples$B.num <- B.num
  samples$B.size <- B.size
  samples$S.num <- S.num
  samples$S.size <- S.size
  samples$alpha <- alpha
  samples$beta <- beta
  samples$pm0 <- pm0
  samples$V0 <- V0
  samples$V <- V

  cat("DONE\n")
  return(samples)
}

