% Main script example for running the EOF algorithm

% Load data file
load Data/ExampleDataset.mat;
[rows, columns] = size(data);

% Necessary parameters, set by user
% Number of Monte Carlo rounds to use in validation
nMonteCarlo = 10;
% Percentage of data to use in each validation round
percentMonteCarlo = 10;
% Initialization method for the EOF
initialization = 'column';
% Maximum number of EOF to use
maxeof = 20;
% Stopping value, defines the minimum amount of change in the data set
% without stopping the EOF process
stopping = 0.1;
% Maximum number of EOF rounds done, if stopping value not fulfilled
rounds = 1000;

% Testset selection
[data2, testData, testIndex] = SelectSet(data, percentMonteCarlo);

% Dataset normalization. Remember to DeNormalize before testerror calculation!
% Normalization does not do anything to the missing values
[dataNorm, normMeans, normStds] = Normalize(data2, 'meancols', 'stdcols');

% Results
valErrors = ones(nMonteCarlo, maxeof) * inf;


%% Learning part
for mc = 1:nMonteCarlo
  % Validationset selection
  [dataMC, valData, valIndex] = SelectSet(dataNorm, percentMonteCarlo);
  
  % Initialization
  [dataInit, mask] = InitMissing(dataMC, initialization);
  
  % EOF estimations for the validation set values
  outputs = EOFMulti(dataInit, mask, maxeof, 'original', valIndex, ...
    stopping, rounds);
  
  % Validation error calculation for each eof
  valErrors(mc, :) = mean((outputs - repmat(valData, maxeof, 1)) .^2, 2);
end


%% Testing part
% Selecting the number of EOF with smallest validation error ...
[aa, bb] = min(mean(valErrors,1));

% ... and plotting the validation errors for all EOF
figure;
plot(mean(valErrors,1));
drawnow;

% Now the test itself
% Initialization
[dataInit, maskTest] = InitMissing(dataNorm, initialization);

% Filling
dataFilled = EOFCore(dataInit, maskTest, bb, stopping, rounds);

% Denormalization, using the reversed order than in Normalization
dataFilled = DeNormalize(dataFilled, 'std', normStds, 'mean', normMeans);

% Test error calculation
testError = mean((dataFilled(testIndex) - testData) .^2)


%% Final filling of the data set
% In case there's missing values
if any(isnan(data))
  % Dataset normalization. Remember to DeNormalize!
  % Normalization does not do anything to the missing values
  [dataNorm, normMeans, normStds] = Normalize(data, 'meancols', 'stdcols');
  
  % Initialization
  [dataInitFinal, maskFinal] = InitMissing(dataNorm, initialization);
  
  % Filling
  dataFilled = EOFCore(dataInitFinal, maskFinal, bb, stopping, rounds);
  
  % DeNormalizing the filled dataset
  dataFilled = ...
    DeNormalize(dataFilled, 'std', normStds, 'mean', normMeans);
  
  dataFilledFinal = data;
  dataFilledFinal(isnan(data)) = dataFilled(isnan(data));
end

% End

