// -*- C++ -*-  $Id: ObjectDetection.C,v 1.163 2014/01/16 15:10:59 jorma Exp $
//
// Copyright 2009-2014 PicSOM Development Group <picsom@cis.hut.fi>
// Aalto University School of Science and Technology
// Department of Information and Computer Science
// P.O.BOX 15400, FI-00076 Aalto, FINLAND
//

#include "ObjectDetection.h"

#include "unistd.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846
#define M_PI_2 1.57079632679489661923
#define M_PI_4 0.785398163397448309616
#endif // M_PI

#ifdef USE_BOOST_ALGORITHM_STRING
#include <boost/algorithm/string.hpp>
using namespace boost::algorithm;
#endif

double ObjectDetection::SURF_HESSIAN_THRESHOLD = 500.0;
bool ObjectDetection::SURF_USE_EXTENDED = 0;
double ObjectDetection::DEFAULT_SCALING_FACTOR = 0.67;
size_t ObjectDetection::MATCHING_NSURVIVORS_TH = 8;
size_t ObjectDetection::MATCHING_NMATCHES_TH = 20;
size_t ObjectDetection::FLANN_NTREES = 4;
double ObjectDetection::FLANN_SIMILARITY_RATIO_JOINT = 0.4;
double ObjectDetection::FLANN_SIMILARITY_RATIO_SEPARATE = 0.6;
size_t ObjectDetection::FLANN_MULTIPLE_MATCH_TH = 999999;
size_t ObjectDetection::FLANN_SEARCH_CHECKS = 64;
size_t ObjectDetection::JOINT_2NDSTAGE_NIMAGES = 5;
size_t ObjectDetection::HOMOGRAPHY_NIMAGES = 5;
double ObjectDetection::HOMOGRAPHY_DETERMINANT_TH = 0.01;
bool ObjectDetection::HOMOGRAPHY_CHECK_ANGLES = 1;
double ObjectDetection::HOMOGRAPHY_MINLENGTH_TH = 20.0;
double ObjectDetection::HOMOGRAPHY_MAXLENGTH_TH = 10000.0;
double ObjectDetection::HOMOGRAPHY_BACKPROJECTION_TH = 10.0;

#ifndef __WINDOWS__
pthread_mutex_t ObjectDetection::pi_mutex = PTHREAD_MUTEX_INITIALIZER;
#endif // __WINDOWS__
size_t ObjectDetection::n_soap_threads = 0;

ObjectDetection* ObjectDetection::op;

string ObjectDetection_C_vcid =
  "@(#)$Id: ObjectDetection.C,v 1.163 2014/01/16 15:10:59 jorma Exp $";

// ---------------------------------------------------------------------

static CvScalar colors[] = {
  {{0,0,255}},
  {{0,128,255}},
  {{0,255,255}},
  {{0,255,0}},
  {{255,128,0}},
  {{255,255,0}},
  {{255,0,0}},
  {{255,0,255}},
  {{255,255,255}},
  {{0,0,0}},
  {{128,128,128}}
};

// ---------------------------------------------------------------------

const string& ObjectDetection::Version() {
  static string v;
  if (v=="") {
    v = ObjectDetection_C_vcid;
    size_t p = v.find(".C,v ");
    if (p!=string::npos)
      v.erase(0, p+5);
    p = v.rfind(" Exp $");
    if (p!=string::npos)
      v.erase(p);
  }
  return v;
}

// ---------------------------------------------------------------------

ObjectDetection::ObjectDetection() {
  storage = NULL;
  //imageKeypoints = NULL;
  //imageDescriptors = NULL;
}

// ---------------------------------------------------------------------

bool ObjectDetection::Initialize(bool quiet) {
  bool ok = true;

  rcfilename = ".odrc";
  if (FileExists(rcfilename)) {
#ifdef USE_BOOST_ALGORITHM_STRING
    if (!quiet)
      cout << "Reading parameter values from " << rcfilename << endl;
    ok = LoadRcFile();
#else
    cerr << "WARNING: Parameter file " << rcfilename 
	 << " exists but cannot be read as this requires boost" << endl;
#endif
  }

  debug = 0;
  start_time = time(NULL);

  nqueries = 0;
  nsoaps = 0;

  used_detector = DET_SURF;
  used_descriptor.name = DES_SURF;
  used_descriptor.root = false;
  used_descriptor.norm = false;

  usebinaryfiles   = true;
  analyseonlybest  = false;
  exhaustivesearch = false;
  usemask          = MASK_NONE;
  useborders       = false;
  showresults      = true;
  showonlineresults = false;
  showonlinedbresults = false;
  displayinfo      = true;
  showkeypoints    = true;
  updateapriori    = false;
  preserveorder    = true;
  secondstage      = true;

  flann_index_mode = KDTREE;

  scale = -1.0;
  newsize = -1;
  applyblur = false;

  outimgname = "output.png";
  outvidname = "";
  outdbname  = "";

  matchingkeypointsmode = NONE;
  matchingkeypointsname = "";

  writematches   = false;
  writematchesfn = "matches.txt";

  filter = NULL;

  storage = cvCreateMemStorage(0);

  //surfparams = cvSURFParams(SURF_HESSIAN_THRESHOLD, SURF_USE_EXTENDED);
  //mserparams = cvMSERParams();

  //dbindex.descriptors = cvCreateSeq(0, sizeof(CvSeq),
  //				    64*CV_ELEM_SIZE(CV_32F), storage);
  // dbindex.keypoints   = cvCreateSeq(0, sizeof(CvSeq),
  // 				    sizeof(CvSURFPoint), storage);

  DbIndicesInUse(false);

  cvInitFont(&statusfont, CV_FONT_HERSHEY_PLAIN, 1, 1);

  op = this;

  bool use_boost_algorithm_string = false;
#ifdef USE_BOOST_ALGORITHM_STRING
  use_boost_algorithm_string = true;
#endif

  bool use_libsvm = false;
#ifdef USE_LIBSVM
  use_libsvm = true;
#endif

  dbresultimg.img = NULL;
  dbresultimg.scale = -1.0;
  dbresultimg.width = 0;
  dbresultimg.height = 128; // height is used to calculate scale and width

  dbitem prev;
  previmgs.push_back(prev);

  frame_no = 0;

  ncuda = cv::gpu::getCudaEnabledDeviceCount();

  det_ext_time.first=det_ext_time.second=0.0;

  if (!quiet)
    cout << "Initialization ready on " << ctime(&start_time)
	 << "Using OpenCV version: "<< CV_VERSION
	 << ", Boost/algorithm/string: " << use_boost_algorithm_string
	 << ", LIBSVM: " << use_libsvm << endl
	 << "System version: "<< ObjectDetection_C_vcid << endl
	 << "Version(): " << Version() << endl
	 << "CudaEnabledDeviceCount: " << ncuda << endl;   

  return ok;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadRcFile() {
  bool dbg = false;
  if (dbg)
    cout << "LoadRcFile(): Starting" << endl;

#ifdef USE_BOOST_ALGORITHM_STRING

  ifstream rcfile(rcfilename.c_str());
  if (!rcfile) {
    if (dbg)
      cout << "LoadRcFile(): File not found, exiting" << endl;
    return true;
  }
  string line;
  while (getline(rcfile, line)) {
    if (dbg)
      cout << "LoadRcFile(): Processing line: \"" << line << "\"" << endl;
    trim(line);
    if (dbg)
      cout << "LoadRcFile(): After trimming: \"" << line << "\"" << endl;

    if (line.find("#") != string::npos) {
      if (dbg)
	cout << "LoadRcFile(): Character '#' found, this line is a comment"
	     << endl;
      continue;
    }
    vector<string> lineparts;

    if (line.find("=") != string::npos) {
      vector<string> lineparts;
      split(lineparts, line, is_any_of("="));
      if (dbg)
	cout << "LoadRcFile(): Found key=" << lineparts[0]
	     << ", val=" << lineparts[1] << endl;
      Interpret(lineparts[0], lineparts[1]);
    }
    else {
      if (dbg)
	cout << "LoadRcFile(): Could not interpret line."
	     << endl;
    }
  }

#else

  if (dbg)
    cout << "LoadRcFile(): This requires boost" << endl;

#endif

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::Interpret(const string& keystr,
				const string& valstr) {

  if (keystr == "")
    return false;
  if (valstr == "")
    return false;

  size_t sizeval  = atoi(valstr.c_str());
  double floatval = atof(valstr.c_str());
  bool boolval    = IsAffirmative(valstr.c_str());

  if (keystr == "SURF_HESSIAN_THRESHOLD") {
    cout << "Interpret(): Changed value of SURF_HESSIAN_THRESHOLD from "
	 << SURF_HESSIAN_THRESHOLD << " to ";
    SURF_HESSIAN_THRESHOLD = floatval;
    cout << SURF_HESSIAN_THRESHOLD << endl;
    return true;
  }

  if (keystr == "SURF_USE_EXTENDED") {
    cout << "Interpret(): Changed value of SURF_USE_EXTENDED from "
	 << SURF_USE_EXTENDED << " to ";
    SURF_USE_EXTENDED = boolval;
    cout << SURF_USE_EXTENDED << endl;
    return true;
  }

  if (keystr == "DEFAULT_SCALING_FACTOR") {
    cout << "Interpret(): Changed value of DEFAULT_SCALING_FACTOR from "
	 << DEFAULT_SCALING_FACTOR << " to ";
    DEFAULT_SCALING_FACTOR = floatval;
    cout << DEFAULT_SCALING_FACTOR << endl;
    return true;
  }

  if (keystr == "MATCHING_NSURVIVORS_TH") {
    cout << "Interpret(): Changed value of MATCHING_NSURVIVORS_TH from "
	 << MATCHING_NSURVIVORS_TH << " to ";
    MATCHING_NSURVIVORS_TH = sizeval;
    cout << MATCHING_NSURVIVORS_TH << endl;
    return true;
  }

  if (keystr == "MATCHING_NMATCHES_TH") {
    cout << "Interpret(): Changed value of MATCHING_NMATCHES_TH from "
	 << MATCHING_NMATCHES_TH << " to ";
    MATCHING_NMATCHES_TH = sizeval;
    cout << MATCHING_NMATCHES_TH << endl;
    return true;
  }

  if (keystr == "FLANN_NTREES") {
    cout << "Interpret(): Changed value of FLANN_NTREES from "
	 << FLANN_NTREES << " to ";
    FLANN_NTREES = sizeval;
    cout << FLANN_NTREES << endl;
    return true;
  }

  if (keystr == "FLANN_SIMILARITY_RATIO_JOINT") {
    cout << "Interpret(): Changed value of FLANN_SIMILARITY_RATIO_JOINT from "
	 << FLANN_SIMILARITY_RATIO_JOINT << " to ";
    FLANN_SIMILARITY_RATIO_JOINT = floatval;
    cout << FLANN_SIMILARITY_RATIO_JOINT << endl;
    return true;
  }

  if (keystr == "FLANN_SIMILARITY_RATIO_SEPARATE") {
    cout << "Interpret(): Changed value of FLANN_SIMILARITY_RATIO_SEPARATE from "
	 << FLANN_SIMILARITY_RATIO_SEPARATE << " to ";
    FLANN_SIMILARITY_RATIO_SEPARATE = floatval;
    cout << FLANN_SIMILARITY_RATIO_SEPARATE << endl;
    return true;
  }

  if (keystr == "FLANN_MULTIPLE_MATCH_TH") {
    cout << "Interpret(): Changed value of FLANN_MULTIPLE_MATCH_TH from "
	 << FLANN_MULTIPLE_MATCH_TH << " to ";
    FLANN_MULTIPLE_MATCH_TH = sizeval;
    cout << FLANN_MULTIPLE_MATCH_TH << endl;
    return true;
  }

  if (keystr == "FLANN_SEARCH_CHECKS") {
    cout << "Interpret(): Changed value of FLANN_SEARCH_CHECKS from "
	 << FLANN_SEARCH_CHECKS << " to ";
    FLANN_SEARCH_CHECKS = sizeval;
    cout << FLANN_SEARCH_CHECKS << endl;
    return true;
  }

  if (keystr == "JOINT_2NDSTAGE_NIMAGES") {
    cout << "Interpret(): Changed value of JOINT_2NDSTAGE_NIMAGES from "
	 << JOINT_2NDSTAGE_NIMAGES << " to ";
    JOINT_2NDSTAGE_NIMAGES = sizeval;
    cout << JOINT_2NDSTAGE_NIMAGES << endl;
    return true;
  }

  if (keystr == "HOMOGRAPHY_NIMAGES") {
    cout << "Interpret(): Changed value of HOMOGRAPHY_NIMAGES from "
	 << HOMOGRAPHY_NIMAGES << " to ";
    HOMOGRAPHY_NIMAGES = sizeval;
    cout << HOMOGRAPHY_NIMAGES << endl;
    return true;
  }

  if (keystr == "HOMOGRAPHY_DETERMINANT_TH") {
    cout << "Interpret(): Changed value of HOMOGRAPHY_DETERMINANT_TH from "
	 << HOMOGRAPHY_DETERMINANT_TH << " to ";
    HOMOGRAPHY_DETERMINANT_TH = floatval;
    cout << HOMOGRAPHY_DETERMINANT_TH << endl;
    return true;
  }

  if (keystr == "HOMOGRAPHY_CHECK_ANGLES") {
    cout << "Interpret(): Changed value of HOMOGRAPHY_CHECK_ANGLES from "
	 << HOMOGRAPHY_CHECK_ANGLES << " to ";
    HOMOGRAPHY_CHECK_ANGLES = boolval;
    cout << HOMOGRAPHY_CHECK_ANGLES << endl;
    return true;
  }

  if (keystr == "HOMOGRAPHY_MINLENGTH_TH") {
    cout << "Interpret(): Changed value of HOMOGRAPHY_MINLENGTH_TH from "
	 << HOMOGRAPHY_MINLENGTH_TH << " to ";
    HOMOGRAPHY_MINLENGTH_TH = floatval;
    cout << HOMOGRAPHY_MINLENGTH_TH << endl;
    return true;
  }

  if (keystr == "HOMOGRAPHY_MAXLENGTH_TH") {
    cout << "Interpret(): Changed value of HOMOGRAPHY_MAXLENGTH_TH from "
	 << HOMOGRAPHY_MAXLENGTH_TH << " to ";
    HOMOGRAPHY_MAXLENGTH_TH = floatval;
    cout << HOMOGRAPHY_MAXLENGTH_TH << endl;
    return true;
  }

  if (keystr == "HOMOGRAPHY_BACKPROJECTION_TH") {
    cout << "Interpret(): Changed value of HOMOGRAPHY_BACKPROJECTION_TH from "
	 << HOMOGRAPHY_BACKPROJECTION_TH << " to ";
    HOMOGRAPHY_BACKPROJECTION_TH = floatval;
    cout << HOMOGRAPHY_BACKPROJECTION_TH << endl;
    return true;
  }

  return SetOption(keystr, valstr);
}

// ---------------------------------------------------------------------

bool ObjectDetection::Interpret(const char *keqv) {
  
  string keyeqval(keqv);
  size_t eqsignpos = keyeqval.find("=");
  if (eqsignpos == string::npos) {
    cerr << "Interpret(): ERROR: No equal sign found in \"" 
	 << keyeqval << "\"" << endl;
    return false;
  }
  bool ok = Interpret(keyeqval.substr(0,eqsignpos), 
		      keyeqval.substr(eqsignpos+1));
  if (!ok)
    cerr << "Interpret(): ERROR: Unrecognized key=val pair \"" 
	 << keyeqval << "\"" << endl;  
  return ok;
}

// ---------------------------------------------------------------------

bool ObjectDetection::SetOption(const string& keystr,
				const string& valstr) {

  if (keystr == "")
    return false;
  if (valstr == "")
    return false;

  size_t sizeval  = atoi(valstr.c_str());
  double floatval = atof(valstr.c_str());
  bool boolval    = IsAffirmative(valstr.c_str());

  if (keystr == "debug")
    SetDebug(sizeval);

  if (debug>1) // note that debug might not yet be set here
    cout << "SetOption(): keystr=" << keystr << " valstr=" << valstr
	 << " sizeval=" << sizeval << " floatval=" << floatval 
	 << " boolval=" << boolval << endl; 

  if (keystr == "debug")
    return true;

  if (keystr == "newsize") {
    SetNewsize(floatval);
    return true;
  }
  if (keystr == "blur") {
    SetBlur(boolval);
    return true;
  }
  if (keystr == "detector") {
    SetUsedDetector(valstr);
    return true;
  }
  if (keystr == "descriptor") {
    SetUsedDescriptor(valstr);
    return true;
  }

  return false;
}

// ---------------------------------------------------------------------

bool ObjectDetection::SetOption(const char *keqv) {
  
  string keyeqval(keqv);
  size_t eqsignpos = keyeqval.find("=");
  if (eqsignpos == string::npos) {
    cerr << "SetOption(): ERROR: No equal sign found in \"" 
	 << keyeqval << "\"" << endl;
    return false;
  }
  bool ok = SetOption(keyeqval.substr(0,eqsignpos), 
		      keyeqval.substr(eqsignpos+1));
  if (!ok)
    cerr << "SetOption(): ERROR: Unrecognized key=val pair \"" 
	 << keyeqval << "\"" << endl;  
  return ok;
}

// ---------------------------------------------------------------------

void ObjectDetection::SetScale(const string &sstr) {
  size_t comma = sstr.find(",");

  if (comma == string::npos) {
    double s = atof(sstr.c_str());
    SetScale(s);
    scales.push_back(s);

  } else {
    double first = atof(sstr.substr(0,comma).c_str());
    SetScale(first);
    scales.push_back(first);

    string tmp = sstr.substr(comma+1);
    while ((comma = tmp.find(",")) != string::npos) {
      scales.push_back(atof(tmp.substr(0,comma).c_str()));
      if (tmp.size()>comma+1)
	tmp = tmp.substr(comma+1);
      else
	tmp = "";
    }
    if (tmp != "")
      scales.push_back(atof(tmp.c_str()));
  }

  if (debug>1) {
    cout << "SetScale(" << sstr << "): scale=" << scale
	 << ", scales=[ ";
    vector<double>::const_iterator it;
    for (it=scales.begin(); it!=scales.end(); ++it)
      cout << *it << " ";
    cout << "]" << endl;    
  }
}

// ---------------------------------------------------------------------

void ObjectDetection::SetUseXMask(const string &o) {
  size_t comma = o.find(",");
  if (comma == string::npos) {
    cerr << "SetUseXMask(): ERROR: No comma found in \"" 
	 << o << "\"" << endl;
    SetUseMask(MASK_NONE);
    return;
  }
  
  maskreplace.first  = o.substr(0,comma);
  maskreplace.second = o.substr(comma+1);
}

// ---------------------------------------------------------------------

void ObjectDetection::SetUseBorders(const string &o) {
  size_t comma = o.find(",");
  if (comma == string::npos) {
    cerr << "SetUseBorders(): ERROR: No comma found in \"" 
	 << o << "\"" << endl;    
    return;
  }
  
  useborders = true;
  borderreplace.first  = o.substr(0,comma);
  borderreplace.second = o.substr(comma+1);
}

// ---------------------------------------------------------------------

void ObjectDetection::SetUsedDetector(const string &d) {
  if (d == "none" || d == "NONE") 
    used_detector = DET_NONE;
  else if (d == "surf" || d == "SURF") 
    used_detector = DET_SURF;
  else if (d == "mser" || d == "MSER")
    used_detector = DET_MSER;
  else if (d == "sift" || d == "SIFT")
    used_detector = DET_SIFT;
  else if (d == "fast" || d == "FAST")
    used_detector = DET_FAST;
  else if (d == "star" || d == "STAR")
    used_detector = DET_STAR;
  else if (d == "orb" || d == "ORB")
    used_detector = DET_ORB;
  else if (d == "gtff" || d == "GTFF")
    used_detector = DET_GTFF;
  else if (d == "harris" || d == "HARRIS")
    used_detector = DET_HARRIS;

  else if (d == "gpusurf" || d == "GPUSURF") {
    if (ncuda > 0)
      used_detector = DET_GPUSURF;
    else {
      used_detector = DET_SURF;
      cout << "SetUsedDetector(): No CUDA devices found, using SURF instead" 
	   << endl;
    }
  }

  else 
    cerr << "SetUsedDetector(): ERROR: Unknown detector: " 
	 << d << endl;
}
  
// ---------------------------------------------------------------------

void ObjectDetection::SetUsedDescriptor(const string &d) {
  if (d.find("gpusurf") != string::npos) {
    if (ncuda > 0)
      used_descriptor.name = DES_GPUSURF;
    else {
      used_descriptor.name = DES_SURF;
      cout << "SetUsedDescriptor(): No CUDA devices found, using SURF instead" 
	   << endl;
    }
  } else if (d.find("surf") != string::npos) 
    used_descriptor.name = DES_SURF;
  else if (d.find("sift") != string::npos)
    used_descriptor.name = DES_SIFT;
  else 
    cerr << "SetUsedDescriptor(): ERROR: Unknown descriptor: " 
	 << d << endl;
  if (d.find("root") != string::npos)
    used_descriptor.root = true;
  if (d.find("norm") != string::npos)
    used_descriptor.norm = true;  
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessImage(const string& fn, savetype& savemode) {
  bool ok = true;

  if (debug ) {
    cout << "ProcessImage(" << fn << ", " << SavemodeStr(savemode)
	 << ")" << endl;
  }

  IplImage *image = NULL;
  ok = LoadImage(&image, fn);
  if (!image) {
    cerr << "ProcessImage(): ERROR: Image pointer is NULL" << endl;
    return false;
  }

  IplImage *mask = NULL;
  if (usemask != MASK_NONE) {
    size_t fromloc = fn.find(maskreplace.first);
    if (fromloc == string::npos) {
      cerr << "ProcessImage(): ERROR: Substring \""<< maskreplace.first << 
	"\" not found in " << fn << endl;
      return false;
    }
    string maskfn = fn;
    maskfn.replace(fromloc, maskreplace.first.size(), maskreplace.second);
    
    if (usemask == MASK_FILE)
      ok = LoadImage(&mask, maskfn);
    else
      ok = CreatePolygonMask(&image, &mask, maskfn);
    
    if (!mask) {
      cerr << "ProcessImage(): ERROR: Mask pointer is NULL" << endl;
      return false;
    }
    if (mask->width != image->width || mask->height != image->height) {
      cerr << "ProcessImage(): ERROR: Mask and image sizes don't match" 
	   << endl;
      return false;
    }
  }

  if (ok)
    ok = ProcessImageCommonPartOne(&image, &mask);

  if (ok && savemode != NO_SAVING) {

    if (FilterInUse() && !FilterLoaded())
      filter->LoadFilter(UsedDescriptorSize());
    if (FilterLoaded())
      filter->FilterDescriptors(imageKeypoints, imageDescriptors);

    if (ok && savemode == MATLAB_MODE) {
      ok = SaveData_Text(fn + (FilterLoaded() ? "."+FilterString():"")+".mat",
			 imageKeypoints, imageDescriptors,
			 FilterLoaded(), false);

    } else if (ok) {

      if (usebinaryfiles) {
	ok = SaveData_Bin(fn + (FilterLoaded()?"."+FilterString():"") +".bin",
			  imageKeypoints, imageDescriptors, FilterLoaded());
      } else {
	ok = SaveData_Text(fn +(FilterLoaded()?"."+FilterString():"")+".surf",
			   imageKeypoints, imageDescriptors, FilterLoaded());
      }

      if (savemode == SEPARATE_INDICES)
	CreateAndSaveIndex(imageKeypoints, imageDescriptors,
			   fn + (FilterLoaded() ? "."+FilterString() : "")
			   + ".idx", "", FilterLoaded());

      else if (savemode == JOINT_INDEX) {
	AppendDescriptors();

      } else {
	cerr << "ERROR: ProcessImage(): Unsupported savemode:"
	     << SavemodeStr(savemode) << endl;
	ok = false;
      }
    }

    if (ok)
      PossiblySaveDbItem(image, fn);
  }

  if (FilterLoaded())
    filter->keypoint_ok.clear();

  PossiblySetShowKeypoints(savemode==NO_SAVING || savemode==MATLAB_MODE);
  return ok ? ProcessImageCommonPartTwo(&image, &mask, fn) : false;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessImage(const string &fn, savetype &savemode, 
				   dbitem &res) {
  ProcessFail(res);
  if (ProcessImage(fn,savemode)) {
    if (db.size()) {
      res = db[0];

      stringstream ssinvcorners;
      for (int i = 1; i < 5; i++) {
	if (i>1)
	  ssinvcorners << ";";
	CvPoint &p = res.dbitemcorners[i];
	ssinvcorners << (int)(p.x/GetScale()) << "," << (int)(p.y/GetScale());
      }
      res.invcorners = ssinvcorners.str();
    }

    return true;
  }
  return false;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessImage(IplImage *image, const string& imgfn) {
  if (debug)
    cout << "ProcessImage(image) \"" << imgfn << "\"" << endl;
  
  bool ok = true;
  
  if (image->nChannels > 1) {
    cerr << "ProcessImage(image): ERROR: Image nChannels > 1" 
	 << endl;
    ok = false;
  }
  
  if (ok)
    ok = ProcessImageCommonPartOne(&image, NULL);
  
  if (filter != NULL) {

    cout << endl 
	 << "ProcessImage(image): QUERY IMAGE FILTERING TEMPORARILY DISABLED !!!" 
	 << endl << endl;

    if (!FilterLoaded())
      filter->LoadFilter(UsedDescriptorSize());
    if (FilterLoaded())
      filter->FilterDescriptors(imageKeypoints, imageDescriptors);

    //int tot_origin = imageKeypoints->total;
    size_t tot_origin = imageKeypoints.size();
    int tot = 0;
    vector<bool>::const_iterator it;
    for (it=filter->keypoint_ok.begin(); it!=filter->keypoint_ok.end(); ++it) {
      if (*it)
	tot++;
    }


    // //CvMemStorage *kpmemtmp= cvCreateMemStorage(imageKeypoints->header_size);
    // //CvMemStorage *kpmemtmp= cvCreateMemStorage();
    // //CvSeq *kpseqtmp= cvCloneSeq(imageKeypoints, kpmemtmp);
    // /CvSeq *kpseqtmp= cvCloneSeq(imageKeypoints);
    // //CvMemStorage *desmemtmp= cvCreateMemStorage(imageDescriptors->header_size);
    // //CvMemStorage *desmemtmp= cvCreateMemStorage();
    // //CvSeq *desseqtmp= cvCloneSeq(imageDescriptors, desmemtmp);
    //CvSeq *desseqtmp= cvCloneSeq(imageDescriptors);
    // cvClearSeq(kpseqtmp);
    //cvClearSeq(desseqtmp);
    
    for (size_t i = 0; i < tot_origin; i++) {
      if (filter->keypoint_ok[i]) {
	imageKeypoints.erase(imageKeypoints.begin()+i);
	//imageDescriptors.erase(imageDescriptors.begin()+i);
      }
    }
    
    if (debug)
      cout << "After applying filter to the query image, keep " << tot << "/" << tot_origin
	   << " keypoints and descriptors " << endl;
    
    ////delete imageKeypoints;
    //cvClearMemStorage(imageKeypoints->storage);
    //imageKeypoints = kpseqtmp;
    ////delete imageDescriptors;
    //cvClearMemStorage(imageDescriptors->storage);
    //imageDescriptors = desseqtmp;
    
  }

  return ok ? ProcessImageCommonPartTwo(&image, NULL, imgfn) : false;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessImageCommonPartOne(IplImage** imgptr, 
						IplImage** maskptr) {
  if (debug>1)
    cout << "ProcessImageCommonPartOne(): Starting" << endl;

  bool ok = true;
  IplImage*& image = *imgptr;
  IplImage*& mask  = *maskptr;

  if (ok && newsize != -1) {
    ok = ResizeImage(&image);
    if (&mask && mask)
      ok = ResizeImage(&mask);
  }

  if (ok && scale > 0.0) {
    ok = ScaleImage(&image);
    if (&mask && mask)
      ok = ScaleImage(&mask);
  } else if (ok && db.size()) {
    ok = ScaleImage(&image, DEFAULT_SCALING_FACTOR);
    if (&mask && mask)
      ok = ScaleImage(&mask, DEFAULT_SCALING_FACTOR);
  }

  if (ok && applyblur)
    cvSmooth(image, image, CV_GAUSSIAN, 5);

  if (ok)
    ok = Detect(image, &mask);

  if (ok)
    ok = Extract(image);
    
  frame_no++;

  return ok;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessImageCommonPartTwo(IplImage** imgptr, 
						IplImage** maskptr, 
						const string& imgfn) {
  if (debug>1)
    cout << "ProcessImageCommonPartTwo(): Starting" << endl;

  bool ok = true;
  IplImage*& image = *imgptr;
  IplImage*& mask  = *maskptr;

  if (!image) {
    cerr << "ProcessImageCommonPartTwo(): ERROR: Image pointer is NULL" 
	 << endl;
    return false;
  }

  if (ok && showkeypoints && showresults && 
      !showonlineresults && !showonlinedbresults)
    showresults = ShowKeypoints(image, &mask);

  if (db.size()==0)
    return true;

  for (size_t ii=0; ii<db.size(); ii++) {
    db[ii].homography  = false;
    db[ii].nsurvivors  = 0;
    db[ii].nmatches    = 0;
    db[ii].ptpairs.clear();
    db[ii].survivors.clear();
    db[ii].gaze = cvPoint2D32f(-1.0, -1.0);
  }

  if (DbIndicesInUse() || exhaustivesearch)
    ok = ProcessWholeDatabase(image);
  else
    ok = ProcessDatabaseUntilGoodMatch(image);

  if (0)
    ok = ProcessPrevious(image);

  DoGazeMatch();

  if (showonlineresults)
    ShowBestResult(image, false); 
  else if (showonlinedbresults)
    ShowDbResults(image); 
  else if (showresults)
    ShowAllMatchingResults(image);

  //cout << writematches << " " << writematchesfn << endl;
  if (writematches) {
    ofstream outfile(writematchesfn.c_str(), ios_base::app);
    if (!outfile) {
      cerr << "ERROR: Cannot open output stream for " << writematchesfn << endl;
      ok = false;
    } else {      
      for (size_t ii=0; ii<db.size(); ii++)
	if (db[ii].nsurvivors >= MATCHING_NSURVIVORS_TH) {
	  outfile << "MATCH " << imgfn << " " << ii << " " << db[ii].imagename
		  << " " << db[ii].nsurvivors << "/" << db[ii].nmatches
		  << endl;	
	  if (db[ii].gaze.x>=0.0)
	    outfile << "GAZE " << imgfn << " " << ii << " " << db[ii].imagename
		    << " " << db[ii].gaze.x << " " << db[ii].gaze.y
		    << endl;
	}
      outfile.close();
    }
  }

  if (image)
    cvReleaseImage(&image);

  for (size_t ii=0; ii<db.size(); ii++) {
    db[ii].ptpairs.clear();
    db[ii].survivors.clear();
  }

  if (0) {
    previmgs.at(0).keypoints   = imageKeypoints;
    previmgs.at(0).descriptors = imageDescriptors;
  }

  //cvClearSeq(imageKeypoints);
  //cvClearSeq(imageDescriptors);
  imageKeypoints.clear();
  imageDescriptors.resize(0);

  if (ok)
    nqueries++;

  return ok;
}

// ---------------------------------------------------------------------

void ObjectDetection::DisplayDetExtTime() {
  if (debug)
    printf("Total time: detection=%gs extraction=%gs combined=%gs\n", 
	   det_ext_time.first/(cvGetTickFrequency()*1000.*1000.),
	   det_ext_time.second/(cvGetTickFrequency()*1000.*1000.),
	   (det_ext_time.first+det_ext_time.second)/(cvGetTickFrequency()*1000.*1000.));
}
// ---------------------------------------------------------------------

void ObjectDetection::AppendDescriptors() {

  if (dbindex.descriptors.cols == 0)
    dbindex.descriptors.create(0, UsedDescriptorSize(), CV_32F);

  if (debug>1)
    cout << "AppendDescriptors(): Appending from kps=" 
	 << dbindex.keypoints.size() << ", descs=" 
	 << dbindex.descriptors.rows << endl;

  CheckKeypointsAndDescriptors();

  for (int i = 0; i < imageDescriptors.rows; i++) {
    cv::Mat descriptor = imageDescriptors.row(i);
    cv::KeyPoint &keypoint = imageKeypoints.at(i);

    if (!FilterLoaded() || filter->keypoint_ok[i]) {
      dbindex.descriptors.push_back(descriptor);
      dbindex.keypoints.push_back(keypoint);
    }
  }

  if (debug>1)
    cout << "AppendDescriptors(): Appending to kps=" 
	 << dbindex.keypoints.size() << ", descs=" 
	 << dbindex.descriptors.rows << endl;
}

// ---------------------------------------------------------------------

void ObjectDetection::CheckKeypointsAndDescriptors(vector<cv::KeyPoint> &keyps,
						   cv::Mat &descs) {

  //if (keyps == NULL)
  if (keyps.empty())
    keyps = imageKeypoints;
  if (descs.empty())
    descs = imageDescriptors;

  if (descs.rows != (int)keyps.size()) {
    cerr << "ERROR: CheckKeypointsAndDescriptors(): Mismatch with "
	 << "descriptors and keypoints: " << descs.rows
	 << " != " << keyps.size() << endl;
    exit(-1);
  } else if (FilterLoaded()) {
    if (!filter->keypoint_ok.empty() &&
	(int)filter->keypoint_ok.size() != descs.rows) {
      cerr << "ERROR: CheckKeypointsAndDescriptors(): Mismatch with "
	   << "descriptors and filtering: " << descs.rows
	   << " != " << filter->keypoint_ok.size() << endl;
      exit(-1);
    }
  }
}

// ---------------------------------------------------------------------

void ObjectDetection::CheckKeypointsAndDescriptors() {
  CheckKeypointsAndDescriptors(imageKeypoints, imageDescriptors);
}

// ---------------------------------------------------------------------

bool ObjectDetection::ShowNthResult(const IplImage *image, size_t ii, 
				    bool openwindow) {
  //cout << "Starting ShowNthResult(ii) with ii=" << ii << endl;

  const size_t thickness = 0; // extra thickness to lines
  const size_t interval = 0;  // add space between images
  const bool white_background = false; // false means black
  const bool borders_over_query_image = true;
  const bool borders_over_db_image = true;

  const int horizshift = image->width + interval;

  if (!FileExists(db[ii].pathname)) {
    cerr << "ERROR: No image found: " << db[ii].pathname << endl;
    return false;
  }
  IplImage *dbimage = cvLoadImage(db[ii].pathname.c_str(),
				  CV_LOAD_IMAGE_GRAYSCALE);
  IplImage *correspond =
    cvCreateImage(cvSize(horizshift+dbimage->width,
			 MAX(image->height,dbimage->height)), 8, 3);
  cvSet(correspond, white_background?cvScalar(255,255,255):cvScalar(0,0,0));
  cvSetImageROI(correspond, cvRect(0, 0, image->width, image->height));
  IplImage *img_color = cvCreateImage(cvGetSize(image), 8, 3);
  cvCvtColor(image, img_color, CV_GRAY2BGR);
  cvCopy(img_color, correspond);
  cvReleaseImage(&img_color);

  cvSetImageROI(correspond, cvRect(horizshift, 0, correspond->width,
				   dbimage->height));
  IplImage *dbimg_color = cvCreateImage(cvGetSize(dbimage), 8, 3);
  cvCvtColor(dbimage, dbimg_color, CV_GRAY2BGR);
  cvCopy(dbimg_color, correspond);
  cvReleaseImage(&dbimg_color);
  cvReleaseImage(&dbimage);
  cvResetImageROI(correspond);

  vector<int> &ptpairs = db[ii].ptpairs;
  vector<bool> &survivors = db[ii].survivors;
  bool &hg = db[ii].homography;

  if (hg) {

    if (borders_over_db_image) {
      CvPoint r = db[ii].querycorners[0];
      if (r.x>-1 && r.y>-1) {
	cvCircle(correspond, cvPoint(r.x+horizshift, r.y), 5+thickness*2,
		 colors[8], -1, 8, 1);
	cvCircle(correspond, cvPoint(r.x+horizshift, r.y), 3+thickness*2,
		 colors[0], -1, 8, 1);
      }
      if (useborders) 
       	for (int i = 1; i < 5; i++) {
	  CvPoint r1 = db[ii].dbitemborders[i];
	  CvPoint r2 = db[ii].dbitemborders[MAX((i+1)%5,1)];
	  cvLine(correspond, cvPoint(r1.x+horizshift, r1.y),
		 cvPoint(r2.x+horizshift, r2.y), colors[8], 3+thickness*2);
	  cvLine(correspond, cvPoint(r1.x+horizshift, r1.y),
		 cvPoint(r2.x+horizshift, r2.y), colors[0], 1+thickness);
	}
      else
	for (int i = 1; i < 5; i++) {
	  CvPoint r1 = db[ii].querycorners[i];
	  CvPoint r2 = db[ii].querycorners[MAX((i+1)%5,1)];
	  cvLine(correspond, cvPoint(r1.x+horizshift, r1.y),
		 cvPoint(r2.x+horizshift, r2.y), colors[8], 3+thickness*2);
	  cvLine(correspond, cvPoint(r1.x+horizshift, r1.y),
		 cvPoint(r2.x+horizshift, r2.y), colors[0], 1+thickness);
	}
    }

    if (borders_over_query_image) {
      CvPoint r = db[ii].dbitemcorners[0];
      if (r.x>-1 && r.y>-1) {
	cvCircle(correspond, cvPoint(r.x, r.y), 5+thickness*2,
		 colors[8], -1, 8, 1);
	cvCircle(correspond, cvPoint(r.x, r.y), 3+thickness*2,
		 colors[0], -1, 8, 1);
      }
      for (int i = 1; i < 5; i++) {
	CvPoint r1 = db[ii].dbitemcorners[i];
	CvPoint r2 = db[ii].dbitemcorners[MAX((i+1)%5,1)];
	cvLine(correspond, cvPoint(r1.x, r1.y),
	       cvPoint(r2.x, r2.y), colors[8], 3+thickness*2);
	cvLine(correspond, cvPoint(r1.x, r1.y),
	       cvPoint(r2.x, r2.y), colors[0], 1+thickness);
      }

      if (0)
	for (size_t i=0; i<100; i++) {
	  CvPoint rndp = cvPoint(rand()%(image->width), rand()%(image->height));
	  if (InsidePolygon(rndp, ii))
	    cvCircle(correspond, rndp, 3+thickness*2, colors[0], -1, 8, 1);
	  else
	    cvCircle(correspond, rndp, 3+thickness*2, colors[3], -1, 8, 1);
	}

    }

  } // if (hg)

  for (int i = 0; i < (int)ptpairs.size()/2; i ++) {
    // CvSURFPoint* r1 =
    //   (CvSURFPoint*)cvGetSeqElem(imageKeypoints, ptpairs[i*2]);
    // CvSURFPoint* r2 =
    //   (CvSURFPoint*)cvGetSeqElem(db[ii].keypoints, ptpairs[i*2+1]);
    cv::KeyPoint &r1 = imageKeypoints.at(ptpairs[i*2]);
    cv::KeyPoint &r2 = db[ii].keypoints.at(ptpairs[i*2+1]);

    CvScalar c = colors[10];
    size_t linew = 1;
    if (hg && survivors[i]) {
      c = colors[0];
      linew = MAX(1,thickness);
    }

    cvLine( correspond, cvPointFrom32f(r1.pt),
	    cvPoint(cvRound(r2.pt.x+horizshift), cvRound(r2.pt.y)),
	    c, linew);
  }

  if (displayinfo) {

    stringstream ptext, rtext;
    ptext << "Matched keypoint pairs: " << db[ii].nmatches << "/"
	  << imageKeypoints.size() << " = "
	  << (100.0*db[ii].nmatches/imageKeypoints.size()) << "% with "
	  << db[ii].objectname;
    cvRectangle(correspond, cvPoint(1,8), cvPoint(correspond->width, 22),
		colors[9], CV_FILLED);
    cvPutText(correspond, ptext.str().c_str(),
	      cvPoint(10,20), &statusfont, colors[8]);
    if (hg)
      rtext << "Homography found, surviving keypoint pairs: "
	    << db[ii].nsurvivors << "/" << imageKeypoints.size()
	    << " = " << (100.0*db[ii].nsurvivors/imageKeypoints.size())
	    << "%";
    else
      rtext << "No homography found";
    cvRectangle(correspond, cvPoint(1,28), cvPoint(correspond->width, 42),
		colors[9], CV_FILLED);
    cvPutText(correspond, rtext.str().c_str(),
	      cvPoint(10,40), &statusfont, colors[8]);
  }

  if (openwindow)
    cvNamedWindow("Matching result", 1);

  ShowSMIResults(correspond);

  cvShowImage("Matching result", correspond);
  char c = (char)cvWaitKey(openwindow ? 0 : 25);

  if (c == 's' || c == 'S') {
    cout << "Saved output image: " << outimgname << endl;
    cvSaveImage(outimgname.c_str(), correspond);
  }

  if (openwindow)
    cvDestroyWindow("Matching result");

  cvReleaseImage(&correspond);

  if (c == 'q' || c == 'Q' )
    return false;

  //cout << "Ending ShowNthResult(ii) with ii=" << ii << endl;
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ShowResultsCommon(const IplImage *image, 
					bool nonmatchesalso) {
  bool showmore = true;
  size_t ii=0;
  while (showmore && ii<db.size() &&
	 (nonmatchesalso || db[ii].nmatches) ) {
    showmore = ShowNthResult(image, ii);
    ii++;
  }
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ShowDbResults(const IplImage *image) {

  const size_t thickness = 3; // extra thickness to lines
  const size_t interval = 2;  // add space between images
  const bool white_background = true; // false means black
  const bool borders_over_query_image = true;
  const bool borders_over_db_image = true;
  const bool correspondences = true; // lines between matching points
  const bool gaze_history_crosses = false;
  const bool gaze_history_heatmap = false;

  const bool dbresultimg_reverse_order = false;

  if (!dbresultimg.img) {
    if (dbresultimg.height == 0) {
      cerr << "ERROR: ShowDbResults(): dbresultimg.height has to be " 
	   << "specified beforehand." << endl;
      return false;
    }

    dbresultimg.img = cvCreateImage(cvSize(image->width, dbresultimg.height),
				    8, 3);
    cvSet(dbresultimg.img, white_background ? 
	  cvScalar(255,255,255) : cvScalar(0,0,0));
    for (size_t ii=0; ii<db.size(); ii++) {
      if (!FileExists(db[ii].pathname)) {
	cerr << "ERROR: ShowDbResults(): No image found: " 
	     << db[ii].pathname << endl;
	continue;
      }
      cout << "ShowDbResults():  Loading " << db[ii].pathname << endl;
      IplImage *dbimage = cvLoadImage(db[ii].pathname.c_str(),
				      CV_LOAD_IMAGE_GRAYSCALE);

      if (dbresultimg.scale<0.0) {
	dbresultimg.scale = double(dbresultimg.height)/double(dbimage->height);
	dbresultimg.width = static_cast<size_t>(dbimage->width*dbresultimg.scale);
      }
      ScaleImage(&dbimage, dbresultimg.scale);
      
      size_t horder = dbresultimg_reverse_order ? 
	db.size()-1-db[ii].n : db[ii].n;

      int horizshift = horder*(dbimage->width+interval);
      if (horizshift < (int)dbresultimg.img->width-dbimage->width) {
	cvSetImageROI(dbresultimg.img, cvRect(horizshift, 0, 
					      dbimage->width, 
					      dbimage->height));
	IplImage *dbimg_color = cvCreateImage(cvGetSize(dbimage), 8, 3);
	cvCvtColor(dbimage, dbimg_color, CV_GRAY2BGR);
	cvCopy(dbimg_color, dbresultimg.img);
	cvResetImageROI(dbresultimg.img);
	cvReleaseImage(&dbimg_color);
	cvReleaseImage(&dbimage);
      }
    }
  }


  IplImage *correspond =
    cvCreateImage(cvSize(image->width, image->height+dbresultimg.height), 8, 3);
  cvSet(correspond, white_background?cvScalar(255,255,255):cvScalar(0,0,0));
  cvSetImageROI(correspond, cvRect(0, 0, image->width, image->height));
  IplImage *img_color = cvCreateImage(cvGetSize(image), 8, 3);
  cvCvtColor(image, img_color, CV_GRAY2BGR);
  cvCopy(img_color, correspond);
  cvResetImageROI(correspond);
  cvReleaseImage(&img_color);

  cvSetImageROI(correspond, cvRect(0, image->height, 
				   dbresultimg.img->width, 
				   dbresultimg.img->height));
  cvCopy(dbresultimg.img, correspond);
  cvResetImageROI(correspond);

  for (size_t ii=0; ii<db.size(); ii++) {
    vector<int> &ptpairs = db[ii].ptpairs;
    vector<bool> &survivors = db[ii].survivors;
    bool &hg = db[ii].homography;
    
    int vertishift = image->height;
    int horizshift = (dbresultimg_reverse_order ? 
		      db.size()-1-db[ii].n : db[ii].n)*
      (dbresultimg.width+interval);

    IplImage *ipl_heatmap_ucr = NULL;
    cv::Mat heatmap_uc;

    if (gaze_history_crosses) {
      for (size_t i=0; i<db[ii].gaze_history.size(); i++) {
	CvPoint2D32f &ghp = db[ii].gaze_history[i].second;      
	int d=5;
	CvPoint g = cvPoint(cvRound(ghp.x*dbresultimg.scale),
			    cvRound(ghp.y*dbresultimg.scale));
	cvLine(correspond, cvPoint(g.x-d+horizshift, g.y+vertishift),  
	       cvPoint(g.x+d+horizshift, g.y+vertishift), colors[8], 1);
	cvLine(correspond, cvPoint(g.x+horizshift, g.y-d+vertishift),  
	       cvPoint(g.x+horizshift, g.y+d+vertishift), colors[8], 1);
      }
    }

    if (gaze_history_heatmap) {

      cv::Mat heatmap(db[ii].size, CV_32F, cv::Scalar(0.0));
      cv::Mat heatmap_u, heatmap_ucr;

      // // Heat map using green channel:
      // for (size_t i=0; i<db[ii].gaze_history.size(); i++) {
      // 	CvPoint2D32f &ghp = db[ii].gaze_history[i].second;      
      // 	heatmap.at<float>(ghp) += 1500.0;
      // }
      // GaussianBlur(heatmap, heatmap, cv::Size(101,101), 25.0);
      // heatmap.convertTo(heatmap_u, CV_8U, 255.0);
      // int from_to[] = {0,1};
      // heatmap_uc.create(heatmap_u.size(), CV_8UC3);
      // heatmap_uc.setTo(cv::Scalar(0,0,0));
      // mixChannels(&heatmap_u, 1, &heatmap_uc, 1, from_to, 1);

      // Heat map using OpenCV colormaps:
      for (size_t i=0; i<db[ii].gaze_history.size(); i++) {
	CvPoint2D32f &ghp = db[ii].gaze_history[i].second;      
	heatmap.at<float>(ghp) += 500.0;
      }
      GaussianBlur(heatmap, heatmap, cv::Size(101,101), 20.0);
      sqrt(heatmap, heatmap);
      heatmap.convertTo(heatmap_u, CV_8U, 255.0);
      applyColorMap(heatmap_u, heatmap_uc, cv::COLORMAP_HOT);
      
      resize(heatmap_uc, heatmap_ucr, cv::Size(), 
	     dbresultimg.scale, dbresultimg.scale);

      cvSetImageROI(correspond, cvRect(horizshift, vertishift, 
				       heatmap_ucr.cols, heatmap_ucr.rows));
      IplImage *ipl_heatmap_ucr;
      ipl_heatmap_ucr = cvCreateImage(cvSize(heatmap_ucr.cols, 
					     heatmap_ucr.rows),
				      IPL_DEPTH_8U, 3);
      ipl_heatmap_ucr->imageData = (char *) heatmap_ucr.data;
      cvMax(ipl_heatmap_ucr, correspond, correspond);
      cvResetImageROI(correspond);
    }

    if (hg) {
      
      if (borders_over_db_image) {
	if (useborders) 
	  for (int i = 1; i < 5; i++) {
	    CvPoint r1 = db[ii].dbitemborders[i];
	    CvPoint r2 = db[ii].dbitemborders[MAX((i+1)%5,1)];
	    r1.x = cvRound(r1.x*dbresultimg.scale); 
	    r1.y = cvRound(r1.y*dbresultimg.scale); 
	    r2.x = cvRound(r2.x*dbresultimg.scale); 
	    r2.y = cvRound(r2.y*dbresultimg.scale);
	    cvLine(correspond, cvPoint(r1.x+horizshift, r1.y+vertishift),
		   cvPoint(r2.x+horizshift, r2.y+vertishift), 
		   colors[8], 3+thickness*2);
	    cvLine(correspond, cvPoint(r1.x+horizshift, r1.y+vertishift),
		   cvPoint(r2.x+horizshift, r2.y+vertishift), 
		   colors[ii], 1+thickness);
	  }	
      }
      
      if (borders_over_query_image) {
	for (int i = 1; i < 5; i++) {
	  CvPoint r1 = db[ii].dbitemcorners[i];
	  CvPoint r2 = db[ii].dbitemcorners[MAX((i+1)%5,1)];
	  cvLine(correspond, cvPoint(r1.x, r1.y),
		 cvPoint(r2.x, r2.y), colors[8], 3+thickness*2);
	  cvLine(correspond, cvPoint(r1.x, r1.y),
		 cvPoint(r2.x, r2.y), colors[ii], 1+thickness);
	}
      }

      if (correspondences) {
	size_t ndrawn = 0;      
	for (int i = 0; i < (int)ptpairs.size()/2; i ++) {
	  if (survivors[i]) {
	    cv::KeyPoint &r1 = imageKeypoints.at(ptpairs[i*2]);
	    cv::KeyPoint &r2 = db[ii].keypoints.at(ptpairs[i*2+1]);
	    
	    cvLine(correspond, cvPointFrom32f(r1.pt),
		   cvPoint(cvRound(dbresultimg.scale*r2.pt.x+horizshift), 
			   cvRound(dbresultimg.scale*r2.pt.y+vertishift)),
		   colors[ii], 1);
	    ndrawn++;
	    
	  }
	  if (ndrawn == SIZE_T_MAX)
	    break;
	}
      }

      if (gaze_history_crosses) {
	for (size_t i=0; i<db[ii].gaze_history.size(); i++) {
	  int d=10;
	  double X, Y;
	  CvPoint2D32f &ghp = db[ii].gaze_history[i].second;
	  
	  CalcHomography(db[ii].ihmatrix, ghp.x, ghp.y, X, Y);
	  cvLine(correspond, cvPoint(cvRound(X)-d, cvRound(Y)),  
		 cvPoint(cvRound(X)+d, cvRound(Y)), colors[ii], 1);
	  cvLine(correspond, cvPoint(cvRound(X), cvRound(Y)-d),  
		 cvPoint(cvRound(X), cvRound(Y)+d), colors[ii], 1);
	}
      }

      if (gaze_history_heatmap) {
	cv::Mat heatmap_w(cv::Size(image->width, image->height), CV_8UC3);
	CvMat _ih = cvMat(3, 3, CV_64F, db[ii].ihmatrix);
	warpPerspective(heatmap_uc, heatmap_w, cv::Mat(&_ih), heatmap_w.size());
	
	cvSetImageROI(correspond, cvRect(0, 0, heatmap_w.cols, heatmap_w.rows));
	IplImage *ipl_heatmap_w;
	ipl_heatmap_w = cvCreateImage(cvSize(heatmap_w.cols, heatmap_w.rows), 
				     IPL_DEPTH_8U, 3);
	ipl_heatmap_w->imageData = (char *) heatmap_w.data;
	cvMax(ipl_heatmap_w, correspond, correspond);
	//cvCopy(ipl_scaledhmuc, correspond,ipl_scaledhmuc);
	cvReleaseImage(&ipl_heatmap_w);
	cvResetImageROI(correspond);
      }
      
      if (db[ii].gaze.x >= 0.0) {	
	int d=5;
	CvPoint g = cvPoint(cvRound(db[ii].gaze.x*dbresultimg.scale),
			    cvRound(db[ii].gaze.y*dbresultimg.scale));
	cvLine(correspond, cvPoint(g.x-d+horizshift, g.y+vertishift),  
	       cvPoint(g.x+d+horizshift, g.y+vertishift), colors[8], 4);
	cvLine(correspond, cvPoint(g.x+horizshift, g.y-d+vertishift),  
	       cvPoint(g.x+horizshift, g.y+d+vertishift), colors[8], 4);
	cvLine(correspond, cvPoint(g.x-d+horizshift, g.y+vertishift),  
	       cvPoint(g.x+d+horizshift, g.y+vertishift), colors[ii], 2);
	cvLine(correspond, cvPoint(g.x+horizshift, g.y-d+vertishift),  
	       cvPoint(g.x+horizshift, g.y+d+vertishift), colors[ii], 2);
      }

    } // if (hg)

    if (ipl_heatmap_ucr)
      cvReleaseImage(&ipl_heatmap_ucr);
  }

  ShowSMIResults(correspond);

  if (outputVideo.isOpened())
    outputVideo << correspond;

  cvShowImage("Matching result", correspond);
  char c = (char)cvWaitKey(25);

  if (c == 's' || c == 'S') {
    cout << "Saved output image: " << outimgname << endl;
    cvSaveImage(outimgname.c_str(), correspond);
  }

  cvReleaseImage(&correspond);

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ShowSMIResults(IplImage *img) {
  CvScalar yeleen = CV_RGB(128, 255, 0), col = yeleen;
  for (size_t i=0; i<db.size() && i<sizeof(colors)/sizeof(colors[0]); i++)
    if (db[i].gaze.x>=0.0) {
      col = colors[i];
      break;
    }

  pair<float,float> b_por = smi.B_POR(frame_no); 
  cout << frame_no << " (" << b_por.first << "," << b_por.second << ")" << endl;
  int x = (int)floor(b_por.first+0.5), y = (int)floor(b_por.second+0.5), d = 10;

  cvLine(img, cvPoint(x-d, y),  cvPoint(x+d, y), colors[8], 4);
  cvLine(img, cvPoint(x, y-d),  cvPoint(x, y+d), colors[8], 4);

  cvLine(img, cvPoint(x-d, y),  cvPoint(x+d, y), col, 2);
  cvLine(img, cvPoint(x, y-d),  cvPoint(x, y+d), col, 2);

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::InsidePolygon(CvPoint p, size_t ii) {
  size_t counter = 0;
  size_t i, N = 4;
  double xinters;
  CvPoint p1, p2;

  p1 = db[ii].dbitemcorners[1];
  for (i=2; i<=N+1; i++) {
    p2 = db[ii].dbitemcorners[MAX(i % (N+1),1)];
    if (p.y > MIN(p1.y, p2.y)) {
      if (p.y <= MAX(p1.y, p2.y)) {
        if (p.x <= MAX(p1.x, p2.x)) {
          if (p1.y != p2.y) {
            xinters = (p.y-p1.y)*(p2.x-p1.x)/(p2.y-p1.y)+p1.x;
            if (p1.x == p2.x || p.x <= xinters)
              counter++;
          }
        }
      }
    }
    p1 = p2;
  }

  if (counter % 2 == 0)
    return false;
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessWholeDatabase(const IplImage *image) {

  if (debug)
    cout << "ProcessWholeDatabase(): Starting with DbIndicesInUse()="
	 << DbIndicesInUse() << endl;

  double tt = (double)cvGetTickCount();

  if (DbIndicesInUse()) {
    sort(db.begin(), db.end(), dbitem_originalorder());
    FindMatches();

    if (debug) {
      double t = (double)cvGetTickCount() - tt;
      printf("ProcessWholeDatabase(): Joint index matching to %d images took "
	     "%gms\n", (int)db.size(), t/(cvGetTickFrequency()*1000.));
    }

    if (secondstage) {
      sort(db.begin(), db.end(), dbitem_matchsort());

      size_t jj=0, jjmax = MIN(db.size(),JOINT_2NDSTAGE_NIMAGES);
//#pragma omp parallel for
      for (; jj < jjmax; jj++) {
	if (debug)
	  cout << "ProcessWholeDatabase(): Second stage: i=" << jj << "/"
	       << JOINT_2NDSTAGE_NIMAGES << ", n=" << db[jj].n
	       << ", name=" << db[jj].objectname
	       << ", ndesc=" << db[jj].descriptors.rows
	       << ", nmatches: " << db[jj].nmatches << endl;
	db[jj].ptpairs.clear();
	FindMatchesUsingDescriptors(jj);
	if (debug)
	  cout << "ProcessWholeDatabase(): Second stage: => " 
	       << db[jj].nmatches << endl;
      }

      for (size_t kk=jj; kk<db.size(); kk++) {
	db[kk].ptpairs.clear();
	db[kk].nmatches = 0;
      }
    }

  } else {
    ///int th_id;
    ///#pragma omp parallel for private(th_id)
    //#pragma omp parallel for
    for (size_t ii=0; ii<db.size(); ii++) {
      //th_id = omp_get_thread_num();
      //std::ostringstream ss;
      //ss << "ProcessWholeDatabase(): Thread " << th_id
      // << " working on item " << ii << std::endl;
      //std::cout << ss.str();
      FindMatchesUsingIndex(ii);
    }
  }

  if (debug) {
    tt = (double)cvGetTickCount() - tt;
    printf( "ProcessWholeDatabase(): Matching to %d images took in total "
	    "%gms\n", (int)db.size(), tt/(cvGetTickFrequency()*1000.));
  }

  sort(db.begin(), db.end(), dbitem_matchsort());

  size_t maxchecks = MIN(db.size(),HOMOGRAPHY_NIMAGES);
  for (size_t ii=0; ii<maxchecks; ii++) {
    if (debug)
      cout << "ProcessWholeDatabase(): Starting to check homography of result " 
	   << ii << " / " << maxchecks
	   << " (" <<db[ii].n << "," << db[ii].objectname << ") with "
	   << db[ii].nmatches << " matches" << endl;

    if (db[ii].nmatches < MATCHING_NMATCHES_TH) {
      if (debug)
	cout << "ProcessWholeDatabase(): Too few matches for homography check: "
	     << db[ii].nmatches << ", threshold=" << MATCHING_NMATCHES_TH
	     << ", ending homography checks." << endl;
      break;
    }

    db[ii].homography = FindHomography(image, ii);

    if (debug)
      cout << "ProcessWholeDatabase(): Homography check done: "
	   << (db[ii].homography ? "FOUND" : "NOT FOUND")
	   << ", with " << db[ii].nsurvivors
	   << " matches remaining, threshold=" << MATCHING_NSURVIVORS_TH
	   << endl;
    
    // For exhaustive search, we check the homography for all candidates:
    if (!exhaustivesearch && (db[ii].nsurvivors >= MATCHING_NSURVIVORS_TH))
      break;
  }

  if (HOMOGRAPHY_NIMAGES > 0)
    sort(db.begin(), db.end(), dbitem_survivorsort());

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessDatabaseUntilGoodMatch(const IplImage *image) {

  if (debug)
    cout << "ProcessDatabaseUntilGoodMatch(): Starting" << endl;

  sort(db.begin(), db.end(), dbitem_apriorisort());

  size_t nimgs = 0;

  double tt = (double)cvGetTickCount();
  for (size_t ii=0; ii<db.size(); ii++) {
    nimgs++;
    FindMatchesUsingIndex(ii);
    size_t nmatches = db[ii].nmatches;
    if (debug)
      cout << "ProcessDatabaseUntilGoodMatch(): Found "
	   << db[ii].nmatches << " matching descriptors in "
	   << db[ii].objectname << "." << endl;

    if (nmatches >= MATCHING_NMATCHES_TH) {

      db[ii].homography = FindHomography(image, ii);

      if (db[ii].nsurvivors >= MATCHING_NSURVIVORS_TH) {
	if (updateapriori)
	  UpdateApriori(ii);
	if (debug)
	  cout << "ProcessDatabaseUntilGoodMatch(): Found "
	       << db[ii].nsurvivors << " surviving descriptors." << endl;
	break;
      }

    }
  }
  if (debug) {
    tt = (double)cvGetTickCount() - tt;
    printf("ProcessDatabaseUntilGoodMatch(): Matching to %d images "
	   "took %gms\n", (int)nimgs, tt/(cvGetTickFrequency()*1000.));
  }

  sort(db.begin(), db.end(), dbitem_survivorsort());

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessPrevious(const IplImage *image) {

  if (debug)
    cout << "ProcessPrevious(): Starting" << endl;

  dbitem& prev = previmgs.at(0);
  
  if (prev.keypoints.size()==0)
    return true;

  prev.ptpairs.clear();
  prev.nmatches = 0;	

  double tt = (double)cvGetTickCount();

  FlannFindPairs(prev.descriptors, prev.ptpairs);
  prev.nmatches = prev.ptpairs.size()/2;

  if (debug>1) {
    tt = (double)cvGetTickCount() - tt;
    printf("ProcessPrevious(): Matching time = %gms, ",
	   tt/(cvGetTickFrequency()*1000.));
    cout << "descriptor pairs: " <<  prev.nmatches << endl;
  }

  prev.homography = FindHomography(image, prev);

  if (debug)
    cout << "ProcessPrevious(): Homography check done: "
	 << (prev.homography ? "FOUND" : "NOT FOUND")
	 << ", with " << prev.nsurvivors
	 << " matches remaining, threshold=" << MATCHING_NSURVIVORS_TH
	 << endl;

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::DoGazeMatch() {
  pair<float,float> b_por = smi.B_POR(frame_no); 
  cout << frame_no << " (" << b_por.first << "," << b_por.second << ")" << endl;
  int x = (int)floor(b_por.first+0.5), y = (int)floor(b_por.second+0.5);
  CvPoint p = cvPoint(x, y);

  for (size_t ii=0; ii<db.size(); ii++) {
    //db[i].gaze = cvPoint2D32f(-1.0, -1.0);

    if (!db[ii].homography)
      continue;

    if (InsidePolygon(p, ii)) {
      /// real coordinates should be solved...     
      //db[ii].gaze.x = db[ii].gaze.y = 1.0;

      double X, Y;
      CalcHomography(db[ii].hmatrix, p.x, p.y, X, Y);
      db[ii].gaze.x = X;
      db[ii].gaze.y = Y;
      
      db[ii].gaze_history.push_back(make_pair(frame_no, db[ii].gaze));
    }
  }

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::UpdateApriori(size_t i) {
  db[i].apriori += 1.0/(double)db.size();

  double sum = 0.0;
  for (size_t ii=0; ii<db.size(); ii++)
    sum += db[ii].apriori;
  for (size_t ii=0; ii<db.size(); ii++)
    db[ii].apriori /= sum;
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessCapture(int i, const string& fn) {

  CvCapture* capture;
  string modestr;

  int ex = 0;

  if (i>-1) {
    capture = cvCaptureFromCAM(i);
    modestr = "webcam";
  } else {
    capture = cvCaptureFromFile(fn.c_str());
    modestr = "video";
    
    ex = static_cast<int>(cvGetCaptureProperty(capture, CV_CAP_PROP_FOURCC));
  }

  if (!capture) {
    cerr << "ERROR: No " << modestr << " found" << endl;
    return false;
  }

  char EXT[] = {(char)(ex & 0XFF) , (char)((ex & 0XFF00) >> 8),
		(char)((ex & 0XFF0000) >> 16),(char)((ex & 0XFF000000) >> 24), 
		0};
  int inputWidth  = (int)cvGetCaptureProperty(capture, CV_CAP_PROP_FRAME_WIDTH);
  int inputHeight = (int)cvGetCaptureProperty(capture, CV_CAP_PROP_FRAME_HEIGHT);
  int inputFPS    = (int)cvGetCaptureProperty(capture, CV_CAP_PROP_FPS);
  int inputFourCC = (int)cvGetCaptureProperty(capture, CV_CAP_PROP_FOURCC);

  IplImage* frame = 0;

  cout << "Processing " << modestr << " input" << endl;
  if (debug)
    cout << "Input size: " << inputWidth << "x" << inputHeight << ", FPS: " 
	 << inputFPS << ", FourCC: " << inputFourCC << " (" << ex << ",[" 
	 << EXT << "])" << endl;
  
  cvNamedWindow( modestr.c_str(), 1 );

  if (GetShowOnlineAnyResults())
    cvNamedWindow("Matching result", 1);

  if (outvidname != "") {
    int outputWidth  = inputWidth;
    int outputHeight = inputHeight+dbresultimg.height;
    int outputFPS    = inputFPS>0 ? inputFPS : 25;
    
    outputVideo.open(outvidname, 0, outputFPS, 
		     cvSize(outputWidth, outputHeight));
    if (!outputVideo.isOpened()) {
      cerr << "ERROR: Failed to open output video" << endl;
      return false;
    }

    if (debug)
      cout << "Opened output video of size: " << outputWidth << "x" 
	   << outputHeight << ", FPS: " 
	   << outputFPS << ", FourCC: " << 0 << endl;
  }

  for (;;) {
    frame = cvQueryFrame(capture);
    if (!frame) {
      cerr << "ERROR: Failed to get frames from the " << modestr << endl;
      cvDestroyWindow(modestr.c_str());
      return false;
    }
    cvShowImage(modestr.c_str(), frame);

    char c = (char)cvWaitKey(GetShowOnlineAnyResults() ? 1 : 25);

    if (c == 'q' || c == 'Q' )
      break;

    if (GetShowOnlineAnyResults() || c == 's' || c == 'S' ) {
      IplImage *image = cvCreateImage(cvSize(frame->width, frame->height),
				      8, 1);
      cvCvtColor(frame, image, CV_BGR2GRAY);
      string imgfn = "webcam";
      if (i<0) {
	stringstream ss;
	ss << fn << "[" << frame_no << "]";
	imgfn = ss.str();
      }
	
      ProcessImage(image, imgfn);
    }
  }
  cout << "Exiting " << modestr << " loop" << endl;

  if (GetShowOnlineAnyResults())
    cvDestroyWindow("Matching result");

  cvReleaseImage(&frame);
  cvDestroyWindow(modestr.c_str());

  return true;
}

// ---------------------------------------------------------------------

#ifndef __WINDOWS__
bool ObjectDetection::ProcessSoap(string &port, bool dummy) {
  cout << "Starting to process soap input, port=" << port
       << ", dummy=" << dummy << endl;

  SetShowKeypoints(false);
  SetShowResults(false);

  if (!FileExists("soap")) {
    cout << "Making a subdirectory \"soap\"." << endl;
    if (mkdir("soap", 0777) == -1)
      cerr << "WARNING: Could not create directory \"soap\"" << endl;
  }

  SoapRouter *router;

  if (debug>1)
    hlog_set_level(HLOG_VERBOSE);
  else
    hlog_set_level(HLOG_INFO);

  char *argv[3];
  argv[0] = strdup(NHTTPD_ARG_PORT);
  argv[1] = strdup((char*)port.c_str());
  argv[2] = NULL;
  log_info1("Initializing");
  herror_t err = soap_server_init_args(2, argv);
  free(argv[0]);
  free(argv[1]);

  //err = soap_server_init_args(0, 0);
  if (err != H_OK) {
    log_error4("%s():%s [%d]", herror_func(err), herror_message(err),
	       herror_code(err));
    herror_release(err);
    return 1;
  }

  router = soap_router_new();

  log_info1("Registering services");
  if (dummy)
    soap_router_register_service(router, SoapExecuteDummy,
				 "sendImage", "urn:od");
  else
    soap_router_register_service(router, SoapExecuteReal,
				 "sendImage", "urn:od");

  soap_router_register_service(router, SoapStatus, "status", "urn:od");

  log_info1("Registering router");
  soap_server_register_router(router, "/od");

  log_info1("Send SIGTERM to shutdown");
  cout << endl << "Server URL: http://localhost:" << port << "/od"
       << endl << endl;
  soap_server_run();

  log_info1("Shutting down\n");
  soap_server_destroy();

  return true;
}
#endif // __WINDOWS__

// ---------------------------------------------------------------------

bool ObjectDetection::AnalyseImages(const string& fn) {

  SetShowKeypoints(false);
  SetShowResults(false);
  SetUpdateApriori(false);

  vector<vector<string> > aimages;
  if (!LoadImgList(aimages, fn)) {
    cerr << "ERROR: Cannot load analyse image file " << fn << endl;
    return false;
  }

  double tt = (double)cvGetTickCount();

  size_t ncorr_img = 0, ncorr_det = 0, ndet = 0;
  vector<string> failed;

  for (size_t ii=0; ii<aimages.size(); ii++) {
    vector<string> &aimg = aimages[ii];
    string aimgname = aimg[0];
    cout << endl << "Analysing image " << ii << "/" << aimages.size()
	 << " [" << aimgname << "] with " << scales.size() << " scales."
	 << endl;

    stringstream finalresult;
    vector<double>::const_iterator it;

    for (it=scales.begin(); it!=scales.end(); ++it) {
      IplImage *image = cvLoadImage(aimgname.c_str(), CV_LOAD_IMAGE_GRAYSCALE);
      SetScale(*it);
      string imgfn = "ZZZ";
      ProcessImage(image, imgfn);
      cout << "Best match: " << db[0].imagename << " with "
	   << (HOMOGRAPHY_NIMAGES ? db[0].nsurvivors : db[0].nmatches)
	   << " points";

      if (analyseonlybest) {
	ndet++;
	if (HOMOGRAPHY_NIMAGES && db[0].nsurvivors < MATCHING_NSURVIVORS_TH) {
	  finalresult << "no-match";
	  cout << " (< " << MATCHING_NSURVIVORS_TH << "): REJECTED" << endl;
	} else {
	  finalresult << db[0].imagename;
	  cout << endl;
	  break;
	}
      } else {
	cout << endl;
	ndet += aimg.size()-1;
	for (size_t jj=0; jj<db.size(); jj++) {
	  if (db[jj].nsurvivors >= MATCHING_NSURVIVORS_TH) {
	    if (jj)
	      finalresult << ";";
	    finalresult << db[jj].imagename;
	  }
	}	
      }

    } // for (it=scales...
      
    size_t corr=0;
    for (size_t i=1; i<aimg.size(); i++) {
      if (aimg[i] == "")
	cerr << "ERROR: Empty aimg: i=" << i << ", aimg[i]=" << aimg[i] << endl;
      size_t m = finalresult.str().find(aimg[i]);
      if (m != string::npos) {
	corr++;
	cout << "CORRECT RESULT FOUND: " << finalresult.str() << " : " 
	     << aimg[i] << endl;	
	if (analyseonlybest) 
	  break;
      }
    }

    ncorr_det += corr;

    if (analyseonlybest ? corr : corr == (aimg.size()-1))
      ncorr_img++;
    else {
      stringstream fail;
      fail << aimgname << " match=" <<
	(db[0].nsurvivors ? db[0].imagename : "no-match")
	   << " (" << finalresult.str() << ") correct=";
      for (size_t i=1; i<aimg.size(); i++) {
	if (i>1)
	  fail << ",";
	fail << aimg[i];
      }
      failed.push_back(fail.str());
    }
  }

  cout << endl;

  tt = (double)cvGetTickCount() - tt;

  if (RecordMatchingKeypoints(ANY)) {
    ofstream mkps(matchingkeypointsname.c_str());
    map<size_t, size_t>::const_iterator mit;
    for (mit=matchingkeypoints.begin(); mit!=matchingkeypoints.end(); ++mit)
      mkps << mit->first << " " << mit->second << endl;
    mkps.close();
    cout << "Wrote matching keypoints to " << matchingkeypointsname << endl;
  }

  cout << "Analyse results:" << endl 
       << ncorr_img << "/" << aimages.size() << " = " 
       << (double)ncorr_img/aimages.size() << " images correct" << endl
       << ncorr_det << "/" << ndet << " = "
       << (double)ncorr_det/ndet << " detections correct" << endl;

  printf( "\nTotal time elapsed = %gms, time per test image = %gms\n\n",
	  tt/(cvGetTickFrequency()*1000.),
	  tt/(cvGetTickFrequency()*1000.)/aimages.size());

  cout << "Incorrect images:" << endl;
  for (size_t ii=0; ii<failed.size(); ii++)
    cout << ii << ": " << failed[ii] << endl;

  return true;
}

// ---------------------------------------------------------------------

#ifndef __WINDOWS__
herror_t ObjectDetection::SoapExecuteCommon(SoapCtx *req, SoapCtx *res,
					    bool dummy) {
  n_soap_threads++;
  int debug = op->GetDebug();

  if (debug)
    cout << endl << "Now in ObjectDetection::SoapExecuteCommon() : "
	 << "threads=" << n_soap_threads
	 << endl << endl;
  herror_t err;

  err = soap_env_new_with_response(req->env, &res->env);
  if (err != H_OK) {
    n_soap_threads--;
    return err;
  }

  string imgname = "foo.jpg";

  xmlNodePtr methodptr = soap_env_get_method(req->env);
  xmlNodePtr node = soap_xml_get_children(methodptr);
  while (node) {
    if (node->name) {
      string nodename = (char*)node->name;
      if (debug>1)
	cout << "name=" << nodename;
      if (node->children && node->children->content) {
	string nodecontent = (char*)node->children->content;
	if (debug>1)
	  cout << ", content=" << nodecontent;
	if (nodename == "filename")
	  imgname = nodecontent;
      }
      if (debug>1)
	cout << endl;
    }
    node = soap_xml_get_next(node);
  }

  bool ok = true;

  if (n_soap_threads > 10) {
    cerr << "ERROR: Too many SOAP threads" << endl;
    ok = false;
  }

  if (!req->attachments) {
    cerr << "ERROR: No attachments found" << endl;
    ok = false;
  }

  part_t *part = req->attachments->parts;
  if (part == NULL) {
    cerr << "ERROR: No attachment parts found" << endl;
    ok = false;
  }

  timeval tval;
  gettimeofday(&tval, NULL);
  stringstream timess;
  timess << tval.tv_sec << ":" << tval.tv_usec;
  string qbasename = timess.str();

  IplImage* soapimage = NULL;
  if (ok) {
    if (debug>1)
      cout << "Image filename: " << part->filename << endl;

    soapimage = cvLoadImage(part->filename, CV_LOAD_IMAGE_GRAYSCALE);
    if (!soapimage) {
      cerr << "ERROR: Cannot load image " << part->filename << endl;
      ok = false;
    }
    
    ifstream infile(part->filename, std::ios_base::binary);
    if (!infile) {
      cerr << "ERROR: Cannot open input stream for " << part->filename << endl;
      ok = false;
    }

    if (ok) {
      string outfilename = "soap/" + qbasename + ".jpg";
      ofstream outfile(outfilename.c_str(), std::ios_base::binary);
      if (!outfile) {
	cerr << "ERROR: Cannot open output stream for " << outfilename << endl;
	ok = false;
      } else {
	outfile << infile.rdbuf();
	outfile.close();
      }
      infile.close();
    }

    // stringstream cpcommstr;
    // cpcommstr << "cp " << part->filename << " soap/"
    // 	      << qbasename << ".jpg";
    // int sysout = system(cpcommstr.str().c_str());
    // if (sysout)
    //   cerr << "WARNING: Could not copy image to directory \"soap\"" << endl;
  }

  dbitem best;
  ProcessFail(best);
  if (ok && !dummy) {
    pthread_mutex_lock(&pi_mutex);
    ok = op->ProcessImageFromStatic(soapimage, best);
    pthread_mutex_unlock(&pi_mutex);
  } else
    ProcessDummy(best);

  //cvReleaseImage(&soapimage); // released in ProcessImageFromStatic()

  bool accept = true;
  if (best.nsurvivors<MATCHING_NSURVIVORS_TH)
    accept = false;

  if (debug)
    cout << "Matched keypoint pairs: " << best.nsurvivors << " with "
	 << best.objectname << " : " << (accept?"ACCEPTED":"REJECTED")
	 << endl;

  stringstream qresstr;
  qresstr << "soap/" << qbasename << ".res";
  ofstream qres(qresstr.str().c_str());

  soap_env_add_itemf(res->env, "xsd:string", "result", accept?"1":"0");
  soap_env_add_itemf(res->env, "xsd:string", "page",
		     best.objectname.c_str());
  soap_env_add_itemf(res->env, "xsd:string", "info",
		     best.imagename.c_str());
  qres << "result=" << (accept?"1":"0") << endl
       << "page=" << best.objectname << endl
       << "info=" << best.imagename << endl;

  stringstream sspoints;
  sspoints << best.nsurvivors;
  soap_env_add_itemf(res->env, "xsd:string", "points",
		     sspoints.str().c_str());
  qres << "points=" << sspoints.str() << endl;

  stringstream sscenter;
  CvPoint &p = best.querycorners[0];
  sscenter << (double)p.x/best.size.width << ","
	   << (double)p.y/best.size.height;
  soap_env_add_itemf(res->env, "xsd:string", "center",
		     sscenter.str().c_str());
  qres << "center=" << sscenter.str() << endl;

  char xx[80] = "";
  sprintf(xx, "%.2f", (double)p.x/best.size.width);
  soap_env_add_itemf(res->env, "xsd:string", "centerx", xx);
  qres << "centerx=" << xx << endl;

  char yy[80] = "";
  sprintf(yy, "%.2f", (double)p.y/best.size.height);
  soap_env_add_itemf(res->env, "xsd:string", "centery", yy);
  qres << "centery=" << yy << endl;

  stringstream sscorners;
  for (int i = 1; i < 5; i++) {
    if (i>1)
      sscorners << "; ";
    CvPoint &p = best.querycorners[i];
    sscorners << (double)p.x/best.size.width << ","
	      << (double)p.y/best.size.height;
  }
  soap_env_add_itemf(res->env, "xsd:string", "corners",
		     sscorners.str().c_str());
  qres << "corners=" << sscorners.str() << endl;

  stringstream ssinvcorners;
  for (int i = 1; i < 5; i++) {
    if (i>1)
      ssinvcorners << ";";
    CvPoint &p = best.dbitemcorners[i];
    ssinvcorners << p.x << "," << p.y;
  }
  soap_env_add_itemf(res->env, "xsd:string", "invcorners",
		     ssinvcorners.str().c_str());
  qres << "invcorners=" << ssinvcorners.str() << endl;

  qres.close();

  n_soap_threads--;
  return H_OK;
}

// ---------------------------------------------------------------------

herror_t ObjectDetection::SoapStatus(SoapCtx *req, SoapCtx *res) {

  int debug = op->GetDebug();

  if (debug)
    cout << endl << "Now in ObjectDetection::SoapStatus()" << endl << endl;
  herror_t err = soap_env_new_with_response(req->env, &res->env);
  if (err != H_OK)
    return err;

  soap_env_add_itemf(res->env, "xsd:string", "status", "ok");

  double dbsize;
  size_t nq, ns;
  time_t st;
  op->GetStatusInfoFromStatic(dbsize, nq, ns, st);

  struct tm *stm = localtime(&st);
  char ols[100];
  if (strftime(ols, sizeof(ols), "%c", stm) != 0)
    soap_env_add_itemf(res->env, "xsd:string", "onlinesince", ols);

  stringstream ssdbsize;
  ssdbsize << dbsize;
  soap_env_add_itemf(res->env, "xsd:string", "databasesize",
		     ssdbsize.str().c_str());

  stringstream ssnq;
  ssnq << nq;
  soap_env_add_itemf(res->env, "xsd:string", "nqueries", ssnq.str().c_str());

  stringstream ssns;
  ssns << ns;
  soap_env_add_itemf(res->env, "xsd:string", "nsoaps", ssns.str().c_str());

  return H_OK;
}
#endif // __WINDOWS__

// ---------------------------------------------------------------------

void ObjectDetection::ProcessDummy(dbitem &dummybest) {
  dummybest.imagename  = "dummy";
  dummybest.objectname = "dummier";
  dummybest.nsurvivors = 666;
  dummybest.gaze = cvPoint2D32f(7.0, 7.0);

  CvPoint c[5] = {{5,5},{0,0},{10,0},{10,10},{0,10}};
  for (size_t i=0; i<5; i++)
    dummybest.querycorners[i] = c[i];
  dummybest.size       = cvSize(123, 456);
}

// ---------------------------------------------------------------------

void ObjectDetection::ProcessFail(dbitem &failbest) {
  failbest.imagename  = "fail";
  failbest.objectname = "image-recognition-failed";
  failbest.nsurvivors = 0;
  failbest.gaze = cvPoint2D32f(-1.0, -1.0);
  CvPoint c[5] = {{5,5},{0,0},{10,0},{10,10},{0,10}};
  for (size_t i=0; i<5; i++)
    failbest.querycorners[i] = c[i];
  failbest.size       = cvSize(123, 456);
}

// ---------------------------------------------------------------------

bool ObjectDetection::ProcessImageFromStatic(IplImage *image, dbitem &res) {
  //image = cvCreateImage(cvSize(img->width, img->height), 8, 1);
  //cvCvtColor(img, image, CV_BGR2GRAY);
  //image = img;

  //sleep(1);
  string imgfn = "ZZZ";
  bool ok = ProcessImage(image, imgfn);
  if (ok) {
    res = db[0];
    nsoaps++;
  }
  return ok;
}

// ---------------------------------------------------------------------

void ObjectDetection::GetStatusInfoFromStatic(double &dbsize, size_t &nq,
					      size_t &ns, time_t &st) {
  dbsize = db.size();
  nq     = nqueries;
  ns     = nsoaps;
  st     = start_time;

  nsoaps++;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadImage(IplImage** imgptr, const string& fn) {

  if (debug)
    cout << "LoadImage(): Starting to load " << fn << endl;

  double tt = (double)cvGetTickCount();
  *imgptr = cvLoadImage(fn.c_str(), CV_LOAD_IMAGE_GRAYSCALE);
  if (!*imgptr) {
    cerr << "ERROR: Cannot load image " << fn << endl;
    return false;
  }
  tt = (double)cvGetTickCount() - tt;
  if (debug)
    printf( "LoadImage(): Image loaded, time = %gms\n",
	    tt/(cvGetTickFrequency()*1000.));

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ScaleImage(IplImage** imgptr) {

  IplImage*& image = *imgptr;
  //static bool firstcall = true;
  IplImage *tmp = cvCreateImage(cvSize((int)(scale*(image->width)),
				       (int)(scale*(image->height))),
				8, 1);
  cvResize(image,tmp);
  cvReleaseImage(&image);
  image = tmp;

  //if (firstcall && debug)
  if (debug>1)
    cout << "ScaleImage(): Using scale=" << scale
	 << ". Image scaled to " << image->width << "x"
	 << image->height << " pixels." << endl;
  //firstcall = false;

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ResizeImage(IplImage** imgptr) {

	IplImage*& image = *imgptr;

	double ratio = (double)(image->width)/(image->height);

	int newwidth=0, newheight=0;
	if(ratio >= 1){
		newwidth=newsize;
		newheight=static_cast<int>(round(newwidth/ratio));
	} else{
		newheight=newsize;
		newwidth=static_cast<int>(round(newheight*ratio));
	}

	IplImage *tmp = cvCreateImage(cvSize(newwidth,newheight),8,1);
	cvResize(image,tmp);
	cvReleaseImage(&image);
	image = tmp;


  if (debug)
    cout << "ResizeImage(): to " << newsize
	 << ". image resize to " << newwidth << "x" << newheight
	 << " pixels." << endl;

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::CreatePolygonMask(IplImage** imgptr, 
					IplImage** maskptr, 
					const string& fn) {

  vector<CvPoint> ppts;
  bool ok = LoadPolygon(ppts, fn);

  if (ppts.size() != 4) {
    cerr << "CreatePolygonMask(): ERROR: Unsupported number of polygon vertices: " 
	 << ppts.size() << "!= 4" << endl;
    return false;
  }

  CvPoint polypts[4];
  for (size_t i = 0; i < ppts.size(); i++)
    polypts[i] = ppts[i];

  IplImage*& image = *imgptr;
  IplImage*& mask  = *maskptr;
  
  mask = cvCreateImage(cvGetSize(image), 8, 1);
  cvSet(mask, colors[9]);
  //CvPoint polypts[4] = {{10,10},{10,56},{43,66},{30,8}}; 
  cvFillConvexPoly(mask, polypts, 4, colors[8]);

  return ok;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadPolygon(vector<CvPoint>& pts, const string& fn) {
  if (debug)
    cout << "LoadPolygon(): Starting to load " << fn << endl;

  ifstream polygonfile(fn.c_str());
  if (!polygonfile) {
    cerr << "LoadPolygon(): ERROR: Polygon file " << fn
	 << " not found" << endl;
    return false;
  }
  
  int polygoncount;
  polygonfile >> polygoncount;

  for (int i = 0; i < polygoncount; i++) {
    CvPoint pt;
    string tempx, tempy;
    polygonfile >> tempx;
    polygonfile >> tempy;
    pt.x = atoi(tempx.c_str());
    pt.y = atoi(tempy.c_str());
    pts.push_back(pt);
  }

  return true;
}

// ---------------------------------------------------------------------

string ObjectDetection::DetectorString(const detector_type &dt, bool lc) {
  switch (dt) {
  case (DET_NONE):
    return (lc?"none":"NONE");
  case (DET_SURF):
    return (lc?"surf":"SURF");
  case (DET_GPUSURF):
    return (lc?"gpusurf":"GPUSURF");
  case (DET_MSER):
    return (lc?"mser":"MSER");
  case (DET_SIFT):
    return (lc?"sift":"SIFT");
  case (DET_FAST):
    return (lc?"fast":"FAST");
  case (DET_STAR):
    return (lc?"star":"STAR");
  case (DET_ORB):
    return (lc?"orb":"ORB");
  case (DET_GTFF):
    return (lc?"gtff":"GTFF");
  case (DET_HARRIS):
    return (lc?"harris":"HARRIS");

  default:
    return "XXX";
  }
}

// ---------------------------------------------------------------------

string ObjectDetection::AllDetectors() {
  return DetectorString(DET_NONE,true) + "," +
    DetectorString(DET_SURF,true) + "," +
    DetectorString(DET_GPUSURF,true) + "," +
    DetectorString(DET_MSER,true) + "," +
    DetectorString(DET_SIFT,true) + "," +
    DetectorString(DET_FAST,true) + "," +
    DetectorString(DET_STAR,true) + "," +
    DetectorString(DET_ORB,true) + "," +
    DetectorString(DET_GTFF,true) + "," +
    DetectorString(DET_HARRIS,true);
}

// ---------------------------------------------------------------------

string ObjectDetection::DescriptorString(const desc_type &dt, bool lc) {
  switch (dt) {
  case (DES_SURF):
    return (lc?"surf":"SURF");
  case (DES_GPUSURF):
    return (lc?"gpusurf":"GPUSURF");
  case (DES_SIFT):
    return (lc?"sift":"SIFT");
  default:
    return "XXX";
    }
}

// ---------------------------------------------------------------------

string ObjectDetection::AllDescriptors() {
  return DescriptorString(DES_SURF,true) + "," +
    DescriptorString(DES_GPUSURF,true) + "," +
    DescriptorString(DES_SIFT,true);
}

// ---------------------------------------------------------------------

int ObjectDetection::UsedDescriptorSize() {
  switch (used_descriptor.name) {
    case (DES_SURF):
    case (DES_GPUSURF):
      // This seems to be suddenly changed to 128 ???
      // Back to 64, it seems:
      return 64;
      //return 128;
    case (DES_SIFT):
      return 128;
    default:
      return 0;
    }
}

// ---------------------------------------------------------------------

bool ObjectDetection::Detect(const IplImage *imagep, IplImage** maskptr) {
  if (!imagep) {
    cerr << "Detect(): ERROR: Image pointer is NULL" << endl;
    return false;
  }
  IplImage*& maskp = *maskptr;

  double tt = (double)cvGetTickCount();
  cv::Mat img(imagep, false);

  cv::Mat mask;
  if (&maskp && maskp) {
    cv::Mat temp(maskp, false);
    temp.convertTo(mask, CV_8UC1);
    threshold(mask, mask, 128, 255, CV_THRESH_BINARY);
  }

  switch (used_detector) {

  case (DET_NONE): {
    imageKeypoints.clear();
    break;
  }

  case (DET_SURF): {
    cv::SurfFeatureDetector surf(SURF_HESSIAN_THRESHOLD);
    surf.detect(img, imageKeypoints, mask);
    break;
  }

  case (DET_GPUSURF): {
    cv::gpu::GpuMat gpuimg;
    gpuimg.upload(img);
    cv::gpu::GpuMat gpukeypoints;
    cv::gpu::SURF_GPU gpusurf(SURF_HESSIAN_THRESHOLD);
    gpusurf(gpuimg, cv::gpu::GpuMat(), gpukeypoints);
    gpusurf.downloadKeypoints(gpukeypoints, imageKeypoints);
    break;
  }

  case (DET_MSER): {
    // For some reason this doesn't work:
    //cv::MserFeatureDetector mser();
    //mser.detect(img, imageKeypoints);
    cv::Ptr<cv::FeatureDetector> detector = 
      cv::FeatureDetector::create("MSER");
    detector->detect(img, imageKeypoints, mask);
    break;
  }

  case (DET_SIFT): {
    cv::Ptr<cv::FeatureDetector> detector = 
      cv::FeatureDetector::create("SIFT");
    detector->detect(img, imageKeypoints, mask);
    break;
  }

  case (DET_FAST): {
    cv::Ptr<cv::FeatureDetector> detector = 
      cv::FeatureDetector::create("FAST");
    detector->detect(img, imageKeypoints, mask);
    break;
  }

  case (DET_STAR): {
    cv::Ptr<cv::FeatureDetector> detector = 
      cv::FeatureDetector::create("STAR");
    detector->detect(img, imageKeypoints, mask);
    break;
  }

  case (DET_ORB): {
    cv::Ptr<cv::FeatureDetector> detector = 
      cv::FeatureDetector::create("ORB");
    detector->detect(img, imageKeypoints, mask);
    break;
  }

  case (DET_GTFF): {
    cv::Ptr<cv::FeatureDetector> detector = 
      cv::FeatureDetector::create("GTFF");
    detector->detect(img, imageKeypoints, mask);
    break;
  }
 
  case (DET_HARRIS): {
    cv::Ptr<cv::FeatureDetector> detector = 
      cv::FeatureDetector::create("HARRIS");
    detector->detect(img, imageKeypoints, mask);
    break;
  }

  default:
    cerr << "Detect(): ERROR: Detector not set" << endl;
    return false;
  }

  /*
  if (maskp) {
    cout << "Detect(): Found " << imageKeypoints.size()
	 << " " << DetectorString(used_detector) << " keypoint" 
	 << (imageKeypoints.size()>1?"s":"") << endl;

    vector<cv::KeyPoint>::const_iterator kpi;
    for (kpi=imageKeypoints.begin(); kpi!=imageKeypoints.end(); ++kpi) {
      const cv::Point2f &kp = (*kpi).pt;
      int xx = kp.x, yy= kp.y; 
      cout << "Keypoint at " << xx << "," << yy << " (" << mask.cols << "," << mask.rows << ")";
      //if (mask.at<unsigned char>(xx, yy) == 0) {
      //cout << " outside mask";
      //}
      cout << endl;
    }
  }
  */

  if (used_detector != DET_NONE && imageKeypoints.empty()) {
    cv::KeyPoint dummy;
    // KeyPoint.pt is a Point2f:
    dummy.pt.x = static_cast<float>((imagep->width)/2.0);
    dummy.pt.y = static_cast<float>((imagep->height)/2.0);  
    dummy.size = 1.0;
    imageKeypoints.push_back(dummy);
  }
  
  if (debug) {
    cout << "Detect(): Found " << imageKeypoints.size()
	 << " " << DetectorString(used_detector) << " keypoint" 
	 << (imageKeypoints.size()!=1?"s":"");
    tt = (double)cvGetTickCount() - tt;
    det_ext_time.first += tt;
    printf(", extraction time = %gms\n", tt/(cvGetTickFrequency()*1000.));
  }
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::Extract(const IplImage *image) {
  if (!image) {
    cerr << "Extract(): ERROR: Image pointer is NULL" << endl;
    return false;
  }
  
  double tt = (double)cvGetTickCount();
  cv::Mat img(image, false);

  switch (used_descriptor.name) {

  case (DES_SURF): {
    cv::SurfDescriptorExtractor surf;
    surf.compute(img, imageKeypoints, imageDescriptors);
    break;
  }

  case (DES_GPUSURF): {
    cv::gpu::GpuMat gpuimg;
    gpuimg.upload(img);
    cv::gpu::GpuMat gpukeypoints;
    cv::gpu::GpuMat gpudescriptors;
    cv::gpu::SURF_GPU gpusurf(SURF_HESSIAN_THRESHOLD);
    //gpusurf.uploadKeypoints(imageKeypoints, gpukeypoints);
    gpusurf(gpuimg, cv::gpu::GpuMat(), gpukeypoints, gpudescriptors);
    gpusurf.downloadKeypoints(gpukeypoints, imageKeypoints);
    vector<float> imgDescriptors;
    gpusurf.downloadDescriptors(gpudescriptors, imgDescriptors);
    imageDescriptors.create(gpudescriptors.rows, gpudescriptors.cols, CV_32F);
    for (size_t i=0; i<imgDescriptors.size(); i++)
      imageDescriptors.at<float>(i/64, i%64) = imgDescriptors.at(i);
    break;
  }

  case (DES_SIFT): {
    cv::SiftDescriptorExtractor sift;
    sift.compute(img, imageKeypoints, imageDescriptors);
    break;
  }

  default:
    cerr << "Extract(): ERROR: Detector not set" << endl;
    return false;
  }

  if (debug) {
    cout << "Extract(): Extracted: " << imageDescriptors.rows
	 << " " << DescriptorString(used_descriptor.name) 
	 << (used_descriptor.root ? " (root)" : "")
	 << (used_descriptor.norm ? " (norm)" : "")
	 << " descriptors (" << imageDescriptors.cols << ") for " 
	 << (used_detector!=DET_NONE ? DetectorString(used_detector) : DescriptorString(used_descriptor.name)) 
	 << " keypoints";
    tt = (double)cvGetTickCount() - tt;
    det_ext_time.second += tt;
    printf(", extraction time = %gms\n", tt/(cvGetTickFrequency()*1000.));
  }

  if (imageDescriptors.cols != UsedDescriptorSize()) {
    cerr << "Extract(): ERROR: Descriptor dimensionality not what was expected: "
	 << imageDescriptors.cols << "!=" << UsedDescriptorSize() << endl;
    return false;
  }

  if (!imageDescriptors.rows)
    return false;

  return true;
}

// ---------------------------------------------------------------------

void ObjectDetection::FlannFindPairs(const cv::Mat &dbDescriptors,
				     vector<int>& ptpairs) {
  double tt = (double)cvGetTickCount();

  //int length = (int)(imageDescriptors->elem_size/sizeof(float));
  //cv::Mat m_image(imageDescriptors->total, length, CV_32F);
  //FillDescriptorMatrix(m_image, imageDescriptors);

  //cv::Mat m_dbimage(dbDescriptors->total, length, CV_32F);
  //FillDescriptorMatrix(m_dbimage, dbDescriptors);

  cv::Mat m_indices(imageDescriptors.rows, 2, CV_32S);
  cv::Mat m_dists(  imageDescriptors.rows, 2, CV_32F);

  double t = (double)cvGetTickCount() - tt;
  if (debug>1)
    printf("FlannFindPairs(): Initialization done, time elapsed = %gms\n",
	   t/(cvGetTickFrequency()*1000.));

  FlannIndexType flann_index(dbDescriptors,
			     cvflann::KDTreeIndexParams(FLANN_NTREES));

  t = (double)cvGetTickCount() - tt;
  if (debug>1) {
    printf("FlannFindPairs(): Index generation done, time elapsed = %gms\n",
	   t/(cvGetTickFrequency()*1000.));
    cout << "FlannFindPairs(): Built index of size=" << flann_index.size()
	 << ", veclen=" << flann_index.veclen() << endl;
  }

  flann_index.knnSearch(imageDescriptors, m_indices, m_dists, 2,
			cvflann::SearchParams(FLANN_SEARCH_CHECKS));

  t = (double)cvGetTickCount() - tt;
  if (debug>1)
    printf("FlannFindPairs(): Search done, time elapsed = %gms\n",
	   t/(cvGetTickFrequency()*1000.));

  int* indices_ptr = m_indices.ptr<int>(0);
  float* dists_ptr = m_dists.ptr<float>(0);

  vector<size_t> indexcounts(flann_index.size());
  for (int i=0;i<m_indices.rows;++i) {
    indexcounts.at(indices_ptr[2*i])++;
  }
  // for (size_t i=0;i<indexcounts.size();++i) {
  //   cout << "i=" << i << ", count=" << indexcounts.at(i) << endl;
  // }

  for (int i=0;i<m_indices.rows;++i) {
    if ((dists_ptr[2*i]<FLANN_SIMILARITY_RATIO_SEPARATE*dists_ptr[2*i+1])
	&& (indexcounts.at(indices_ptr[2*i])<FLANN_MULTIPLE_MATCH_TH))
      {
	ptpairs.push_back(i);
	ptpairs.push_back(indices_ptr[2*i]);
      }
  }
  tt = (double)cvGetTickCount() - tt;
  if (debug>1)
    printf("FlannFindPairs(): Total time = %gms\n",
	   tt/(cvGetTickFrequency()*1000.));
}

// ---------------------------------------------------------------------

void ObjectDetection::FlannFindPairs(FlannIndexType *index,
				     vector<int>& ptpairs) {
  if (!index) {
    if (debug)
      printf("FlannFindPairs(): NULL index, nothing to do here.\n");
    return;
  }

  double tt = (double)cvGetTickCount();

  //int length = (int)(imageDescriptors->elem_size/sizeof(float));
  //cv::Mat m_image(imageDescriptors->total, length, CV_32F);
  //FillDescriptorMatrix(m_image, imageDescriptors);

  cv::Mat m_indices(imageDescriptors.rows, 2, CV_32S);
  cv::Mat m_dists(  imageDescriptors.rows, 2, CV_32F);

  double t = (double)cvGetTickCount() - tt;
  if (debug>1)
    printf("FlannFindPairs(): Initialization done, time elapsed = %gms\n",
	   t/(cvGetTickFrequency()*1000.));

  index->knnSearch(imageDescriptors, m_indices, m_dists, 2,
		   cvflann::SearchParams(FLANN_SEARCH_CHECKS));

  t = (double)cvGetTickCount() - tt;
  if (debug>1)
    printf("FlannFindPairs(): Search done, time elapsed = %gms\n",
	   t/(cvGetTickFrequency()*1000.));

  int* indices_ptr = m_indices.ptr<int>(0);
  float* dists_ptr = m_dists.ptr<float>(0);
  double similarity_ratio = FLANN_SIMILARITY_RATIO_SEPARATE;
  if (DbIndicesInUse())
    similarity_ratio = FLANN_SIMILARITY_RATIO_JOINT;
  for (int i=0;i<m_indices.rows;++i) {
    if (dists_ptr[2*i]<similarity_ratio*dists_ptr[2*i+1]) {
      ptpairs.push_back(i);
      ptpairs.push_back(indices_ptr[2*i]);
    }
  }

  tt = (double)cvGetTickCount() - tt;
  if (debug>1)
    printf("FlannFindPairs(): Total time = %gms\n",
	   tt/(cvGetTickFrequency()*1000.));
}

// ---------------------------------------------------------------------

/*
void ObjectDetection::FillDescriptorMatrix(cv::Mat &mat,
					   const CvSeq* descriptors) {

  int length = (int)(descriptors->elem_size/sizeof(float));
  CvSeqReader img_reader;
  float* img_ptr = mat.ptr<float>(0);
  cvStartReadSeq( descriptors, &img_reader );
  for(int i = 0; i < descriptors->total; i++ ) {
    const float* desc_ptr = (const float*)img_reader.ptr;
    CV_NEXT_SEQ_ELEM( img_reader.seq->elem_size, img_reader );
    memcpy(img_ptr, desc_ptr, length*sizeof(float));
    img_ptr += length;
  }
}
*/

// ---------------------------------------------------------------------

void ObjectDetection::FindMatches(const size_t ii, bool descmode) {

  double tt = (double)cvGetTickCount();
  if (debug>1)
    cout << "FindMatches(): Processing "
	 << ( descmode ? "descriptors" : "index") << " of "
	 << db[ii].imagename << " (" << db[ii].objectname
	 << ", " << db[ii].apriori << ")" << endl;

  if (descmode)
    FlannFindPairs(db[ii].descriptors, db[ii].ptpairs);
  else
    FlannFindPairs(db[ii].index, db[ii].ptpairs);

  db[ii].nmatches = db[ii].ptpairs.size()/2;

  if (debug>1) {
    tt = (double)cvGetTickCount() - tt;
    printf("FindMatches(): Matching time = %gms, ",
	   tt/(cvGetTickFrequency()*1000.));
    cout << "descriptor pairs: " <<  db[ii].nmatches << endl;
  }
}

// ---------------------------------------------------------------------

void ObjectDetection::FindMatches() {

  double tt = (double)cvGetTickCount();
  if (debug>1)
    cout << "FindMatches(): Processing " << dbindices.size()
	 << " joint indices" << endl;

  size_t totalpairs = 0;
  vector<jointindex>::const_iterator it;

  for (it=dbindices.begin(); it!=dbindices.end(); ++it) {

    vector<int> ptpairs;
    FlannFindPairs((*it).index, ptpairs);
    totalpairs += ptpairs.size()/2;

    for (size_t i=0; i<ptpairs.size(); i=i+2) {
      const pair<size_t, size_t> &kid = (*it).keypoint_ids[ptpairs[i+1]];
      //cout << "ptpair: " << ptpairs[i] << "," << ptpairs[i+1]
      //	   << ", kid: " << kid.first << "," << kid.second
      //     << ", keypoint_ids.size()=" << (*it).keypoint_ids.size() << endl;
      db[kid.first].ptpairs.push_back(ptpairs[i]);
      db[kid.first].ptpairs.push_back(kid.second);
      if (RecordMatchingKeypoints(ALL_KEYPOINTS))
	matchingkeypoints[ptpairs[i+1]]++;
    }
  }

  for (size_t ii=0; ii<db.size(); ii++)
    db[ii].nmatches = db[ii].ptpairs.size()/2;

  if (debug>1) {
    printf( "FindMatches(): Matching time = %gms, ",
	    tt/(cvGetTickFrequency()*1000.));
    cout << "descriptor pairs: " <<  totalpairs << endl;
  }
}

// ---------------------------------------------------------------------

bool ObjectDetection::FindHomography(const IplImage *image, dbitem &dbit) {

  dbit.survivors.clear();
  dbit.nsurvivors = 0;

  const CvPoint2D32f src_querycorners[5] = {{float((image->width)/2.0),
					     float((image->height)/2.0)},
					    {0.0,0.0}, {float(image->width),0.0},
					    {float(image->width), float(image->height)},
					    {0.0, float(image->height)}};

  // Replaced by dbit.dbitemborders:
  // const CvPoint2D32f src_dbitemcorners[5] = {{float((dbit.size.width)/2.0),
  // 					      float((dbit.size.height)/2.0)},
  // 					     {0.0,0.0}, {float(dbit.size.width),0.0},
  // 					     {float(dbit.size.width), float(dbit.size.height)},
  // 					     {0.0, float(dbit.size.height)}};

  vector<CvPoint2D32f> pt1, pt2;
  const size_t n = dbit.ptpairs.size()/2;
  size_t nn = n;

  // if (1) {
  pt1.resize(n);
  pt2.resize(n);
  for (size_t i=0; i < n; i++) {
    pt1[i] = imageKeypoints.at(dbit.ptpairs[i*2]).pt;
    pt2[i] = dbit.keypoints.at(dbit.ptpairs[i*2+1]).pt;
  }
  // } 
  // For some reason this does not work right:
  // else { 
  //   for (size_t i=0; i < n; i++) {
  //     if (imageKeypoints.at(dbit.ptpairs[i*2]).class_id<0) {
  // 	pt1.push_back(imageKeypoints.at(dbit.ptpairs[i*2]).pt);
  // 	pt2.push_back(dbit.keypoints.at(dbit.ptpairs[i*2+1]).pt);
  //     }
  //   }
  //   nn = pt1.size();
  //   cout << "FindHomography(): Pruned previous survivors: " 
  // 	 << n << "=>" << nn << endl;
  // }

  if(nn<4) {
    if (debug>1)
      cout << "FindHomography(): Less than four points remaining, "
	   << "cannot compute homography" << endl;
    return false;
  }
  
  CvMat _pt1, _pt2;
  _pt1 = cvMat(1, nn, CV_32FC2, &pt1[0]);
  _pt2 = cvMat(1, nn, CV_32FC2, &pt2[0]);

  double *h = dbit.hmatrix;
    //double h[9];
  CvMat _h = cvMat(3, 3, CV_64F, h);
  double tt = (double)cvGetTickCount();
  bool hg = cvFindHomography(&_pt1, &_pt2, &_h, CV_RANSAC, 5) ? true : false;
  tt = (double)cvGetTickCount() - tt;
  if (debug>1)
    printf("FindHomography(): cvFindHomography() took %gms\n",
	   tt/(cvGetTickFrequency()*1000.));

  if (!hg) {
    if (debug>1)
      cout << "FindHomography(): Homography not found" << endl;
    return false;
  }

  if (debug>1) {
    printf("FindHomography(): Homography matrix:\n");
    printf("  %10.4f %10.4f %10.4f\n", h[0], h[1], h[2]);
    printf("  %10.4f %10.4f %10.4f\n", h[3], h[4], h[5]);
    printf("  %10.4f %10.4f %10.4f\n", h[6], h[7], h[8]);
  }

  double determinant = cvDet(&_h);
  if (debug>1)
    cout << "FindHomography(): Determinant: " << determinant << endl;

  for (size_t i=0; i < 5; i++) {
    //const double &x = src_querycorners[i].x, &y = src_querycorners[i].y;
    //const double Z = 1./(h[6]*x + h[7]*y + h[8]);
    //const double X = (h[0]*x + h[1]*y + h[2])*Z;
    //const double Y = (h[3]*x + h[4]*y + h[5])*Z;

    double X, Y;
    CalcHomography(h, src_querycorners[i].x, src_querycorners[i].y, X, Y);

    dbit.querycorners[i] = cvPoint(cvRound(X), cvRound(Y));
  }

  bool do_sanity_checks = true;
  if (debug>1)
    cout << "FindHomography(): Homography sanity checks:" << endl;

  if (do_sanity_checks && abs(determinant) < HOMOGRAPHY_DETERMINANT_TH) {
    if (debug>1)
      cout << "FindHomography(): Determinant too close to zero" << endl;
    return false;
  }

  double *ih = dbit.ihmatrix;
  //double ih[9];
  CvMat _ih = cvMat(3, 3, CV_64F, ih);
  cvInvert(&_h, &_ih);

  if (debug>1) {
    printf("FindHomography(): Inverse homography matrix:\n");
    printf("  %10.4f %10.4f %10.4f\n", ih[0], ih[1], ih[2]);
    printf("  %10.4f %10.4f %10.4f\n", ih[3], ih[4], ih[5]);
    printf("  %10.4f %10.4f %10.4f\n", ih[6], ih[7], ih[8]);
  }

  for (size_t i=0; i < 5; i++) {
    const double &x = dbit.dbitemborders[i].x, &y = dbit.dbitemborders[i].y;
    if (x>-1 && y>-1) {
      //const double Z = 1./(ih[6]*x + ih[7]*y + ih[8]);
      //const double X = (ih[0]*x + ih[1]*y + ih[2])*Z;
      //const double Y = (ih[3]*x + ih[4]*y + ih[5])*Z;

      double X, Y;
      CalcHomography(ih, x, y, X, Y);

      dbit.dbitemcorners[i] = cvPoint(cvRound(X), cvRound(Y));
    } else {
      dbit.dbitemcorners[i] = cvPoint(-1, -1);
    }
  }    

  vector<double> angles(4);
  double minlength = 99999999999.9, maxlength = 0.0;
  for (size_t i = 1; i < 5; i++) {
    const CvPoint r1 = dbit.querycorners[i];
    const CvPoint r2 = dbit.querycorners[MAX((i+1)%5,1)];
    double len = sqrt(pow((double)(r1.x-r2.x),2)+pow((double)(r1.y-r2.y),2));
    angles.at(i-1) = atan2((double)(r1.y-r2.y),(double)(r1.x-r2.x));
    if (len>maxlength)
      maxlength = len;
    else if (len<minlength)
      minlength = len;
  }

  for (size_t i = 0; i < 4; i++) {
    double angdiff = angles.at((i+1)%4)-angles.at(i);
    //cout << "angdiff was=" << angdiff;
    if (angdiff > M_PI)
      angdiff -= M_PI;
    else if (angdiff < -M_PI)
      angdiff += 2*M_PI;
    else if (angdiff < 0.0)
      angdiff += M_PI;
    //cout << " now=" << angdiff << endl;
    if (do_sanity_checks && HOMOGRAPHY_CHECK_ANGLES && abs(angdiff-M_PI_2)>M_PI_4) {
      if (debug>1)
	cout << "FindHomography(): Angle too far from pi/2: " <<
	  angdiff << endl;
      return false;
    }
  }

  if (debug>1)
    cout << "FindHomography(): min length: " << minlength << ", max length: "
	 << maxlength << endl;
  if (do_sanity_checks && HOMOGRAPHY_MINLENGTH_TH > 0 && 
      minlength < HOMOGRAPHY_MINLENGTH_TH) {
    if (debug>1)
      cout << "FindHomography(): Minimum length too small" << endl;
    return false;
  }
  if (do_sanity_checks && HOMOGRAPHY_MAXLENGTH_TH > 0 && 
      maxlength > HOMOGRAPHY_MAXLENGTH_TH) {
    if (debug>1)
      cout << "FindHomography(): Maximum length too big" << endl;
    return false;
  }

  double total_error = 0.0;
  for (size_t i=0; i<nn; i++) {
    const double &x = pt1[i].x, &y = pt1[i].y;
    const double Z = 1./(h[6]*x + h[7]*y + h[8]);
    const double bp_error = sqrt(pow((pt2[i].x - (h[0]*x+h[1]*y+h[2])*Z), 2) +
				 pow((pt2[i].y - (h[3]*x+h[4]*y+h[5])*Z), 2));

    //cout << bp_error << " " << x << " " << y << endl;
    if (bp_error < HOMOGRAPHY_BACKPROJECTION_TH &&
	imageKeypoints.at(dbit.ptpairs[i*2]).class_id<0) {
      dbit.survivors.push_back(true);      
      imageKeypoints.at(dbit.ptpairs[i*2]).class_id = dbit.n;
    } else {
      dbit.survivors.push_back(false);
    }

    total_error += bp_error;
  }
  if (debug>1)
    cout << "FindHomography(): Back-projection error per matched keypoint "
	 << "pair: " << total_error/n << endl;

  for (int i = 0; i < (int)n; i++)

    if (dbit.survivors[i]) {
      dbit.nsurvivors++;

      if (RecordMatchingKeypoints(AFTER_HOMOGRAPHY)) {
	int res = -1;
	for (size_t j=0; j<dbindices[0].keypoint_ids.size(); j++) {
	  pair<size_t, size_t> &kid = dbindices[0].keypoint_ids[j];
	  //cout << dbit.n << "," << kid.first << ","
	  //     << dbit.ptpairs[i*2+1] << "," << kid.second << endl;
	  if (dbit.n==kid.first && dbit.ptpairs[i*2+1]==(int)kid.second) {
	    res = j;
	    break;
	  }
	}
	if (res>-1) {
	  //cout << "Adding keypoint counter for " << res << endl;
	  matchingkeypoints[res]++;
	}
      }

    }

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::CalcHomography(const double *h, double x, double y, 
				     double &X, double &Y) {

  double Z = 1./(h[6]*x + h[7]*y + h[8]);
  X = (h[0]*x + h[1]*y + h[2])*Z;
  Y = (h[3]*x + h[4]*y + h[5])*Z;
  
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::ShowKeypoints(const IplImage *image, IplImage** maskptr) {

  IplImage*& mask = *maskptr;

  IplImage *img_color = cvCreateImage(cvGetSize(image), 8, 3);
  cvCvtColor(image, img_color, CV_GRAY2BGR);

  for (size_t i = 0; i < imageKeypoints.size(); i++) {
    //CvSURFPoint* r = (CvSURFPoint*)cvGetSeqElem(imageKeypoints, i);
    cv::KeyPoint &kp = imageKeypoints.at(i);
    CvPoint center;
    int radius;
    bool accepted = (!FilterLoaded() || filter->keypoint_ok[i]);

    center.x = cvRound(kp.pt.x);
    center.y = cvRound(kp.pt.y);
    radius   = cvRound(kp.size*1.2/9.*2);
    cvCircle(img_color, center, radius,
	     accepted ? colors[0] : colors[2],
	     1, 8, 0);
  }
  cvNamedWindow("Detected keypoints", 1);
  cvShowImage("Detected keypoints", img_color);
  if (&mask && mask) {
    cvNamedWindow("Used mask", 1);
    cvShowImage("Used mask", mask);
  }

  char c = (char)cvWaitKey(0);

  cvDestroyWindow("Detected keypoints");
  if (&mask && mask)
    cvDestroyWindow("Used mask");

  cvReleaseImage(&img_color);

  if (c == 'q' || c == 'Q' )
    return false;
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::SaveData_Text(const string &fn, const vector<cv::KeyPoint> &keypoints,
				    const cv::Mat &descriptors, bool usefilter,
				    bool printsize) {

  CheckKeypointsAndDescriptors();

  if (!usefilter) {
    ifstream filetest(fn.c_str());
    if (filetest) {
      if (debug)
	cout << "SaveData_Text(): File " << fn
	     << " already exists, skipping." << endl;
      return true;
    }
  }

  size_t tot = keypoints.size();
  if (usefilter) {
    tot = 0;
    vector<bool>::const_iterator it;
    for (it=filter->keypoint_ok.begin(); it!=filter->keypoint_ok.end(); ++it)
      if (*it)
	tot++;
  }

  ofstream outfile(fn.c_str());
  if (printsize)
    outfile << tot << endl;

  size_t ii=0;
  for (size_t i = 0; i < keypoints.size(); i++) {
    //CvSURFPoint* r = (CvSURFPoint*)cvGetSeqElem(keypoints, i);
    //float* d = (float*)cvGetSeqElem(descriptors, i);
    const cv::KeyPoint &r = keypoints.at(i);
    const cv::Mat &d = descriptors.row(i);

    if (!usefilter || filter->keypoint_ok[i]) {
      outfile << r.pt.x << " " << r.pt.y << " " << r.size << " " << r.angle
	      << " " << r.response << " " << r.octave << " " << r.class_id;
      for (int j=0; j<d.cols; j++)
	outfile << " " << d.at<float>(j);
      outfile << endl;
      ii++;
    }
  }
  outfile.close();

  if (tot != ii) {
    cerr << "ERROR: SaveData_Text(): Keypoint count mismatch, "
	 << "this shouldn't happen!" << endl;
    exit (-1);
  }
  if (debug)
    cout << "SaveData_Text(): " << tot << "/" << keypoints.size()
	 << " keypoints and descriptors saved as text to: "
	 << fn << endl;
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::SaveData_Bin(const string &fn, const vector<cv::KeyPoint> &keypoints,
				   const cv::Mat &descriptors, bool usefilter) {

  CheckKeypointsAndDescriptors();

  if (!usefilter) {
    ifstream filetest(fn.c_str());
    if (filetest) {
      if (debug)
	cout << "SaveData_Bin(): File " << fn
	     << " already exists, skipping." << endl;
      return true;
    }
  }

  size_t tot = keypoints.size();
  if (usefilter) {
    tot = 0;
    vector<bool>::const_iterator it;
    for (it=filter->keypoint_ok.begin(); it!=filter->keypoint_ok.end(); ++it)
      if (*it)
	tot++;
  }

  ofstream outfile(fn.c_str(), ios::out | ios::binary);
  outfile.write((char*)&tot, sizeof(int));

  size_t ii = 0;
  for (size_t i = 0; i < keypoints.size(); i++) {
    //CvSURFPoint* r = (CvSURFPoint*)cvGetSeqElem(keypoints, i);
    //char* d = (char*)cvGetSeqElem(descriptors, i);
    const cv::KeyPoint &r = keypoints.at(i);
    const cv::Mat &d = descriptors.row(i);

    if (!usefilter || filter->keypoint_ok[i]) {
      outfile.write((char*)&r, sizeof(cv::KeyPoint));
      for (int j=0; j<d.cols; j++) {
	outfile.write((char*)&d.at<float>(0,j), sizeof(float));
      }
      ii++;
    }
  }
  outfile.close();

  if (tot != ii) {
    cerr << "ERROR: SaveData_Bin(): Keypoint count mismatch, "
	 << "this shouldn't happen!" << endl;
    exit (-1);
  }

  if (debug)
    cout << "SaveData_Bin(): " << tot << "/" << keypoints.size()
	 << " keypoints and descriptors of length " << descriptors.cols 
	 << " saved as binary to: "
	 << fn << endl;
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::CreateAndSaveIndex(vector<cv::KeyPoint> &keypoints,
					 cv::Mat &descriptors,
					 const string &fn,
					 const string &descfn,
					 bool usefilter) {

  double tt = (double)cvGetTickCount();

  if (debug)
    cout << "CreateAndSaveIndex(): Starting" << endl;

  if (descriptors.empty())
    descriptors = dbindex.descriptors;
  if (keypoints.empty())
    keypoints = dbindex.keypoints;

  CheckKeypointsAndDescriptors(keypoints, descriptors);

  cv::Mat m_image(0, descriptors.cols, CV_32F);
  
  for(int i = 0; i < descriptors.rows; i++ ) {  
    if (!usefilter || filter->keypoint_ok[i]) {
      m_image.push_back(descriptors.row(i));
    }
  }

  // for(int i = 0; i < descriptors->total; i++ ) {
  //   const float* descriptor = (const float*)img_reader.ptr;
  //   CV_NEXT_SEQ_ELEM( img_reader.seq->elem_size, img_reader );
  //   if (!usefilter || filter->keypoint_ok[i]) {
  //     memcpy(img_ptr, descriptor, length*sizeof(float));
  //     img_ptr += length;
  //   }
  // }

  if (descfn != "") {
    if (debug)
      cout << "CreateAndSaveIndex(): Saving keypoint and descriptor data "
	   << "as text to: " << descfn << endl;
    SaveData_Text(descfn, keypoints, descriptors, false, false);
  }

  CvMat m = m_image;
  string xmlfn = fn+".xml";
  CvFileStorage *fs = cvOpenFileStorage(xmlfn.c_str(), 0, CV_STORAGE_WRITE);
  cvWrite(fs, "descriptors", &m, cvAttrList(0,0));
  cvReleaseFileStorage(&fs);
  if (debug)
    cout << "CreateAndSaveIndex(): Descriptors (" << m_image.rows << ", " << m_image.cols << ") saved to: " << xmlfn << endl;

  int flannsize = 0, flannveclen = 0;
  if (flann_index_mode == KDTREE) {
    FlannIndexType flann_index(m_image,
			       cvflann::KDTreeIndexParams(FLANN_NTREES));
    flann_index.save(fn);
    if (debug)
      cout << "CreateAndSaveIndex(): Saved a KDTREE FLANN index to: "
	   << fn << endl;
    flannsize = flann_index.size();
    flannveclen = flann_index.veclen();

  } else {
    FlannIndexType flann_index(m_image, cvflann::KMeansIndexParams());
    flann_index.save(fn);
      if (debug)
	cout << "CreateAndSaveIndex(): Saved a KMEANS FLANN index to: "
	     << fn << endl;
    flannsize = flann_index.size();
    flannveclen = flann_index.veclen();
  }

  if (debug) {
    tt = (double)cvGetTickCount() - tt;
    printf("CreateAndSaveIndex(): "
	   "Index generation and saving done, time elapsed = %gms\n",
	   tt/(cvGetTickFrequency()*1000.));
    cout << "CreateAndSaveIndex(): Built index of size=" << flannsize
	 << ", veclen=" << flannveclen << endl;
  }

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::CreateAndSaveIndex(const string &fn,
					 const string &descfn,
					 bool usefilter) {
  return CreateAndSaveIndex(dbindex.keypoints, dbindex.descriptors,
			    fn, descfn, usefilter);
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadDbItems(vector<vector<string> >& list,
				  const string &fn, string &indexfn,
				  string &dsuffix, string &path) {

  if (debug>1)
    cout << "LoadDbItems(): Starting" << endl;
  ifstream listfile(fn.c_str());
  if (!listfile) {
    cerr << "ERROR: LoadDbItems(): File " << fn << " not found" << endl;
    return false;
  }

  string firstline;
  getline(listfile, firstline);
  istringstream iss(firstline);
  vector<string> items;
  copy(istream_iterator<string>(iss),
       istream_iterator<string>(),
       back_inserter<vector<string> >(items));

  //  if (items.size()<2 || items.size()>3) {
  if (items.size() != 4) {
    cerr << "ERROR: LoadDbItems(): Unable to parse first line of "
	 << fn << endl;
    return false;
  }

  int listcount = atoi(items[0].c_str());
  if (indexfn == "")
    indexfn = items[1];
  if (items[2] != "-")
    dsuffix = items[2];
  path = items[3];

  // int listcount;
  // listfile >> listcount;
  // string listindex;
  // listfile >> listindex;
  // if (indexfn == "")
  //   indexfn = listindex;

  for (int i = 0; i < listcount; i++) {
    vector<string> items;
    string temp;
    for (int j = 0; j < 4; j++) {
      listfile >> temp;
      items.push_back(temp);
    }
    list.push_back(items);
  }

  if (debug>1)
    cout << "LoadDbItems(): Finished ok, indexfn=" << indexfn
	 << ", dsuffix=" << dsuffix << ", path=" << path << endl;

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadDbAprioris(vector<double>& aprioris,
				     string& fn) {

  ifstream listfile(fn.c_str());
  if (!listfile) {
    if (debug)
      cout << "LoadDbAprioris(): No a priori file " << fn
	   << " found" << endl;
    return false;
  }

  int listcount;
  listfile >> listcount;
  for (int i = 0; i < listcount; i++) {
    string temp;
    listfile >> temp;
    aprioris.push_back(atof(temp.c_str()));
  }
  if (debug)
  cout << "LoadDbAprioris(): Found " << listcount
       << " a priori values in " << fn << endl;
  return true;
}

// ---------------------------------------------------------------------

void ObjectDetection::LoadDb(const string &listfn, string indexfn) {

  vector<vector<string> > db_fns;
  string descsuffix = "";
  string path = "";
  if (!LoadDbItems(db_fns, listfn, indexfn, descsuffix, path)) {
    cerr << "ERROR: Cannot load database file " << listfn << endl;
    exit(-1);
  }

  if (debug)
    cout << "LoadDb(): Previous database size " << db.size()
	 << ", loading " << db_fns.size() << " new entries from "
	 << listfn << " ..." << endl;

  jointindex dbidx;
  if (indexfn != "" && indexfn != "noindex")
    DbIndicesInUse(true);

  vector<double> db_aprioris;
  if (!DbIndicesInUse()) {
    string aprfn = listfn + ".apr";
    LoadDbAprioris(db_aprioris, aprfn);
    if (db_aprioris.size() && db_aprioris.size()!=db_fns.size()) {
      cerr << "ERROR: Size mismatch in a priori file " << aprfn << " : "
	   << db_aprioris.size() << "!=" << db_fns.size() << endl;
      db_aprioris.clear();
    }
  }

  double tt = (double)cvGetTickCount();

  for (size_t ii=0; ii<db_fns.size(); ii++) {

    vector<string> &items = db_fns[ii];

    dbitem loaded;

    loaded.n           = db.size();

    size_t lastslash = items[0].rfind("/");
    if (lastslash == string::npos) {
      loaded.imagename   = items[0];
      loaded.pathname    = path + "/" + items[0];
    } else {
      loaded.imagename   = items[0].substr(lastslash+1);
      loaded.pathname    = items[0];
    }
    //cout << "imagename=" << loaded.imagename << ", pathname=" << loaded.pathname << endl;

    loaded.objectname  = items[1];

    loaded.size = cvSize(atoi(items[2].c_str()),
			 atoi(items[3].c_str()));

    if (preserveorder)
      loaded.apriori = (double)db_fns.size()-(double)ii;
    else if (db_aprioris.size() && db_aprioris.size()==db_fns.size())
      loaded.apriori = db_aprioris[ii];
    else
      loaded.apriori = 1.0/(double)db_fns.size();

    // loaded.keypoints   = cvCreateSeq(0, sizeof(CvSeq),
    // 				     sizeof(CvSURFPoint), storage);
    //loaded.descriptors = cvCreateSeq(0, sizeof(CvSeq),
    //			     64*CV_ELEM_SIZE(CV_32F), storage);
    loaded.descriptors.create(0, UsedDescriptorSize(), CV_32F);

    loaded.homography  = false;
    loaded.nsurvivors  = 0;
    loaded.nmatches    = 0;

    loaded.gaze = cvPoint2D32f(-1.0, -1.0);

    if (usebinaryfiles) {
      loaded.descname  = loaded.pathname + descsuffix + ".bin";
      LoadData_Bin(loaded);
    } else {
      loaded.descname  = loaded.pathname + descsuffix + ".surf";
      LoadData_Text(loaded);
    }

    if (DbIndicesInUse()) {
      size_t ii = db.size();
      if (debug>1)
	cout << "LoadDb(): Filling keypoint_ids, ii=" << ii
	     << ", nkps=" << loaded.nkps << endl;
      for (size_t jj = 0; (int)jj < loaded.nkps; jj++)
	dbidx.keypoint_ids.push_back(make_pair(ii,jj));

    } else {
      loaded.indexname = loaded.pathname + descsuffix + ".idx";
      loaded.index = LoadIndex(loaded.indexname);
    }

    if (useborders)
      LoadBorders(loaded);
    else {
      loaded.dbitemborders[0] = cvPoint((loaded.size.width)/2,
					(loaded.size.height)/2);
      loaded.dbitemborders[1] = cvPoint(0, 0);
      loaded.dbitemborders[2] = cvPoint(loaded.size.width, 0);
      loaded.dbitemborders[3] = cvPoint(loaded.size.width, 
					loaded.size.height);
      loaded.dbitemborders[4] = cvPoint(0, loaded.size.height);
    }

    db.push_back(loaded);
  }

  if (DbIndicesInUse()) {
    dbidx.index = LoadIndex(indexfn);
    dbindices.push_back(dbidx);
  }

  if (preserveorder) {
    double sum = 0.0;
    for (size_t ii=0; ii<db.size(); ii++)
      sum += db[ii].apriori;
    for (size_t ii=0; ii<db.size(); ii++)
      db[ii].apriori /= sum;
  }

  if (debug) {
    tt = (double)cvGetTickCount() - tt;
    printf("LoadDb(): Total time = %gms, ", tt/(cvGetTickFrequency()*1000.));
    cout << "database size now " << db.size() << endl;
  }

}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadDbs(const string &listfns, const string &indexfns) {
  bool ok = true;
  size_t listc = listfns.find(",");
  size_t indexc = indexfns.find(",");
  if (listc == string::npos && indexc == string::npos) {
    LoadDb(listfns, indexfns);
  } else {

#ifdef USE_BOOST_ALGORITHM_STRING

    vector<string> listparts;
    split(listparts, listfns, is_any_of(","));
    vector<string> indexparts;
    if (indexfns != "")
      split(indexparts, indexfns, is_any_of(","));

    ok = LoadDbs(listparts, indexparts);

    if (debug && DbIndicesInUse()) {
      unsigned int kptotal = 0;
      vector<jointindex>::const_iterator it;
      for (it=dbindices.begin(); it!=dbindices.end(); ++it)
	kptotal += (*it).index->size();
      cout << "LoadDbs(): Using " << dbindices.size()
	   << " indices with a total of " << kptotal << " keypoints." << endl;
    }

#else

    cerr << "ERROR: LoadDbs(): Using multiple databases requires boost,"
	 << " quitting." << endl;
    return false;

#endif

  }
  return ok;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadDbs(const vector<string> &listfns,
			      const vector<string> &indexfns) {

  if (indexfns.size() && (listfns.size() != indexfns.size())) {
    cerr << "LoadDbs(): The number of databases must match the number "
	 << "of indices." << endl;
    return false;
  }

  string empty = "";
  for (size_t ii=0; ii<listfns.size(); ii++)
    LoadDb(listfns[ii], (indexfns.size() ? indexfns[ii] : empty));

  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadDbsList(const string &fn) {

  if (debug>1)
    cout << "LoadDbsList(): Starting" << endl;
  ifstream listfile(fn.c_str());
  if (!listfile) {
    cerr << "ERROR: LoadDbsList(): File " << fn << " not found" << endl;
    return false;
  }

  int listcount;
  listfile >> listcount;

  vector<string> fns;
  for (int i = 0; i < listcount; i++) {
    string temp;
    listfile >> temp;
    fns.push_back(temp);
  }

  if (debug>1)
    cout << "LoadDbsList(): Found " << fns.size() << " entries in "
	 << fn << " ..." << endl;

  vector<string> empty;
  LoadDbs(fns, empty);

  return true;
}

// ---------------------------------------------------------------------

void ObjectDetection::SaveDbHeader(size_t n, const string &idxname) {
  if (outdbname == "")
    return;

  if (debug>1)
    cout << "SaveDbHeader(): Starting" << endl;

  ofstream outfile(outdbname.c_str());
  outfile << n << " " << idxname;
  outfile << " " <<  (FilterInUse() ? "."+FilterString() : "-");
  char currentpath[1024];
#ifndef _MSC_VER
  if (!getcwd(currentpath, 1024))
#else
  if (!_getcwd(currentpath, 1024))
#endif // _MSC_VER
    cerr << "ERROR: SaveDbHeader(): getcwd() returned NULL" << endl;
  outfile << " " << currentpath << endl;
  outfile.close();
}

// ---------------------------------------------------------------------

void ObjectDetection::PossiblySaveDbItem(const IplImage *image,
					 const string &fn) {
  if (outdbname == "")
    return;

  if (debug>1)
    cout << "PossiblySaveDbItem(): Starting" << endl;

  ofstream outfile(outdbname.c_str(), ios_base::app);
  outfile << fn << " " << fn << " "
	  << image->width << " " << image->height
	  << endl;
  outfile.close();
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadData_Text(dbitem &item) {

  if (debug>1)
    cout << "LoadData_Text(): Loading keypoints "
	 << (DbIndicesInUse() ? "and descriptors " : "")
	 << "from text file: " << item.descname << endl;

  ifstream infile(item.descname.c_str());
  if (!infile) {
    cerr << "ERROR: LoadData_Text(): File " << item.descname
	 << " not found" << endl;
    return false;
  }

  infile >> item.nkps;
  for (int i = 0; i < item.nkps; i++) {

    cv::KeyPoint kp;
    infile >> kp.pt.x >> kp.pt.y >> kp.size >> kp.angle
	   >> kp.response >> kp.octave >> kp.class_id;
    item.keypoints.push_back(kp);

    /*
    CvSURFPoint s;

    infile >> s.pt.x;
    infile >> s.pt.y;
    infile >> s.laplacian;
    infile >> s.size;
    infile >> s.dir;
    infile >> s.hessian;

    cvSeqPush(item.keypoints, &s);
    */

    cv::Mat desc(1, UsedDescriptorSize(), CV_32F);
    for (int j=0; j<desc.cols; j++)
      infile >> desc.at<float>(0,j);

    if (DbIndicesInUse())
      item.descriptors.push_back(desc); // descriptors loaded from xml.
    //cvSeqPush(item.descriptors, &desc); // descriptors loaded from xml.
  }
  return true;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadData_Bin(dbitem &item) {

  ifstream infile(item.descname.c_str(), ios::in | ios::binary);
  if (!infile) {
    cerr << "ERROR: LoadData_Bin(): File " << item.descname
	 << " not found" << endl;
    return false;
  }

  int tot;
  infile.read((char*)&tot, sizeof(int));
  item.nkps = tot;

  if (debug>1)
    cout << "LoadData_Bin(): Loading " << tot << " keypoints "
	 << (DbIndicesInUse() ? "and descriptors " : "")
	 << "from binary file: " << item.descname << endl;

  for (int i = 0; i < item.nkps; i++) {
    // CvSURFPoint s;
    // infile.read((char*)&s, sizeof(CvSURFPoint));
    // cvSeqPush(item.keypoints, &s);
    cv::KeyPoint kp;
    infile.read((char*)&kp, sizeof(cv::KeyPoint));
    item.keypoints.push_back(kp);

    cv::Mat desc(1, UsedDescriptorSize(), CV_32F);
    for (int j=0; j<desc.cols; j++)
      infile.read((char*)&desc.at<float>(0,j), sizeof(float));

    if (DbIndicesInUse())
      item.descriptors.push_back(desc); // descriptors loaded from xml.
  }
  return true;
}

// ---------------------------------------------------------------------

FlannIndexType* ObjectDetection::LoadIndex(string &indexname, bool dbg) {

  string xmlfn = indexname+".xml";
  if (debug>1 || dbg)
    cout << "LoadIndex(): Loading descriptors from XML file " << xmlfn
	 << endl;

  CvFileStorage * fs = cvOpenFileStorage(xmlfn.c_str(), 0, CV_STORAGE_READ);
  const CvMat *m = (const CvMat *)cvReadByName(fs, 0, "descriptors", 0);
  cvReleaseFileStorage(&fs);
  cv::Mat m_image(m);

  double minval = 0.0, maxval=0.0;
  minMaxLoc(m_image, &minval, &maxval);
  if (debug>1 || dbg)
    cout << "LoadIndex(): descriptors minval=" << minval 
	 << ", maxval=" << maxval << endl;
  if (abs(minval) < 0.01 && abs(maxval) < 0.01) {
    if (debug || dbg)
      cout << "LoadIndex(): value of all descriptors is zero, " 
	   << "returning NULL index" << endl;
    return NULL;
  }

  if (debug || dbg) {
    cv::Size size = m_image.size();
    cout << "LoadIndex(): Loading an index from file " << indexname
	 << " for " <<  size.height << " descriptors" << endl;
  }
  FlannIndexType *flann_index = new
    FlannIndexType(m_image, cvflann::SavedIndexParams(indexname));

  if (debug>1 || dbg) {
    cout << "LoadIndex(): Loaded index of size=" << flann_index->size()
	 << ", veclen=" << flann_index->veclen() << endl;
  }
  const int MIN_INDEX_SIZE = 10;
  if (flann_index->size() < MIN_INDEX_SIZE) {
    if (debug || dbg)
      cout << "LoadIndex(): Index size has to be at least " << MIN_INDEX_SIZE
	   << ", returning NULL index" << endl;
    return NULL;
  }

  return flann_index;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadBorders(dbitem &item) {

  string& fn = item.pathname;
  size_t fromloc = fn.find(borderreplace.first);
  if (fromloc == string::npos) {
    cerr << "LoadBorders(): ERROR: Substring \""<< borderreplace.first << 
      "\" not found in " << fn << endl;
    return false;
  }
  string borderfn = fn;
  borderfn.replace(fromloc, borderreplace.first.size(), borderreplace.second);
    
  vector<CvPoint> ppts;
  bool ok = LoadPolygon(ppts, borderfn);

  if (ppts.size() != 4) {
    cerr << "LoadBorders(): ERROR: Unsupported number of polygon vertices: " 
	 << ppts.size() << "!= 4" << endl;
    return false;
  }

  item.dbitemborders[0] = cvPoint(-1,-1);
  for (size_t i = 0; i < ppts.size(); i++)
    item.dbitemborders[i+1] = ppts[i];  

  return ok;
}

// ---------------------------------------------------------------------

bool ObjectDetection::LoadImgList(vector<vector<string> >& list,
				  const string& fn) {

  cout << "LoadImgList(): Starting" << endl;

#ifdef USE_BOOST_ALGORITHM_STRING

  ifstream listfile(fn.c_str());
  if (!listfile) {
    cerr << "ERROR: LoadImgList(): File " << fn << " not found" << endl;
    return false;
  }

  int listcount;
  listfile >> listcount;
  string foo;
  getline(listfile, foo);
  for (int i = 0; i < listcount; i++) {
    string line;
    getline(listfile, line);
    vector<string> parts;
    split(parts, line, is_any_of(" "));
    list.push_back(parts);
  }
  cout << "LoadImgList(): Found " << listcount << " analyse images in "
       << fn << endl;
  return true;

#else

  cerr << "LoadImgList(): Boost required, quitting." << endl;
  return false;

#endif

}

// ---------------------------------------------------------------------

bool ObjectDetection::RecordMatchingKeypoints(const record_keypoints_type
					      &mkm) {
  if (matchingkeypointsname == "")
    return false;
  if (mkm == ANY)
    return true;
  if (mkm == matchingkeypointsmode)
    return true;

  return false;
}

// ---------------------------------------------------------------------

void ObjectDetection::SetFilter(const string &f, const string &s, size_t t) {
  size_t svm = f.find("svm");
  size_t rnd = f.find("random");
  size_t hes = f.find("hessian");

  if (svm != string::npos) {
#ifdef USE_LIBSVM
    filter = new SVMFilter();
#else
    cerr << "ERROR: SetFilter(): LIBSVM disabled, cannot "
	 << "use SVM filters" << endl;
    exit(-1);
#endif
  } else if (rnd != string::npos)
    filter = new RandomFilter();
  else if (hes != string::npos)
    filter = new HessianFilter();
  else
    filter = new ClusterFilter();

  filter->SetFilter(f, t);
  if (!s.empty())
    filter->SetNameString(s);
}

// ---------------------------------------------------------------------

bool ObjectDetection::FileExists(const string &filename) {
  ifstream ifile(filename.c_str());
  if (ifile.good()) {
    ifile.close();
    return true;
  } else {
    ifile.close();
    return false;
  }
  // return ifile; // was this until 11-11-2013
}

// ---------------------------------------------------------------------

bool ObjectDetection::IsAffirmative(const char *s) {
  if (!s || !*s) return false;
  if (strlen(s)==1 && strchr("YyTt1", *s)) return true;
#ifndef _MSC_VER
  if (!strcasecmp(s, "yes") || !strcasecmp(s, "true") ||
      !strcasecmp(s, "on")) return true;
#else
  if (!_strnicmp(s, "yes", 3) || !_strnicmp(s, "true", 4) ||
      !_strnicmp(s, "on", 2)) return true;  
#endif // _MSC_VER
  return false;
}

// ---------------------------------------------------------------------
