// -*- C++ -*-  $Id: ObjectDetection.h,v 1.130 2014/01/16 15:17:27 jorma Exp $
//
// Copyright 2009-2014 PicSOM Development Group <picsom@cis.hut.fi>
// Aalto University School of Science
// Department of Information and Computer Science
// P.O.BOX 15400, FI-00076 Aalto, FINLAND
//

/*! \mainpage ObjectDetection documentation
 * \image html example1-small.png
 * $Id: ObjectDetection.h,v 1.130 2014/01/16 15:17:27 jorma Exp $
 *
 * \author PicSOM Development Group <picsom@cis.hut.fi>\n
 * Aalto University School of Science\n
 * Department of Information and Computer Science\n
 * P.O.BOX 15400, FI-00076 Aalto, FINLAND
 *
 * \date 2009-2013
 *
 * @see \ref usage
 * @see ObjectDetection
 * @see od_defs.h
 * @see KeypointFilter
 * @see SMI
 */

#ifndef _ObjectDetection_h_
#define _ObjectDetection_h_

#if defined(__MINGW32__) || defined(_MSC_VER)
#define __WINDOWS__
#endif

//#include <omp.h>

// pre-2.2 OpenCV:
//#include <opencv/cv.h>
//#include <opencv/highgui.h>

// 2.2 and beyond OpenCV:
#include <opencv2/core/core.hpp>
#include <opencv2/imgproc/imgproc_c.h>
#include <opencv2/imgproc/imgproc.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/flann/flann.hpp>
#include <opencv2/features2d/features2d.hpp>
#include <opencv2/calib3d/calib3d.hpp>

// 2.4.2 (at least) needs this for SIFT and SURF: 
#include <opencv2/nonfree/nonfree.hpp>

// Needed for GPU SURF: 
#include "opencv2/nonfree/gpu.hpp"

// Needed for heat map colormaps:
#include <opencv2/contrib/contrib.hpp>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#ifndef _MSC_VER
#include <sys/time.h>
#else
#include <direct.h>
#include <time.h>
#define round(val) (floor(val + 0.5))
#include <cmath>
#endif // _MSC_VER

#ifndef __WINDOWS__
#include <nanohttp/nanohttp-logging.h>
#include <libcsoap/soap-server.h>
#endif // __WINDOWS__

#include <iostream>
#include <string>
#include <fstream>
#include <vector>
#include <iterator>
#include <algorithm>

#include "od_defs.h"
#include "KeypointFilter.h"
#include "SMI.h"

#ifndef SIZE_T_MAX
#define SIZE_T_MAX (size_t)-1
#endif  // SIZE_T_MAX

using namespace std;

typedef enum {DET_NONE,DET_SURF,DET_GPUSURF,DET_MSER,DET_SIFT,DET_FAST,
	      DET_STAR,DET_ORB,DET_GTFF,DET_HARRIS} detector_type;
typedef enum {DES_SURF,DES_GPUSURF,DES_SIFT} desc_type;
typedef enum {NO_SAVING,SEPARATE_INDICES,JOINT_INDEX,MATLAB_MODE} savetype;
typedef enum {KDTREE,KMEANS} flann_index_type;
typedef enum {NONE,ALL_KEYPOINTS,AFTER_HOMOGRAPHY,ANY} record_keypoints_type;
typedef enum {MASK_NONE,MASK_FILE,MASK_POLYGON} mask_type;

// This used to work before OpenCV 2.3.1:
//typedef cv::flann::Index FlannIndexType;
// This is deprecated in OpenCV 2.3.1:
//typedef cv::flann::Index_<float> FlannIndexType;
// This seems to work in OpenCV 2.3.1:
typedef cv::flann::GenericIndex<cvflann::L2<float> > FlannIndexType;

/// A struct containing everything related to a image/object in the database.
struct dbitem {
  /// The ordinal value of this #dbitem.
  size_t n;
  /// The name of the image file corresponding to this #dbitem.
  string imagename;
  /// The name of the image file with full path corresponding to this #dbitem.
  string pathname;
  /// An alternative name for this #dbitem. May be equal to #imagename or
  /// possibly something more descriptive.
  string objectname;
  /// The name of the data file containing the #keypoints and #descriptors
  /// corresponding to this #dbitem.
  string descname;
  /// The name of the data file containing the #index corresponding to
  /// this #dbitem.
  string indexname;
  /// The total number of #keypoints.
  int nkps;
  /// The size (width and height) of the image file corresponding to this
  /// #dbitem.
  CvSize size;
  /// The apriori probability of this image. Can be used to sort #dbitem's
  /// for ObjectDetection::ProcessDatabaseUntilGoodMatch().
  double apriori;
  /// The number of matched keypoints after the pairwise matching stage.
  size_t nmatches;
  /// The number of matched keypoints surviving after the homography test.
  size_t nsurvivors;
  /// Detected keypoints
  //CvSeq *keypoints;
  vector<cv::KeyPoint> keypoints;
  /// SURF descriptors
  //CvSeq *descriptors;
  cv::Mat descriptors;
  /// Pointer to a FLANN index created for #descriptors of only this #dbitem.
  FlannIndexType *index;
  /// Pairs of matching keypoints between the query ObjectDetection::image
  /// and this #dbitem.
  vector<int> ptpairs;
  /// The survival status of matched keypoints after the homography test.
  vector<bool> survivors;
  /// Whether a homography was found for the matched keypoints by
  /// ObjectDetection::FindHomography().
  bool homography;
  /// A vectorization of the homography matrix.
  double hmatrix[9];
  /// A vectorization of the inverse homography matrix.
  double ihmatrix[9];
  /// A mapping from the center point and four corners of the query image
  /// to this #dbitem. Used mainly for visualizing matching results.
  CvPoint querycorners[5];
  /// The the center point and four corners of this #dbitem.
  CvPoint dbitemborders[5];
  /// A mapping from dbitemborders of this #dbitem
  /// to the query image. Used mainly for visualizing matching results.
  CvPoint dbitemcorners[5];
  /// This was needed at some point in the SOAP mode.
  string invcorners;
  /// The location of the current gaze on this #dbitem.
  CvPoint2D32f gaze;
  ///The gaze history on this #dbitem.
  vector<pair<size_t, CvPoint2D32f> > gaze_history;
};

/// Sorting #dbitem's by their original order
struct dbitem_originalorder {
  /// Sorting function based on \a dbitem.n
  bool operator() (const dbitem &a, const dbitem &b) {
    return a.n < b.n;
  }
};

/// Sorting #dbitem's by the number of matched keypoints
struct dbitem_matchsort {
  /// Sorting function based on \a dbitem.nmatches
  bool operator() (const dbitem &a, const dbitem &b) {
    return a.nmatches > b.nmatches;
  }
};

/// Sorting #dbitem's by the number of keypoints surviving the homography test
struct dbitem_survivorsort {
    /// Sorting function based on \a dbitem.nsurvivors
  bool operator() (const dbitem &a, const dbitem &b) {
    return a.nsurvivors > b.nsurvivors;
  }
};

/// Sorting #dbitem's by their a priori probabilities
struct dbitem_apriorisort {
    /// Sorting function based on \a dbitem.apriori
  bool operator() (const dbitem &a, const dbitem &b) {
    return a.apriori > b.apriori;
  }
};

/// A joint index for several objects in the database.
struct jointindex {
  /// Descriptors of all objects sharing this index. Used only while
  /// creating the index, as a loaded index contains all corresponding
  /// descriptors.
  //CvSeq *descriptors;
  cv::Mat descriptors;
  /// Corresponding keypoints. Used only when creating the index as
  /// optional output.
  //CvSeq *keypoints;
  vector<cv::KeyPoint> keypoints;
  /// A pointer to the actual FLANN index
  FlannIndexType *index;
  /// The IDs of all keypoints as pairs <n_dbitem, n_keypoint>.
  vector<pair<size_t, size_t> > keypoint_ids;
};

/// A struct used to draw thumbnails of the database images.
/// @see ShowDbResults().
struct dbresultimage {
  /// The actual image data.
  IplImage *img;
  /// The scaling used for each database image when generating the image.
  double scale;
  /// Width of each database image after the scaling.
  size_t width;
  /// Width of each database image after the scaling.
  size_t height;
};

/// A struct replacing (and containing) the old descriptor typedef.
struct descriptor_type {
  /// The used descriptor, currently SIFT or SURF.
  desc_type name;
  /// Whether to square-root the descriptor (i.e. RootSIFT). 
  bool root;
  /// Whether to L2 normalize the descriptor (after possible square-rooting).
  bool norm;
};

/// The main class for the the object detection functionality.
class ObjectDetection {

 public:
  /// Returns the version of this software.
  static const string& Version();

  /// Constructor. Does very little, initializations are done in Initialize().
  ObjectDetection();

  /// Performs all initializations. Has to be called immediately after
  /// an ObjectDetection object has been created.
  bool Initialize(bool quiet=false);

  /// Loads and interprets file \a rcfilename and sets found parameter values.
  bool LoadRcFile();

  /// Sets almost any variable \a keystr to value \a valstr. Includes SetOption().
  bool Interpret(const string& keystr, const string& valstr);

  /// Splits \a keqv to \a keystr and \a valstr and calls Interpret(keystr,valstr).
  bool Interpret(const char *keqv);

  /// Sets a command-line variable \a keystr to value \a valstr.
  bool SetOption(const string& keystr, const string& valstr);

  /// Splits \a keqv to \a keystr and \a valstr and calls SetOption(keystr,valstr).
  bool SetOption(const char *keqv);

  /// Sets #debug
  void SetDebug(size_t d) {
    debug = d;
    if (filter != NULL)
      filter->SetDebug(d);
  }

  /// Needed by static SOAP functions. \return debug
  size_t GetDebug() { return debug; }

  /// Sets #scale
  void SetScale(double s) { scale = s; }
  
  /// Sets #scales and #scale.
  void SetScale(const string &sstr);  

  /// Returns the real scale in use.
  double GetScale() {
    if (scale>0.0)
      return scale;
    return DEFAULT_SCALING_FACTOR;
  }

  /// Sets longest edge after resizing the images
  void SetNewsize(int ns) { newsize = ns; }

  /// Sets #applyblur
  void SetBlur(bool b) { applyblur = b; }

  /// Sets #outimgname
  void SetOutImgName(const string &o) { outimgname = o; }

  /// Sets #outvidname
  void SetOutVidName(const string &o) { outvidname = o; }

  /// Sets #outdbname
  void SetOutDbName(const string &o) { outdbname = o; }

  /// Sets #showresults
  void SetShowResults(bool s) { showresults = s; }

  /// Sets #showonlineresults
  void SetShowOnlineResults(bool s) { showonlineresults = s; }

  /// Sets #showonlinedbresults
  void SetShowOnlineDbResults(bool s) { showonlinedbresults = s; }

  /// Returns true if either showonlineresults or showonlinedbresults.
  bool GetShowOnlineAnyResults() { 
    return (showonlineresults || showonlinedbresults); 
  }

  /// Sets #flann_index_mode
  void SetFlannIndexMode(flann_index_type f) { flann_index_mode = f; }

  /// Sets #exhaustivesearch
  void SetExhaustiveSearch(bool e) { exhaustivesearch = e; }

  /// Sets #analyseonlybest
  void SetAnalyseOnlyBest(bool a) { analyseonlybest = a; }

  /// Sets #writematches
  void SetWriteMatches(bool w) { writematches = w; }

  /// Sets #writematchesfn and #writematches to true;
  void SetWriteMatchesFn(const string &fn) { 
    writematchesfn = fn;
    SetWriteMatches(true);
 }

  /// Sets #secondstage
  void SetSecondStage(bool s) { secondstage = s; }

  /// Sets #usebinaryfiles
  void SetUseBinaryFiles(bool u) { usebinaryfiles = u; }

  /// Sets #displayinfo
  void SetDisplayInfo(bool d) { displayinfo = d; }

  /// Sets #showkeypoints
  void SetShowKeypoints(bool s) { showkeypoints = s; }

  /// Sets #usemask
  void SetUseMask(mask_type mt) { usemask = mt; }

  /// Starts to use file-based masks for images.
  void SetUseFileMask(const string &o) { 
    SetUseMask(MASK_FILE);
    SetUseXMask(o);
  }

  /// Starts to use polygon-based masks for images.
  void SetUsePolygonMask(const string &o) {
    SetUseMask(MASK_POLYGON);
    SetUseXMask(o);
  }

  /// Common implementation for setting up masks.
  void SetUseXMask(const string &o);

  /// Use specified polygon (quadrilateral) borders for database images.
  void SetUseBorders(const string &o);

  /// Sets #showkeypoints if true, i.e. if not explicitly set to false 
  /// at some point.
  void PossiblySetShowKeypoints(bool s) { 
    if (showkeypoints) showkeypoints = s; 
  }

  /// Sets #used_detector
  void SetUsedDetector(const string &d);
  
  /// Sets #used_descriptor
  void SetUsedDescriptor(const string &d);

  /// Sets #updateapriori
  void SetUpdateApriori(bool u) { updateapriori = u; }

  /// Sets #matchingkeypointsname
  void SetMatchingKeypointsName(const string &m) {
    matchingkeypointsname = m;
  }

  /// Sets #matchingkeypointsmode
  void SetMatchingKeypointsMode(const record_keypoints_type &mkm) {
    matchingkeypointsmode = mkm;
  }

  /// Whether matching keypoints are recorded in analysis mode.
  bool RecordMatchingKeypoints(const record_keypoints_type &mkm);

  /// Creates a keypoint filter and sets its #KeypointFilter.name
  /// and #KeypointFilter.threshold
  /// \param f filter name
  /// \param s filter forced namestring, may be empty.
  ///  See #KeypointFilter.namestring.
  /// \param t threshold
  void SetFilter(const string &f, const string &s, size_t t);

  /// Whether a filter is to be used. This doesn't mean that it is ready
  /// for use, so you probably want to test FilterLoaded() instead.
  bool FilterInUse() {
    return (filter && filter->IsInUse());
  }

  /// Whether a filter is loaded and ready.
  bool FilterLoaded() {
    return (filter && filter->IsLoaded());
  }

  /// Returns a filter-specific string to be included in filenames when
  /// filtering is used.
  string FilterString() {
    if (!filter)
      return "XXX";
    else
      return filter->NameString();
  }

  /// Whether joint indices i.e. #dbindices are used.
  bool DbIndicesInUse() { return dbindices_in_use; }

  /// Sets #dbindices_in_use
  void DbIndicesInUse(bool i) { dbindices_in_use = i; }

  /// Processes a file-based image. \param fn filename \param savemode
  /// specifies whether and how the keypoints, descriptors and index should
  /// be saved.
  bool ProcessImage(const string &fn, savetype &savemode);

  /// 
  bool ProcessImage(const string &fn, savetype &savemode, dbitem &res);

  /// The other way to process an image. Assumes that the input image
  /// already exists (loaded separately, obtained from webcam etc.)
  /// Used with SOAP, in analysis mode, and in webcam mode.
  bool ProcessImage(IplImage *image, const string& imgfn);

  /// The first part of common tasks for all ProcessImage()'s.
  bool ProcessImageCommonPartOne(IplImage** imgptr, IplImage** maskptr);

  /// The second part of common tasks for all ProcessImage()'s,
  /// which eventually call this function if everything is ok.
  bool ProcessImageCommonPartTwo(IplImage** imgptr, IplImage** maskptr, 
				 const string& imgfn);

  /// Returns savetype as a string.
  string SavemodeStr(savetype &savemode) {
    if (savemode == NO_SAVING) return "NO_SAVING";
    else if (savemode == SEPARATE_INDICES) return "SEPARATE_INDICES";
    else if (savemode == JOINT_INDEX) return "JOINT_INDEX";
    else if (savemode == MATLAB_MODE) return "MATLAB_MODE";
    else return "UNKNOWN_SAVEMODE";
  }

  /// Displays total time used in detection and extraction.
  void DisplayDetExtTime();

  /// Appends #imageDescriptors to #jointindex.descriptors and
  /// #imageKeypoints to #jointindex.keypoints. @see dbindices.
  void AppendDescriptors();

  /// Checks that the numbers of descriptors, keypoints, and (optional)
  /// filtering results match. For arbitrary \a keyps and \a descs.  
  void CheckKeypointsAndDescriptors(vector<cv::KeyPoint> &keyps,
				    cv::Mat &descs);

  /// Checks that the numbers of descriptors, keypoints, and (optional)
  /// filtering results match. For #imageKeypoints and #imageDescriptors.
  void CheckKeypointsAndDescriptors();

  /// Shows \a ii 'th matching result.
  /// \param openwindow whether to open a new OpenCV window for the results
  bool ShowNthResult(const IplImage *image, size_t ii, bool openwindow=true);

  ///  Shows \a 0 'th matching result using ShowNthResult().
  /// \param openwindow whether to open a new OpenCV window for the results
  bool ShowBestResult(const IplImage *image, bool openwindow=true) {
    return ShowNthResult(image, 0, openwindow);
  }

  /// Show \b all resulting matches
  bool ShowAllResults(const IplImage *image) {
    return ShowResultsCommon(image, true);
  }

  /// Show all results that have at least one matching point.
  bool ShowAllMatchingResults(const IplImage *image) {
    return ShowResultsCommon(image, false);
  }

  /// Common implementation of ShowAllResults() and ShowAllMatchingResults().
  bool ShowResultsCommon(const IplImage *image, bool nonmatchesalso=true);

  /// Shows online recognition results over the whole database.
  /// @see showonlinedbresults.
  bool ShowDbResults(const IplImage *image);

  /// Shows SMI's gaze point.
  bool ShowSMIResults(IplImage *img);

  /// An implementation of a ray-casting algorithm to determine whether 
  /// a point \a p is inside the dbitemborders polygon of \a ii 'th result.
  /// Adapted from: http://bbs.dartmouth.edu/~fangq/MATH/download/source/Determining%20if%20a%20point%20lies%20on%20the%20interior%20of%20a%20polygon.htm
  bool InsidePolygon(CvPoint p, size_t ii);

  /// Processes always the whole database to find the best matches
  bool ProcessWholeDatabase(const IplImage *image);

  /// Process database items one by one until a good-enough match is found
  bool ProcessDatabaseUntilGoodMatch(const IplImage *image);

  /// Processed previous images from #previmgs to find matches
  bool ProcessPrevious(const IplImage *image);

  /// Matches SMI::B_POR() with with polygons.
  bool DoGazeMatch();

  /// Updates the apriori values after a match has been found.
  /// @see updateapriori.
  bool UpdateApriori(size_t);

  /// Entry point for the webcam and video modes. 
  /// \param i webcam number, typically 0; or -1 for video mode 
  /// \param fn video filename
  bool ProcessCapture(int i=0, const string& fn = "");

  /// Entry point for the SOAP server mode. \param port server port to use
  /// \param dummy refers to test server that always returns a dummy answer.
  bool ProcessSoap(string &port, bool dummy=false);

  /// Entry point for the analysis mode. Runs the matching for a pre-specified
  /// set of test images and displays the results. \param fn file containing
  /// the images to analyse.
  bool AnalyseImages(const string& fn);

#ifndef __WINDOWS__
  /// The common implementation of SoapExecuteReal() and SoapExecuteDummy().
  static herror_t SoapExecuteCommon(SoapCtx*, SoapCtx*, bool);

  /// The \b real SOAP service registered for \a sendImage. Performs matching
  /// for the attached image and generates a response.
  /// @see SoapExecuteCommon().
  /// \param req the SOAP request \param res the generated SOAP response
  static herror_t SoapExecuteReal(SoapCtx *req, SoapCtx *res) {
    return SoapExecuteCommon(req, res, false);
  }

  /// A \b dummy SOAP service registered for \a sendImage. Generates a fixed
  /// response intended for testing purposes. @see SoapExecuteCommon()
  /// \param req the SOAP request \param res the generated SOAP response
  static herror_t SoapExecuteDummy(SoapCtx *req, SoapCtx *res) {
    return SoapExecuteCommon(req, res, true);
  }

  /// A SOAP service registered for \a status. Provides status information
  /// about the SOAP server.
  /// \param req the SOAP request \param res the generated SOAP response
  static herror_t SoapStatus(SoapCtx *req, SoapCtx *res);
#endif // __WINDOWS__

  /// Fills a dummy result in \a dummybest.
  static void ProcessDummy(dbitem &dummybest);

  /// Fills a failed result in \a failbest.
  static void ProcessFail(dbitem &failbest);

  /// A method for accessing ProcessImage() from static SoapExecuteCommon()
  /// using the object pointer #op.
  /// \param image image
  /// \param res the best matching result
  bool ProcessImageFromStatic(IplImage *image, dbitem &res);

  /// A method for accessing various pieces of information from static
  /// SoapStatus() using the object pointer #op.
  void GetStatusInfoFromStatic(double&, size_t&, size_t&, time_t&);

  /// Loads an image from disk.
  /// \param imgptr pointer to the \a image pointer
  /// \param fn filename.
  bool LoadImage(IplImage **imgptr, const string& fn);

  /// Scales the input image by the factor specified in #scale.
  /// \param imgptr pointer to the \a image pointer
  bool ScaleImage(IplImage** imgptr);

  /// Scales the input image 
  /// \param imgptr pointer to the \a image pointer  
  /// \param s scale to use
  bool ScaleImage(IplImage** imgptr, double s) {
    bool ok = true;
    double tmp = scale;
    SetScale(s);
    ok = ScaleImage(imgptr);
    scale = tmp;
    return ok;
  }

  /// Resize the image with the longest edge to the input options.
  bool ResizeImage(IplImage** imgptr);

  /// Creates a mask for descriptor extraction based on a polygon.
  bool CreatePolygonMask(IplImage** imgptr, IplImage** maskptr, 
			 const string& fn);

  /// Loads a polygon from a text-format file \a fn.
  bool LoadPolygon(vector<CvPoint>& pts, const string& fn);

  ///
  string DetectorString(const detector_type &dt, bool lc=false);

  ///
  string AllDetectors();

  /// Converts desc_type into a string.
  string DescriptorString(const desc_type &dt, bool lc=false);

  ///
  string AllDescriptors();

  ///
  int UsedDescriptorSize();

  /// Detect keypoints from \a image. @see imageKeypoints
  /// \param maskptr can be used to spatially limit keypoint detection
  bool Detect(const IplImage *image, IplImage **maskptr);

  /// Extracts descriptors from \a image. @see imageDescriptors
  bool Extract(const IplImage *image);

  /// Runs FLANN for #imageDescriptors against a set of SURF descriptors
  /// given in \a dbDescriptors. First creates an index and
  /// then uses the index to search.
  /// \param dbDescriptors target descriptors for which the index is created
  /// \param ptpairs the resulting matches are stored here as two ints.
  void FlannFindPairs(const cv::Mat &dbDescriptors,
		      vector<int>& ptpairs);

  /// Runs FLANN for #imageDescriptors against an existing \a index.
  /// \param index the FLANN index to use.
  /// \param ptpairs the resulting matches are stored here as two ints.
  void FlannFindPairs(FlannIndexType *index,
		      vector<int>& ptpairs);

  /// Helper function for FlannFindPairs(). Fills \a mat with
  /// \a descriptors. NOT NEEDED ANYMORE
  //void FillDescriptorMatrix(cv::Mat &mat,
  //			    const cv::Mat &descriptors);

  /// Finds matching descriptors to #imageDescriptors from \a ii 'th #dbitem
  /// in #db using FlannFindPairs().
  /// \param ii the number of #dbitem in #db to test
  /// \param descmode whether to use descriptors or a FLANN index in matching
  void FindMatches(const size_t ii, bool descmode=false);

  /// A way to call FindMatches(size_t, bool).
  void FindMatchesUsingDescriptors(const size_t ii) {
    return FindMatches(ii, true);
  }

  /// A way to call FindMatches(size_t, bool).
  void FindMatchesUsingIndex(const size_t ii) {
    return FindMatches(ii, false);
  }

  /// Finds matching descriptors to #imageDescriptors from \a ii 'th #dbitem
  /// in #db using FlannFindPairs().
  void FindMatches();

  /// Performs a homography test for \a dbitem.keypoints for \a ii 'th #dbitem
  /// in #db. Also updates vector<bool> \a dbindex.survivors of surviving
  /// keypoints and calculates size_t \a dbindex.nsurvivors.
  /// @see HOMOGRAPHY_NIMAGES, HOMOGRAPHY_DETERMINANT_TH, 
  /// @see HOMOGRAPHY_CHECK_ANGLES,
  /// @see HOMOGRAPHY_MINLENGTH_TH, HOMOGRAPHY_MAXLENGTH_TH
  /// @see HOMOGRAPHY_BACKPROJECTION_TH
  /// \return whether a homography was found or not
  bool FindHomography(const IplImage *image, const size_t ii) {
    return FindHomography(image, db[ii]);
  }
  
  /// Performs a homography test for \a dbit.keypoints. Also updates vector<bool> 
  /// \a dbit.survivors and calculates size_t \a dbit.nsurvivors.
  /// \return whether a homography was found or not
  bool FindHomography(const IplImage *image, dbitem &dbit);

  /// Calculates a perspective projection of (\a x,\a y) to (\a X, \a Y) using 
  /// the vectorized homography matrix \a h. 
  bool CalcHomography(const double *h, double x, double y, double &X, double &Y);

  /// Show the location and scale of #imageKeypoints over a copy of \a image.
  /// \param maskptr if mask is used, it is showed in a separate window
  /// \return \a false if "q" was pressed, \a true otherwise.
  bool ShowKeypoints(const IplImage *image, IplImage** maskptr);

  /// Loads \b text format keypoints and descriptors for a #dbitem \a item
  /// from \a item.descname.
  bool LoadData_Text(dbitem &item);

  /// Loads \b binary format keypoints and descriptors for a #dbitem \a item
  /// from \a item.descname.
  bool LoadData_Bin(dbitem &item);

  /// Loads XML format descriptor data from \a indexname+".xml" and an index
  /// from \a indexname. \return a pointer to the loaded index.
  /// \param indexname name of index to load
  /// \param dbg turn on debug information
  FlannIndexType* LoadIndex(string &indexname, bool dbg=false);

  /// Loads a polygon file and stores the polygon in \a item.dbitemborders
  bool LoadBorders(dbitem &item);

  /// Saves \a keypoints and \a descriptors in \b text format.
  /// \param keypoints keypoints
  /// \param descriptors descriptors
  /// \param fn filename
  /// \param usefilter whether to use keypoints filtered
  ///        with #KeypointFilter.keypoint_ok
  /// \param printsize whether to print the number of rows to the first line
  bool SaveData_Text(const string &fn, const vector<cv::KeyPoint> &keypoints,
		     const cv::Mat &descriptors, bool usefilter=false,
		     bool printsize=true);

  /// Saves \a keypoints and \a descriptors in \b binary format.
  /// \param keypoints keypoints
  /// \param descriptors descriptors
  /// \param fn filename
  /// \param usefilter whether to use keypoints filtered
  ///        with #KeypointFilter.keypoint_ok
  bool SaveData_Bin(const string &fn, const vector<cv::KeyPoint> &keypoints,
		    const cv::Mat &descriptors, bool usefilter=false);

  /// Creates a FLANN index for \a descriptors and saves it to file \a fn
  /// The corresponding descriptors are saved to file \a fn.xml. If \a descfn
  /// is non-empty, the descriptors are saved there in plain text and some
  /// info about keypoints is saved to \a descfn+".kpdata".
  /// \param keypoints keypoints
  /// \param descriptors descriptors
  /// \param fn filename of index
  /// \param descfn optional filename for saving descriptors in plain text
  /// \param usefilter whether to use keypoints filtered
  ///        with #KeypointFilter.keypoint_ok
  bool CreateAndSaveIndex(vector<cv::KeyPoint> &keypoints, 
			  cv::Mat &descriptors,
			  const string &fn, const string &descfn,
			  bool usefilter=false);

  /// Creates a FLANN index for dbindex.descriptors and saves it to file \a fn
  /// The corresponding descriptors are saved to file \a fn.xml. If \a descfn
  /// is non-empty, the descriptors are saved there in plain text and some
  /// info about keypoints is saved to \a descfn+".kpdata".
  /// \param fn filename of index
  /// \param descfn optional filename for saving descriptors in plain text
  /// \param usefilter whether to use keypoints filtered
  ///        with #KeypointFilter.keypoint_ok
  bool CreateAndSaveIndex(const string &fn, const string &descfn,
			  bool usefilter=false);

  /// Loads a database list from file \a fn and stores it in \a list. The
  /// filename of the default index is copied from \a fn to \a indexfn if it
  /// was not explicitly specified in the call of LoadDb().
  /// \param list list of databases
  /// \param fn filename
  /// \param indexfn filename of index
  /// \param dsuffix suffix for the descriptor files
  /// \param path path to image files
  bool LoadDbItems(vector<vector<string> > &list, const string& fn,
		   string& indexfn, string &dsuffix, string &path);

  /// Loads an apriori file from file \a fn and stores the result in
  /// \a aprioris.
  bool LoadDbAprioris(vector<double>& aprioris, string& fn);

  /// Loads an existing database, which will be stored in #db
  /// \param listfn name of the database list file
  /// \param indexfn name of the optional joint index file
  void LoadDb(const string &listfn, string indexfn);

  /// Entry point for loading existing databases from a comma-separated
  /// string.
  /// \param listfns comma separated list of database list files
  /// \param indexfns optional comma separated list of index files
  bool LoadDbs(const string &listfns, const string &indexfns);

  /// Entry point for loading existing databases from a vector<string>.
  /// \param listfns comma separated list of database list files
  /// \param indexfns optional comma separated list of index files
  bool LoadDbs(const vector<string> &listfns, const vector<string> &indexfns);

  /// Entry point for loading existing databases using a list file. Parses
  /// databases from \a fn and then calls LoadDbs().
  /// \param fn database list file
  bool LoadDbsList(const string &fn);

  /// Saves the first line of the TOC file #outdbname. Used in creating
  /// new databases.
  /// \param n size of database
  /// \param idxname name of default index
  void SaveDbHeader(size_t n, const string &idxname);

  /// Saves information about the processed image \a fn to a TOC file
  /// #outdbname. Used in creating new databases.
  void PossiblySaveDbItem(const IplImage *image, const string &fn);

  /// Loads an image list from \a fn and stores the results in \a list,
  /// used in AnalyseImages(). Requires boost/algorithm/string.hpp,
  /// and can be disabled by commenting out
  /// "#define USE_BOOST_ALGORITHM_STRING" in ObjectDetection.h.
  bool LoadImgList(vector<vector<string> > &list, const string& fn);

  /// Checks if a file exists
  bool FileExists(const string &filename);

  /// Is \a s true or false?
  bool IsAffirmative(const char *s);

  ///
  bool ReadSMI(const string& f) { return smi.read(f); }

 protected:

  /// The threshold for keypoint.hessian, only features with larger values
  /// are extracted. OpenCV documentation suggests that good default value
  /// is ~300-500 (can depend on the average local contrast and sharpness
  /// of the image).
  static double SURF_HESSIAN_THRESHOLD;

  /// false means basic SURF descriptors (64 elements each),
  /// true means extended descriptors (128 elements each).
  static bool SURF_USE_EXTENDED;

  /// The default scaling applied to query images, if a scaling factor is not
  /// specified. Usually followed by by gaussian blurring.
  /// @see scale
  static double DEFAULT_SCALING_FACTOR;

  /// Threshold for accepting a correct match: 8 seems to work quite well,
  /// 10 is perhaps overly cautious.
  static size_t MATCHING_NSURVIVORS_TH;

  /// Threshold for the number of matching keypoints for doing the
  /// homography test in non-exhaustive search
  static size_t MATCHING_NMATCHES_TH;

  /// The number of parallel kd-trees to use. Good values are in the
  /// range [1..16].
  static size_t FLANN_NTREES;

  /// The similarity ratio between best and 2nd best matches. Value 0.6 is
  /// traditionally used with SIFT, value over 1.0 means no pruning of
  /// matches. JOINT refers to joint indices.
  static double FLANN_SIMILARITY_RATIO_JOINT;

  /// The similarity ratio between best and 2nd best matches. Value 0.6 is
  /// traditionally used with SIFT, value over 1.0 means no pruning of
  /// matches. SEPARATE refers to imagewise indices.
  static double FLANN_SIMILARITY_RATIO_SEPARATE;

  /// Threshold for rejecting all matches to a target keypoint that has this
  /// many multiple matches from the query image.
  //static size_t FLANN_MULTIPLE_MATCH_TH = 5;
  static size_t FLANN_MULTIPLE_MATCH_TH;

  /// The number of times the Flann tree(s) in the index should be
  /// recursively traversed.
  static size_t FLANN_SEARCH_CHECKS;

  /// The number of images tested on the second stage of joint index mode.
  static size_t JOINT_2NDSTAGE_NIMAGES;

  /// Homography test is done for HOMOGRAPHY_NIMAGES best matches in
  /// exhaustive search
  static size_t HOMOGRAPHY_NIMAGES;

  /// Absolute value of the determinant of the homography matrix has
  /// to be larger than this threshold
  static double HOMOGRAPHY_DETERMINANT_TH;

  /// Whether the check that the angles in the homography mapping are near pi/2
  static bool HOMOGRAPHY_CHECK_ANGLES;

  /// The edges of the quadrilateral have to be longer than this threshold
  static double HOMOGRAPHY_MINLENGTH_TH;

  /// The edges of the quadrilateral have to be smaller than this threshold
  static double HOMOGRAPHY_MAXLENGTH_TH;

  /// Error threshold for backprojection of points based on homography
  static double HOMOGRAPHY_BACKPROJECTION_TH;

  /// Object pointer pointing to the object itself. Used to return back from
  /// the static functions SoapExecuteCommon() and SoapStatus().
  static ObjectDetection *op;

#ifndef __WINDOWS__
  /// Mutex for calls to ProcessImageFromStatic() from SoapExecuteCommon().
  static pthread_mutex_t pi_mutex;
#endif // __WINDOWS__

  /// A simple counter for the number of soap threads currently active.
  static size_t n_soap_threads;

  /// Name of rc file containing modifiable parameter values.
  string rcfilename;

  /// Debug level: 0=none, 1=some, 2=all
  size_t debug;

  /// Time of starting the application
  time_t start_time;

  /// Number of image queries successfully processed.
  size_t nqueries;

  /// Number of SOAP calls processed.
  size_t nsoaps;

  /// 
  detector_type used_detector;

  ///
  descriptor_type used_descriptor;

  /// Whether to load and save data in binary format. True by default.
  bool usebinaryfiles;

  /// Whether joint indices are used in matching.
  bool dbindices_in_use;

  /// The scaling factor applied to the input image, if explicitly specified.
  /// Otherwise, if matching, #DEFAULT_SCALING_FACTOR is used instead.
  /// Therefore you probably want to use GetScale() instead than this directly.
  double scale;

  /// The input images will be resized to the \a newsize pixels on the longer 
  /// axis if specified. Set to -1 by default.
  int newsize;

  /// A set of scales to be applied to the input image. The current scale is 
  /// always stored in #scale.
  vector<double> scales;

  /// Whether do apply gaussian blurring on the input image. Default is true
  /// when matching, false otherwise.
  bool applyblur;

  /// Parameters for SURF extraction
  //CvSURFParams surfparams;

  /// Parameters for MSER extraction
  //CvMSERParams mserparams;

  /// The database of images (actually descriptors, indices and other
  /// associated data) is stored here.
  vector<dbitem> db;

  /// The joint indices containing descriptors from multiple database images.
  /// This is used while matching; in index creation #dbindex is used instead.
  vector<jointindex> dbindices;

  /// A single joint index containing descriptors from multiple database
  /// images. This is used only when creating new indices; when matching,
  /// #dbindices is used instead.
  jointindex dbindex;

  /// The storage used for all OpenCV memory allocations.
  CvMemStorage* storage;

  /// Keypoints of the current image
  //CvSeq *imageKeypoints;
  vector<cv::KeyPoint> imageKeypoints;

  /// Descriptors corresponding to #imageKeypoints
  //CvSeq *imageDescriptors;
  cv::Mat imageDescriptors;

  // The processed image. When matching, the query image.
  // This was removed on Nov 5, 2010, which may cause weird things!
  //IplImage* image;

  /// Font used in displaying information overlaid to shown images.
  CvFont statusfont;

  /// The type of FLANN index to create and save.
  flann_index_type flann_index_mode;

  /// Whether the whole database is always searched 
  /// (doesn't work with joint indices).
  bool exhaustivesearch;

  /// Whether to consider only the best match in analysis.  Multiple
  /// groundtruth are treated as alternatives if true, as separate if false.
  bool analyseonlybest;

  /// Whether to write all matches to a text file.
  bool writematches;

  /// File name for writing matches to a file.
  string writematchesfn;

  /// Whether the second matching stage is done (with joint indices)
  bool secondstage;

  /// The matching keypoints in the index, as <keypoint,nmatches>.
  /// These are optionally recorded in analysis mode.
  map<size_t, size_t> matchingkeypoints;

  /// Whether to use mask to spatially limit keypoint detection.  
  /// Supports two types, MASK_FILE and MASK_POLYGON.  Default is MASK_NONE.
  mask_type usemask;

  /// The "from" and "to" strings in filename replace from masks. 
  pair<string,string> maskreplace;

  /// Whether to use specified quadrilateral borders for database images.
  bool useborders;

  /// The "from" and "to" strings in filename replace from borders. 
  pair<string,string> borderreplace;

  /// Whether to show the recognition results images.
  /// Overrides #showkeypoints if false.
  bool showresults;

  /// Whether to show online recognition results. Used in capture mode but
  /// can be used by calling ProcessImage() directly as well.
  /// Overrides #showresults if true.
  bool showonlineresults;

  /// Whether to show online recognition results over the whole database.
  /// Used in capture mode.
  /// Overrides #showresults if true.
  bool showonlinedbresults;

  /// The image containing the thumbnails of database images and some other 
  /// needed variables. Used with #showonlinedbresults. 
  /// @see ShowDbResults().
  dbresultimage dbresultimg;

  /// whether to display textual information on results images
  bool displayinfo;

  /// whether to show the keypoints found in an image
  bool showkeypoints;

  /// whether to update the a priori probabilities of the database items
  /// @see UpdateApriori()
  bool updateapriori;

  /// preserve the order of database items, i.e. do not use a priori sorting
  bool preserveorder;

  /// whether to listen for SOAP connections
  bool soap;

  /// Previously processed images that can also be matched against
  vector<dbitem> previmgs;

  /// The name of the image written if 's' is pressed when displaying results
  string outimgname;

  /// The name of the output video written.  Used only with ProcessCapture().
  string outvidname;

  /// The name of the image database TOC file, written if so requested.
  string outdbname;

  /// The mode used for recording matching keypoints.
  record_keypoints_type matchingkeypointsmode;

  /// The name of the output file containing the matching keypoints.
  string matchingkeypointsname;

  /// The filter optionally used to reduce the number of keypoints and
  /// descriptors after descriptor extraction.
  KeypointFilter *filter;

  /// A video output for showing results.
  cv::VideoWriter outputVideo;

  /// Running numer of processed frames.
  size_t frame_no;

  /// Number of CUDA devices found. Is 0 or -1 if disabled/none.
  int ncuda;

  /// Total time used in detection and extraction.
  pair<double,double> det_ext_time;

  /// SMI gaze tracker data
  SMI smi;
};

#endif // _ObjectDetection_h_
