// -*- C++ -*-  $Id: od.C,v 1.60 2014/01/16 15:11:47 jorma Exp $
//
// Copyright 2009-2014 PicSOM Development Group <picsom@cis.hut.fi>
// Aalto University School of Science and Technology
// Department of Information and Computer Science
// P.O.BOX 15400, FI-00076 Aalto, FINLAND
//

#include "ObjectDetection.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

#include <iostream>
#include <string>
#include <fstream>
#include <vector>

#ifdef USE_BOOST_ALGORITHM_STRING
#include <boost/algorithm/string.hpp>
using namespace boost::algorithm;
#endif

const string DEFAULT_SOAP_PORT = "10000";

// ---------------------------------------------------------------------

void WARN(const string &msg) {
  cerr << "WARNING: " << msg << endl;
}

// ---------------------------------------------------------------------

void ERROR(const string &msg) {
  cerr << "ERROR: " << msg << endl;
  exit(1);
}

// ---------------------------------------------------------------------

bool processImageList(vector<string> &imgs, const string imglist);

// ---------------------------------------------------------------------

int main(int argc, char** argv) {

  const char* help_opt = "--help";
  int help_opt_len = (int)strlen(help_opt);
  const char* param_opt = "--param:";
  int param_opt_len = (int)strlen(param_opt);
  const char* option_opt = "--option:";
  int option_opt_len = (int)strlen(option_opt);
  const char* debug_opt = "--debug=";
  int debug_opt_len = (int)strlen(debug_opt);
  const char* match_opt = "--match=";
  int match_opt_len = (int)strlen(match_opt);
  const char* matchlist_opt = "--matchlist=";
  int matchlist_opt_len = (int)strlen(matchlist_opt);
  const char* imagelist_opt = "--imagelist=";
  int imagelist_opt_len = (int)strlen(imagelist_opt);
  const char* video_opt = "--video=";
  int video_opt_len = (int)strlen(video_opt);
  const char* smi_opt = "--smi=";
  int smi_opt_len = (int)strlen(smi_opt);
  const char* create_opt = "--create=";
  int create_opt_len = (int)strlen(create_opt);
  const char* index_opt = "--index=";
  int index_opt_len = (int)strlen(index_opt);
  const char* filter_opt = "--filter=";
  int filter_opt_len = (int)strlen(filter_opt);
  const char* kpsfile_opt = "--kpsfile=";
  int kpsfile_opt_len = (int)strlen(kpsfile_opt);
  const char* analyse_opt = "--analyse=";
  int analyse_opt_len = (int)strlen(analyse_opt);
  const char* analysekps_opt = "--analysekps=";
  int analysekps_opt_len = (int)strlen(analysekps_opt);
  const char* analysebest_opt = "--analysebest";
  int analysebest_opt_len = (int)strlen(analysebest_opt);
  const char* scale_opt = "--scale=";
  int scale_opt_len = (int)strlen(scale_opt);
  const char* newsize_opt = "--newsize=";
  int newsize_opt_len = (int)strlen(newsize_opt);
  const char* noblur_opt = "--noblur";
  int noblur_opt_len = (int)strlen(noblur_opt);
  const char* usemask_opt = "--usemask=";
  int usemask_opt_len = (int)strlen(usemask_opt);
  const char* usepolygon_opt = "--usepolygon=";
  int usepolygon_opt_len = (int)strlen(usepolygon_opt);
  const char* useborders_opt = "--useborders=";
  int useborders_opt_len = (int)strlen(useborders_opt);
  const char* output_opt = "--output=";
  int output_opt_len = (int)strlen(output_opt);
  const char* output_video_opt = "--outputvideo=";
  int output_video_opt_len = (int)strlen(output_video_opt);
  const char* kmeans_opt = "--kmeans";
  int kmeans_opt_len = (int)strlen(kmeans_opt);
  const char* exhaustive_opt = "--exhaustive";
  int exhaustive_opt_len = (int)strlen(exhaustive_opt);
  const char* writematches_opt = "--writematches=";
  int writematches_opt_len = (int)strlen(writematches_opt);
  const char* matlab_opt = "--matlab";
  int matlab_opt_len = (int)strlen(matlab_opt);
  const char* checkindex_opt = "--checkindex";
  int checkindex_opt_len = (int)strlen(checkindex_opt);
  const char* nobinary_opt = "--nobinary";
  int nobinary_opt_len = (int)strlen(nobinary_opt);
  const char* noshowres_opt = "--noshowresults";
  int noshowres_opt_len = (int)strlen(noshowres_opt);
  const char* showolres_opt = "--showonlineresults";
  int showolres_opt_len = (int)strlen(showolres_opt);
  const char* showoldbres_opt = "--showonlinedbresults";
  int showoldbres_opt_len = (int)strlen(showoldbres_opt);
  const char* noshowkps_opt = "--noshowkeypoints";
  int noshowkps_opt_len = (int)strlen(noshowkps_opt);
  const char* nodispinfo_opt = "--nodisplayinfo";
  int nodispinfo_opt_len = (int)strlen(nodispinfo_opt);
  const char* soap_opt = "--soap";
  int soap_opt_len = (int)strlen(soap_opt);
  const char* soap2_opt = "--soap=";
  int soap2_opt_len = (int)strlen(soap2_opt);
  const char* soapdummy_opt = "--soapdummy";
  int soapdummy_opt_len = (int)strlen(soapdummy_opt);
  const char* soapdummy2_opt = "--soapdummy=";
  int soapdummy2_opt_len = (int)strlen(soapdummy2_opt);
  const char* detector_opt = "--detector=";
  int detector_opt_len = (int)strlen(detector_opt);
  const char* descriptor_opt = "--descriptor=";
  int descriptor_opt_len = (int)strlen(descriptor_opt);

  int debug_value = -1;
  vector<string> img_fns;
  string dblist_fn, dblist_fn_out, matchlist_fn;
  string imagelist_fn, analyse_fn, index_fn, kps_fn;
  string soap_port_str, scale_str, video_fn, smi_fn;
  bool do_matching = false, blur_query_images = true;
  bool soap_mode = false, soap_dummy_mode = false, do_exhaustive = false;
  bool matlab_mode = false, check_index = false;

  ObjectDetection od;

  od.Initialize();

  cout << "Command line arguments: ";
    for (int i = 1; i < argc; i++)
      cout << argv[i] << " ";
  cout << endl;

  for (int i = 1; i < argc; i++) {
    bool found = false;

    if (strncmp(argv[i], help_opt, help_opt_len) == 0) {
      cout << "Usage: od [ --options ] [ imgfile1 imgfile2 ... ]"
	   << endl << endl
	   << "Options: --help           : print this help"          << endl
	   << "\t --param:X=Y      : set parameter X to value Y"     << endl   
	   << "\t --option:X=Y     : set option X to Y (all options not supported)" 
	   << endl       
	   << "\t --debug=X        : set debugging level to X, "   
	   << "currently X=1 and X=2 are supported"                  << endl
	   << "\t --match=X        : match images to database X"     << endl
	   << "\t --match=X,Y,Z    : match images to databases X, Y and Z" 
	   << endl
	   << "\t --matchlist=X    : match images to databases listed in X" 
	   << endl
	   << "\t --imagelist=X    : process images listed in X"     << endl
	   << "\t --video=X        : process the video file X"       << endl
	   << "\t --smi=X          : read SMI gaze data file X"      << endl
	   << "\t --create=X       : create database file X"         << endl
	   << "\t --index=X        : create/use a joint index X; "
	   << "use X=noindex to force imagewise indices"             << endl
	   << "\t --index=X,Y,Z    : use joint indices X, Y and Z "
	   << "in matching; these have to coincide with --match"     << endl
	   << "\t --detector=X     : use X as keypoint detector, "   
	   << "supported are: " << od.AllDetectors()                 << endl
	   << "\t --descriptor=X   : use X as keypoint descriptor, "
	   << "supported are: " << od.AllDescriptors()               << endl
	   << "\t --usemask=X,Y    : use mask to limit keypoints, "
	   << "mask file replaces \"X\" with \"Y\" in filename"      << endl
	   << "\t --usepolygon=X,Y : use polygon to create a convex mask, "
	   << "polygon file replaces \"X\" with \"Y\" in filename"   << endl
	   << "\t --useborders=X,Y : use quadrilateral borders for db images, "
	   << "polygon file replaces \"X\" with \"Y\" in filename"   << endl
	   << "\t --filter=y,X     : use filter X with threshold y"  << endl
	   << "\t --filter=y,X,Z   : as above but use Z as namestring"
	   << endl
	   << "\t --kpsfile=X      : write keypoint and descriptor "
	   << "data as text to X"                                    << endl 
	   << "\t --analyse=X      : analyse images in file X"       << endl
	   << "\t --analysekps=y,X : record matching keypoints to X; " 
	   << "y specifies mode: y=0: all; y=1: after homography"    << endl
	   << "\t --analysebest    : consider only best match in analysis;"      
	   << "i.e. the old approach"                                << endl
	   << "\t --scale=X,Y,Z    : scale images with factors X,Y,Z"<< endl
	   << "\t --newsize=X      : resize images to X pixels on the longer axis" 
	   << endl
	   << "\t --noblur         : do not blur query images"       << endl
	   << "\t --output=X       : use X as output image name"     << endl
	   << "\t --outputvideo=X  : use X as output video name"     << endl
	   << "\t --kmeans         : use hierarchical kmeans"        << endl
	   << "\t --exhaustive     : perform exhaustive search"      << endl
	   << "\t --writematches=X : write all matches to file \"X\"" 
	   << endl
	   << "\t --matlab         : write only Matlab output files" << endl
	   << "\t --checkindex     : load an index and check its status"
	   << endl
	   << "\t --nobinary       : write output files as text"     << endl
	   << "\t --noshowresults  : disable showing of result windows" 
	   << endl
	   << "\t --showonlineresults : show online results in capture mode" 
	   << endl
	   << "\t --showonlinedbresults : show database results in capture mode" 
	   << endl
	   << "\t --noshowkeypoints : disable showing of keypoints windows" 
	   << endl
	   << "\t --nodisplayinfo  : disable osd information"        << endl
	   << "\t --soap           : SOAP mode with default port ("
	   << DEFAULT_SOAP_PORT << ")"                               << endl
	   << "\t --soap=X         : SOAP mode with port=X"          << endl
	   << "\t --soapdummy      : dummy SOAP with default port ("
	   << DEFAULT_SOAP_PORT << ")"                               << endl
	   << "\t --soapdummy=X    : dummy SOAP with port=X"         << endl
	   << endl;

      cout << "See doc/html/index.html for more documentation. " 
	   << "(You might have to run \"make doc\" first.)" << endl << endl;
      return 0;
    }

    if (strncmp(argv[i], param_opt, param_opt_len) == 0) {
      found = true;
      od.Interpret(argv[i]+param_opt_len); 
    }

    if (strncmp(argv[i], option_opt, option_opt_len) == 0) {
      found = true;
      od.SetOption(argv[i]+option_opt_len); 
    }

    if (strncmp(argv[i], debug_opt, debug_opt_len) == 0) {
      found = true;
      debug_value = atoi(argv[i]+debug_opt_len); 
    }

    if (strncmp(argv[i], match_opt, match_opt_len) == 0) {
      found = true;
      do_matching = true;
      dblist_fn = argv[i]+match_opt_len;
    }

    if (strncmp(argv[i], matchlist_opt, matchlist_opt_len) == 0) {
      found = true;
      do_matching = true;
      matchlist_fn = argv[i]+matchlist_opt_len;
    }

    if (strncmp(argv[i], imagelist_opt, imagelist_opt_len) == 0) {
      found = true;
      imagelist_fn = argv[i]+imagelist_opt_len;
    }

    if (strncmp(argv[i], video_opt, video_opt_len) == 0) {
      found = true;
      video_fn = argv[i]+video_opt_len;
    }

    if (strncmp(argv[i], smi_opt, smi_opt_len) == 0) {
      found = true;
      smi_fn = argv[i]+smi_opt_len;
    }

    if (strncmp(argv[i], create_opt, create_opt_len) == 0) {
      found = true;
      dblist_fn_out = argv[i]+create_opt_len;
      od.SetOutDbName(dblist_fn_out);
    }

    if (strncmp(argv[i], index_opt, index_opt_len) == 0) {
      found = true;
      index_fn = argv[i]+index_opt_len;
    }

    if (strncmp(argv[i], filter_opt, filter_opt_len) == 0) {
      found = true;
      string tmp(argv[i]+filter_opt_len);

      size_t comma = tmp.find(",");
      if (comma == string::npos || 
	  tmp.substr(0,comma).empty() || tmp.substr(comma+1).empty())
	ERROR("Switch " + string(argv[i]) + " not in recognized format (1)");
      string thstr = tmp.substr(0,comma);
      string fstr = tmp.substr(comma+1);
      string sstr = "";

      comma = fstr.find(",");
      if (comma != string::npos) {
	if (fstr.substr(0,comma).empty() || fstr.substr(comma+1).empty())
	  ERROR("Switch " + string(argv[i]) + " not in recognized format (2)");
	sstr = fstr.substr(comma+1);
	fstr = fstr.substr(0,comma);
      }

      //cout << "fstr=" << fstr << ", sstr=" << sstr << ", thstr=" << thstr << endl;
      od.SetFilter(fstr, sstr, atoi(thstr.c_str()));
    }

    if (strncmp(argv[i], kpsfile_opt, kpsfile_opt_len) == 0) {
      found = true;
      kps_fn = argv[i]+kpsfile_opt_len;
    }

    if (strncmp(argv[i], analyse_opt, analyse_opt_len) == 0) {
      found = true;
      analyse_fn = argv[i]+analyse_opt_len;
    }

    if (strncmp(argv[i], analysekps_opt, analysekps_opt_len) == 0) {
      found = true;
      string tmp(argv[i]+analysekps_opt_len);
      if (tmp.size()<3)
	ERROR("Switch " + string(argv[i]) + " not in recognized format");
      od.SetMatchingKeypointsName(tmp.substr(2));
      if (tmp.substr(0,2)=="0,")
	od.SetMatchingKeypointsMode(ALL_KEYPOINTS);
      else if (tmp.substr(0,2)=="1,") {
	od.SetMatchingKeypointsMode(AFTER_HOMOGRAPHY);
	od.SetSecondStage(false);
      } else 	
	ERROR("Switch " + string(argv[i]) + " not in recognized format"); 
    }

    if (strncmp(argv[i], analysebest_opt, analysebest_opt_len) == 0) {
      found = true;
      od.SetAnalyseOnlyBest(true);
    }

    if (strncmp(argv[i], scale_opt, scale_opt_len) == 0) {
      found = true;
      scale_str = argv[i]+scale_opt_len;
    }
    if (strncmp(argv[i], newsize_opt, newsize_opt_len) == 0) {
      found = true;
      od.SetNewsize(atof(argv[i]+newsize_opt_len));
    }
    if (strncmp(argv[i], noblur_opt, noblur_opt_len) == 0) {
      found = true;
      blur_query_images = false;
    }
    if (strncmp(argv[i], usemask_opt, usemask_opt_len) == 0) {
      found = true;
      od.SetUseFileMask(argv[i]+usemask_opt_len);
    }
    if (strncmp(argv[i], usepolygon_opt, usepolygon_opt_len) == 0) {
      found = true;
      od.SetUsePolygonMask(argv[i]+usepolygon_opt_len);
    }
    if (strncmp(argv[i], useborders_opt, useborders_opt_len) == 0) {
      found = true;
      od.SetUseBorders(argv[i]+useborders_opt_len);
    }
    if (strncmp(argv[i], output_opt, output_opt_len) == 0) {
      found = true;
      od.SetOutImgName(argv[i]+output_opt_len);
    }
    if (strncmp(argv[i], output_video_opt, output_video_opt_len) == 0) {
      found = true;
      od.SetOutVidName(argv[i]+output_video_opt_len);
    }

    if (strncmp(argv[i], kmeans_opt, kmeans_opt_len) == 0) {
      found = true;
      od.SetFlannIndexMode(KMEANS);
    }

    if (strncmp(argv[i], exhaustive_opt, exhaustive_opt_len) == 0) {
      found = true;
      do_exhaustive = true;
    }

    if (strncmp(argv[i], writematches_opt, writematches_opt_len) == 0) {
      found = true;
      od.SetWriteMatchesFn(argv[i]+writematches_opt_len);
    }

    if (strncmp(argv[i], matlab_opt, matlab_opt_len) == 0) {
      found = true;
      matlab_mode = true;
    }

    if (strncmp(argv[i], checkindex_opt, checkindex_opt_len) == 0) {
      found = true;
      check_index = true;
    }

    if (strncmp(argv[i], nobinary_opt, nobinary_opt_len) == 0) {
      found = true;
      od.SetUseBinaryFiles(false);
    }

    if (strncmp(argv[i], noshowres_opt, noshowres_opt_len) == 0) {
      found = true;
      od.SetShowResults(false);
    }

    if (strncmp(argv[i], showolres_opt, showolres_opt_len) == 0) {
      found = true;
      od.SetShowOnlineResults(true);
    }

    if (strncmp(argv[i], showoldbres_opt, showoldbres_opt_len) == 0) {
      found = true;
      od.SetShowOnlineDbResults(true);
    }

    if (strncmp(argv[i], noshowkps_opt, noshowkps_opt_len) == 0) {
      found = true;
      od.SetShowKeypoints(false);
    }

    if (strncmp(argv[i], nodispinfo_opt, nodispinfo_opt_len) == 0) {
      found = true;
      od.SetDisplayInfo(false);
    }

    if (strncmp(argv[i], soap_opt, soap_opt_len) == 0) {
      found = true;
      soap_mode = true;
      if (strncmp(argv[i], soap2_opt, soap2_opt_len) == 0)
	soap_port_str = argv[i]+soap2_opt_len;
    }

    if (strncmp(argv[i], soapdummy_opt, soapdummy_opt_len) == 0) {
      found = true;
      soap_dummy_mode = true;
      if (strncmp(argv[i], soapdummy2_opt, soapdummy2_opt_len) == 0)
	soap_port_str = argv[i]+soapdummy2_opt_len;
    }

    if (strncmp(argv[i], detector_opt, detector_opt_len) == 0) {
      found = true;
      od.SetUsedDetector(argv[i]+detector_opt_len);
    }

    if (strncmp(argv[i], descriptor_opt, descriptor_opt_len) == 0) {
      found = true;
      od.SetUsedDescriptor(argv[i]+descriptor_opt_len);
    }

    if (!found && strncmp(argv[i], "--", 2) == 0)
      ERROR("Switch " + string(argv[i]) + " not recognized");

    if (!found)
      img_fns.push_back(argv[i]);
  }

  if (dblist_fn != "" && matchlist_fn != "")
    WARN("Switch --matchlist has precedence over --match");

  if (debug_value > 0)
    od.SetDebug(debug_value);

  if (scale_str != "")
    od.SetScale(scale_str);

  if (imagelist_fn != "")
    processImageList(img_fns, imagelist_fn);

  if (do_matching) {
    if (matchlist_fn != "") {
      if (index_fn != "")
	WARN("Switch --index does not have any effect with --matchlist");

      if (!od.LoadDbsList(matchlist_fn))
	ERROR("Loading databases failed");

    } else 
      if (!od.LoadDbs(dblist_fn, index_fn)) 
	ERROR("Loading databases failed");

    od.SetBlur(blur_query_images);
  }

  if (do_exhaustive) {
    od.SetExhaustiveSearch(true);
    if (od.DbIndicesInUse())
      WARN("Switch --exhaustive only controls the homography estimation "
	   "when using joint indices");
  }

  if (soap_port_str == "")
    soap_port_str = DEFAULT_SOAP_PORT;

  if (soap_dummy_mode) {
#ifndef __WINDOWS__
    od.ProcessSoap(soap_port_str, true);
#endif // __WINDOWS__

  } else if (soap_mode) {
    if (!do_matching)
      ERROR("Switch --soap requires also --match");

#ifndef __WINDOWS__
    od.ProcessSoap(soap_port_str);
#endif // __WINDOWS__

  } else if (check_index) {
    if (index_fn == "")
      ERROR("Switch --checkindex requires also --index");
    
    od.LoadIndex(index_fn, true);

  } else {

    if (smi_fn!="")
      od.ReadSMI(smi_fn);

    if (analyse_fn != "") {
      if (!do_matching)
    	ERROR("Switch --analyse requires also --match");

      od.AnalyseImages(analyse_fn);

    } else if (video_fn != "") {
      od.ProcessCapture(-1, video_fn);
     
    } else if (img_fns.size()==0) {
      od.ProcessCapture(0);

    } else {
      if (dblist_fn_out != "") {
	size_t c = index_fn.find(",");
	if (c != string::npos)
	  ERROR("Switch --create does not support multiple indices");

	od.SaveDbHeader(img_fns.size(),
			(index_fn != "" ? index_fn : "noindex"));
      }

      savetype savemode = NO_SAVING;
      if (matlab_mode) {
	savemode = MATLAB_MODE;
      } else if (!do_matching) {
	if (index_fn != "")
	  savemode = JOINT_INDEX;
	else
	  savemode = SEPARATE_INDICES;
      }
      
      bool ok = true;
      for (size_t ii=0; ii<img_fns.size(); ii++)
	ok = ok && (od.ProcessImage(img_fns[ii], savemode));

      if (ok)
	od.DisplayDetExtTime();

      if (ok && savemode == JOINT_INDEX) {
	
	od.CreateAndSaveIndex(index_fn, kps_fn, false);
      }
    }

  }

  return 0;
}

// ---------------------------------------------------------------------

bool processImageList(vector<string> &imgs, const string imglist) {
  bool dbg = false;

#ifdef USE_BOOST_ALGORITHM_STRING

  ifstream listfile(imglist.c_str());
  if (!listfile) {
    if (dbg)
      cout << "processImageList(): File not found, exiting" << endl;
    return false;
  }

  string line;
  while (getline(listfile, line)) {
    if (dbg)
      cout << "processImageList(): Processing line: \"" << line << "\"" << endl;
    trim(line);
    if (dbg)
      cout << "processImageList(): After trimming: \"" << line << "\"" << endl;

    if (line.find("#") != string::npos) {
      if (dbg)
	cout << "processImageList(): Character '#' found, this line is a comment"
	     << endl;
      continue;
    }
    
    if (dbg)
      cout << "processImageList(): Adding \"" << line << "\" to the image list"
	   << endl;
    imgs.push_back(line);
  }

#else

  if (dbg)
    cout << "processImageList(): This requires boost" << endl;

#endif

  return true;
}

// ---------------------------------------------------------------------
