// -*- C++ -*-  $Id: KeypointFilter.h,v 1.15 2014/01/16 15:17:38 jorma Exp $
//
// Copyright 2009-2014 PicSOM Development Group <picsom@cis.hut.fi>
// Aalto University School of Science and Technology
// Department of Information and Computer Science
// P.O.BOX 15400, FI-00076 Aalto, FINLAND
//

#ifndef _KeypointFilter_h_
#define _KeypointFilter_h_

// pre-2.2 OpenCV:
//#include <opencv/cv.h>
//#include <opencv/highgui.h>

// 2.2 and beyond OpenCV:
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/features2d/features2d.hpp>

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

#include <iostream>
#include <string>
#include <fstream>
#include <vector>
#include <iterator>

#include "od_defs.h"

#ifdef USE_LIBSVM
#include <svm.h>
#endif

using namespace std;

/// A base class for all filters of extracted keypoints.
class KeypointFilter {
  
public:
  
  /// Constructor.
  KeypointFilter() { 
    name = "";
    namestring = "XXX";
    threshold = 0;
  }
  
  /// Sets #debug.
  void SetDebug(size_t d) { debug = d; }
  
  /// Sets #name and #threshold.
  void SetFilter(const string &f, size_t t) { 
    name = f; 
    threshold = t; 
  }

  /// Sets #namestring.
  void SetNameString(const string &s) {
    namestring = s;
  }

  /// Loads the filter from file #name.
  virtual bool LoadFilter(int size=0) = 0;
  
  /// Filters the descriptors in \a descs.
  virtual void FilterDescriptors(const vector<cv::KeyPoint> &keyps, 
				 const cv::Mat &descs) = 0;
  
  /// Whether the filter is to be used. For actual use the filter needs to 
  /// be loaded, which is tested with IsLoaded().
  bool IsInUse() const { return name != ""; }

  /// Whether the filter is loaded and ready to be used.
  virtual bool IsLoaded() const = 0;

  /// Filter-specific part of relevant filenames.
  string NameString() { return namestring; }

  /// Temporary storage for results of the filtering. Contains one bool
  /// per descriptor stating if the descriptor passed the filtering.
  vector<bool> keypoint_ok;

protected:

  /// Debug level: 0=none, 1=some, 2=all
  size_t debug;

  /// The name of the codebook file.
  string name;

  /// The string to be added to filenames of filtered descriptors
  string namestring;

  /// The used threshold for filtering descriptors
  size_t threshold;

}; // class KeypointFilter

/// A clustering-based filter for extracted keypoints.
class ClusterFilter : public KeypointFilter {

public:

  /// Constructor.
  ClusterFilter() {
    codebook = NULL;
    namestring = "filt";
  }

  /// Whether the filter is loaded and ready to be used.
  bool IsLoaded() const { return codebook != NULL; }

  /// Loads the filter clusters into #codebook from file #name
  bool LoadFilter(int size=0);

  /// Filters the descriptors in #ObjectDetection.imageDescriptors using 
  /// #codebook and #threshold. Note that #ObjectDetection.imageDescriptors 
  /// is not changed, but rather the results are stored at #keypoint_ok.
  void FilterDescriptors(const vector<cv::KeyPoint> &keyps, 
			 const cv::Mat &descs);

protected:

  /// The codebook used in filtering as matrix
  cv::Mat *codebook;

}; // class ClusterFilter

#ifdef USE_LIBSVM

/// A SVM-based filter for extracted keypoints. Uses LIBSVM.
class SVMFilter : public KeypointFilter {

public:

  /// Constructor.
  SVMFilter() {
    model = NULL;
    namestring = "svmfilt";
  }

  /// Whether the filter is loaded and ready to be used.
  bool IsLoaded() const { return model != NULL; }
 
  /// Loads the SVM model into #model from file #name
  bool LoadFilter(int size=0);

  /// Filters the descriptors in #ObjectDetection.imageDescriptors using 
  /// the SVM model #model. Note that #ObjectDetection.imageDescriptors 
  /// is not changed, but rather the results are stored at #keypoint_ok.
  void FilterDescriptors(const vector<cv::KeyPoint> &keyps, 
			 const cv::Mat &descs);

protected:

  /// The SVM model.
  struct svm_model* model;

  /// Type of SVM, should be always 0.
  int svm_type;
  
  /// Number of classes, should be always 2.
  int nr_class;

  /// Threshold used with SVM filtering.
  double svmthreshold;

  /// Converts vector \a v to a format accepted by svm_predict_probability().
  struct svm_node* ConvertVector(const cv::Mat &v);

}; // class SVMFilter

#endif

/// A random filter for extracted keypoints. Intended as a benchmark.
class RandomFilter : public KeypointFilter {

public:

  /// Constructor.
  RandomFilter() {
    namestring = "rndfilt";
  }

  /// The random filter is always loaded and ready to be used.
  bool IsLoaded() const { return true; }
 
  /// The random filter is always loaded and ready to be used.
  bool LoadFilter(int size=0);

  /// Filters the descriptors in #ObjectDetection.imageDescriptors using 
  /// the random number generator. Note that #ObjectDetection.imageDescriptors 
  /// is not changed, but rather the results are stored at #keypoint_ok.
  void FilterDescriptors(const vector<cv::KeyPoint> &keyps, 
			 const cv::Mat &descs);

}; // class RandomFilter

/// A filter based on hessian values for extracted keypoints. 
class HessianFilter : public KeypointFilter {

public:

  /// Constructor.
  HessianFilter() {
    namestring = "hessfilt";
  }

  /// The hessian filter is always loaded and ready to be used.
  bool IsLoaded() const { return true; }
 
  /// The hessian filter is always loaded and ready to be used.
  bool LoadFilter(int size=0);
  
  /// Filters the descriptors in #ObjectDetection.imageKeypoints using 
  /// the keypoints' hessian values. Note that #ObjectDetection.imageKeypoints
  /// is not changed, but rather the results are stored at #keypoint_ok.
  /// Note that unlike other filters, this uses #ObjectDetection.imageKeypoints, 
  /// not #ObjectDetection.imageDescriptors.
  void FilterDescriptors(const vector<cv::KeyPoint> &keyps, 
			 const cv::Mat &descs);

}; // class HessianFilter


#endif // _KeypointFilter_h_
