#include "AnalysisController.hpp"
#include "FeatureCreator.hpp"
#include "pythonapi.hpp"

using std::vector;
using cv::Mat;
using std::string;



namespace {
  // Initialisation function for the field name vector
  vector<string> getInitialisedFieldNames() {
    vector<string> f(slmotion::SOM_PAK_COMPONENTS.size());
    for (size_t i = 0; i < f.size(); ++i)
      f[i] = slmotion::SOM_PAK_COMPONENTS[i].getName();
    return f;
  }



  /**
   * Returns an initialised field name map that can be used to initialise
   * static const members.
   */
  std::map<string, unsigned int> getInitialisedFieldNameMap() {
    // create field-name-to-index mapping
    std::map<string, unsigned int> fieldNameMap;
    const vector<string> fieldNameVector = getInitialisedFieldNames();
    for (size_t i = 0; i < fieldNameVector.size(); i++)
      fieldNameMap[fieldNameVector[i]] = i;
    return fieldNameMap;
  }
}



namespace slmotion {
  extern int debug;

  void AnalysisController::analyseVideofiles(Analyser& analyser, // vector<Analyser>& analysers,
					     size_t skipFrames,
					     size_t framesMax,
					     bool storeImages,
                                             // bool disableAnnotationBar,
                                             const std::string& elanFilename,
                                             const std::string& annTemplateFilename,
                                             const std::string& inputFilename,
                                             // Visualiser::CombinationStyle combinationStyle,
                                             python::PythonEnvironment* pyEnv,
                                             UiCallback* uiCallback,
                                             const std::string& outTierXmlFile) {
    size_t firstFrame = skipFrames;
    size_t lastFrame = framesMax + skipFrames;

    // make sure the numbers don't roll over
    if (lastFrame < framesMax || lastFrame < skipFrames)
      lastFrame = ~0;

    // ensure that lastFrame does not exceed the number of frames in any 
    // analysers; otherwise, limit the number of frames to process there
    if (lastFrame >= analyser.getFrameSource().size()) 
      lastFrame = analyser.getFrameSource().size();
      
    try {
      if (pyEnv)
        pyEnv->setFrameSource(analyser.getFrameSourcePtr());
      analyser.process(firstFrame, lastFrame, uiCallback);
    }
    catch (std::exception& e) {
      throw AnalysisException("Failed to process " + inputFilename + ": " + e.what());
    }
    catch (...) {
      // throw AnalysisException("Something went terribly and utterly unexpectedly wrong during "
      //                         "the processing phase!");
      throw;
    }


    // feature extraction
    size_t fileNumber = 0;
    BlackBoardPointer<FeatureVector> featureVectorPtr;
    FeatureVector* featureVector = nullptr;
    FeatureVector dummy;
    if (analyser.getBlackBoard().has(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY)) {
      featureVectorPtr = analyser.getBlackBoard().get<FeatureVector>(FEATURECREATOR_FEATUREVECTOR_BLACKBOARD_ENTRY);
      featureVector = &*featureVectorPtr;
    }
    else
      featureVector = &dummy;
      

    if (debug > 1)
      std::cerr << "slmotion: AnalysisController: storeFeatureData" << std::endl;;
    slio->storeFeatureData(*featureVector, fileNumber++,
                           analyser.getFrameSource().getLabel(),
                           analyser.getBlackBoard());

    if (elanFilename.length() > 0 && 
        annTemplateFilename.length() > 0) {
      csvToEafConverter.convert(featureVector->toDoubleVectorDeque(), 
                                annTemplateFilename, 
                                getInitialisedFieldNameMap(),
                                elanFilename, inputFilename);
    }
    if (outTierXmlFile != "") {
      // assert(returnList.size() == 1);
      csvToEafConverter.convertVR(featureVector->toDoubleVectorDeque(),
                                  annTemplateFilename, 
                                  getInitialisedFieldNameMap(),
                                  outTierXmlFile);
    }

    if (storeImages) {
      uiCallback->setCurrentComponentName("Video output");
      uiCallback->setCurrentComponentIndex(0);
      uiCallback->setNComponents(1);
      for (size_t frnumber = firstFrame; frnumber < lastFrame; ++frnumber) {
        visualise(analyser, frnumber, analyser.getFrameSource().getLabel(), 
                  storeImages);
        (*uiCallback)((frnumber+1-firstFrame)*100./(lastFrame-firstFrame));
      }
    }
  }



  bool AnalysisController::handleInput() {
    char key = cv::waitKey(pauseMode ? 0 : 20);       

    switch(key) {
    case 'P':
    case 'p':
      pauseMode = !pauseMode;
      return false;
    
    case ' ':
      cv::waitKey(0);
      return false;

    case 'q':
    case 'Q':
      return true;

    default:
      return false;
    }
  }



  void AnalysisController::visualise(Analyser& anal, size_t frnumber, 
                                     const std::string& label, 
                                     bool storeImages) const {
    Mat temp = anal.visualise(frnumber);

    // if (!disableAnnotationBar) {
    // temp.resize(temp.rows + ANNBARHEIGHT);
    // temp(cv::Rect(0, temp.rows-ANNBARHEIGHT, 
    //               temp.cols, ANNBARHEIGHT)).setTo(cv::Scalar::all(0));
    //   anal.getVisualiser()->addAnnotationBar(temp, frnumber, label,
    //                                          anal.getBlackBoard());
    // }

    if (storeImages) 
      anal.getSlio()->storeImage(temp, frnumber, label, anal.getBlackBoard());
  }



  void AnalysisController::visualise(Analyser& analyser, // vector<Analyser>& analysers, 
                                     size_t frnumber,
				     const std::string& label,
				     bool storeImages) {
    // corresponding out-frames
    // vector<Mat> outFrames(analysers.size());

    // for (size_t i = 0; i < analysers.size(); ++i) 
    //   outFrames[i] = analysers[i].visualise(frnumber);     
    cv::Mat outFrame = analyser.visualise(frnumber);

    // combination time
    // Mat temp = visualiser->combineVisuals(outFrames, frnumber, label,
    //     				  analysers.front().getBlackBoard());
    if (storeImages)
      slio->storeImage(outFrame, frnumber, label, analyser.getBlackBoard());
  }
}
