#include <list>
#include <boost/lexical_cast.hpp>
#include "AsmTracker.hpp"
#include "Asm.hpp"
#include "configuration.hpp"
#include "BlackBoardDumpWriter.hpp"

using std::list;
using std::cerr;
using std::endl;
using cv::Mat;
using cv::Point;
using cv::Rect;
using std::string;
using cv::Scalar;
using boost::program_options::value;
using std::string;
using cv::waitKey;
using cv::Point2d;

namespace slmotion {
  extern int debug;


  
  static AsmTracker DUMMY(true);

  AsmTracker::AsmTracker(bool) : Component(true) { 
    BlackBoardDumpWriter::registerAnyWriter<Asm>();
    BlackBoardDumpWriter::registerAnyWriter<Asm::Instance>();

    BlackBoardDumpWriter::registerDumbWriter<Pdm>([](BlackBoardDumpWriter* const w, std::ostream& ofs, const Pdm& data) {
        // format:
        // [uint64_t][Mat][Mat][Mat][Point2d]
        //     |       |    |    |      |
        //     |       |    |    |      +---mean anchor
        //     |       |    |    +---eigenvalues
        //     |       |    +---eigenvectors
        //     |       +---mean shape    
        //     +---number n of landmarks   
        cv::Mat meanShape = data.generateShape(cv::Mat(0,0,CV_64FC1));
        assert(meanShape.rows > 0 && meanShape.rows % 2 == 0 && 
               meanShape.cols == 1);
        uint64_t nLandmarks = meanShape.rows/2;
        w->dumbWrite(ofs, nLandmarks);
        w->dumbWrite(ofs, meanShape);
        w->dumbWrite(ofs, data.getEigenVectors());
        w->dumbWrite(ofs, data.getEigenValues());
        w->dumbWrite(ofs, data.getMeanAnchor());
      });

    BlackBoardDumpWriter::registerSizeComputer<Pdm>([](const BlackBoardDumpWriter* const w, const Pdm& pdm) {
        // format:
        // [uint64_t][Mat][Mat][Mat][Point2d]
        //     |       |    |    |      |
        //     |       |    |    |      +---mean anchor
        //     |       |    |    +---eigenvalues
        //     |       |    +---eigenvectors
        //     |       +---mean shape    
        //     +---number n of landmarks   
        return sizeof(uint64_t) + 
          w->getSize(pdm.getMeanAnchor()) + 
          w->getSize(pdm.generateShape(cv::Mat(0,0,CV_64FC1))) +
          w->getSize(pdm.getEigenVectors()) +
          w->getSize(pdm.getEigenValues());
      });

    BlackBoardDumpWriter::registerDumbWriter<Asm::Instance>([](BlackBoardDumpWriter* const w, std::ostream& ofs, const Asm::Instance& data) {
        // the size varies depending if the PDM needs to be stored. The PDM is 
        // stored ONLY with the first time it is referenced by an ASM object. That
        // is, the format is as follows:
        // [uint64_t][uint8_t][?PDM][Mat][PoseParameter][Mat]
        //     |         |       |    |        |          |
        //     |         |       |    |        |          +---landmarks
        //     |         |       |    |        +---pose parameters
        //     |         |       |    +---shape parameter
        //     |         |       +---PDM if necessary
        //     |         +---Bool, true if PDM should be stored
        //     +---PDM pointer
        const Pdm& pdm = data.getPdm();
        const void* pdmPointer = &pdm;
        uint64_t pdmPointer64 = reinterpret_cast<uintptr_t>(pdmPointer);
        w->dumbWrite(ofs, pdmPointer64);
        uint8_t shouldStorePdm = !w->hasStoredPointer(pdmPointer);
        w->dumbWrite(ofs, shouldStorePdm);
        if (shouldStorePdm)
          w->dumbWrite(ofs, &pdm);
        w->dumbWrite(ofs, data.getBt());
        w->dumbWrite(ofs, data.getPose());
        w->dumbWrite(ofs, data.getLandmarks());
      });

    
    BlackBoardDumpWriter::registerSizeComputer<Asm::Instance>([](const BlackBoardDumpWriter* const w, 
                                                                 const Asm::Instance& instance) {
        // the size varies depending if the PDM needs to be stored. The PDM is 
        // stored ONLY with the first time it is referenced by an ASM object. That
        // is, the format is as follows:
        // [uint64_t][uint8_t][?PDM][Mat][PoseParameter][Mat]
        //     |         |       |    |        |          |
        //     |         |       |    |        |          +---landmarks
        //     |         |       |    |        +---pose parameters
        //     |         |       |    +---shape parameter
        //     |         |       +---PDM if necessary
        //     |         +---Bool, true if PDM should be stored
        //     +---PDM pointer
        const Pdm& pdm = instance.getPdm();
        const void* pdmPointer = &pdm;
        uint8_t needPdm = !w->hasStoredPointer(pdmPointer);
        return sizeof(uint64_t) + sizeof(uint8_t) + 
          w->getSize(instance.getBt()) + w->getSize(instance.getPose()) +
          w->getSize(instance.getLandmarks()) +
          (needPdm ? w->getSize(&pdm) : 0);
      });

    BlackBoardDumpWriter::registerSizeComputer<Asm>([](const BlackBoardDumpWriter* const w, const Asm& data) {
        const Pdm& pdm = data.getPdm();
        const void* pdmPointer = &pdm;
        uint8_t needPdm = !w->hasStoredPointer(pdmPointer);
        return 2*sizeof(uint64_t) + sizeof(uint8_t) + sizeof(double)
          + (needPdm ? w->getSize(pdm) : 0);
      });

    BlackBoardDumpWriter::registerConditionalStoredPointersFunction<Asm::Instance>([](BlackBoardDumpWriter const * const w, const Asm::Instance& i) {
        if (!w->hasStoredPointer(&i.getPdm()))
          return std::set<const void*> { &i.getPdm() };
        else
          return std::set<const void*>();
      });

    BlackBoardDumpWriter::registerConditionalStoredPointersFunction<Asm>([](BlackBoardDumpWriter const * const w, const Asm& i) {
        if (!w->hasStoredPointer(&i.getPdm()))
          return std::set<const void*> { &i.getPdm() };
        else
          return std::set<const void*>();
      });


    BlackBoardDumpWriter::registerDumbWriter<Asm>([](BlackBoardDumpWriter* const w, std::ostream& ofs, const Asm& data) {
        /**
         * format:
         * [uint64_t][uint8_t][PDM?][uint64_t][double]
         *     |         |      |       |         |
         *     |         |      |       |         +---max shape parameter 
         *     |         |      |       |             deviation
         *     |         |      |       +---number of components
         *     |         |      +---PDM if necessary
         *     |         +---boolean: true if PDM is stored as well
         *     +---PDM pointer                     
         */
        const Pdm& pdm = data.getPdm();
        const void* pdmPointer = &pdm;
        uint64_t pdmPointer64 = reinterpret_cast<uintptr_t>(pdmPointer);
        w->dumbWrite(ofs, pdmPointer64);
        uint8_t shouldStorePdm = !w->hasStoredPointer(pdmPointer);
        w->dumbWrite(ofs, shouldStorePdm);
        if (shouldStorePdm)
          w->dumbWrite(ofs, &pdm);
        uint64_t nComponents = data.getNComponents();
        w->dumbWrite(ofs, nComponents);
        w->dumbWrite(ofs, data.getMaxShapeParameterDeviation());
      });

    BlackBoardDumpReader::registerUnDumper<Asm::Instance>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        uint64_t pdmPointer;
        w->dumbRead(ifs, pdmPointer);
        uint8_t shouldPdmHaveBeenStored;
        w->dumbRead(ifs, shouldPdmHaveBeenStored);
        if (shouldPdmHaveBeenStored == w->hasNewSharedPtr(pdmPointer))
          throw IOException("Corrupt dump file: A PDM that should not exist "
                            "does indeed exist, or should exist but does not");
        if (shouldPdmHaveBeenStored)
          w->undumpSharedPointer<Pdm>(ifs, pdmPointer);
        assert(w->hasNewSharedPtr(pdmPointer));
        std::shared_ptr<Pdm> pdm = w->getNewSharedPtr<Pdm>(pdmPointer);
        cv::Mat bt, landmarks;
        w->dumbRead(ifs, bt);
        Pdm::PoseParameter pose;
        w->dumbRead(ifs, pose);
        w->dumbRead(ifs, landmarks);
        return Asm::Instance(pdm, bt, pose, landmarks);
      });

    BlackBoardDumpReader::registerUnDumper<Pdm>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        // format:
        // [uint64_t][Mat][Mat][Mat][Point2d]
        //     |       |    |    |      |
        //     |       |    |    |      +---mean anchor
        //     |       |    |    +---eigenvalues
        //     |       |    +---eigenvectors
        //     |       +---mean shape    
        //     +---number n of landmarks   
        uint64_t nLandmarks;
        cv::Mat meanShape, eigenVectors, eigenValues;
        cv::Point2d meanAnchor;
        w->dumbRead(ifs, nLandmarks);
        w->dumbRead(ifs, meanShape);
        w->dumbRead(ifs, eigenVectors);
        w->dumbRead(ifs, eigenValues);
        w->dumbRead(ifs, meanAnchor);
        return Pdm(nLandmarks, meanShape, eigenVectors, 
                   eigenValues, meanAnchor);
      });

    BlackBoardDumpReader::registerUnDumper<Asm>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        /**
         * format:
         * [uint64_t][uint8_t][PDM?][uint64_t][double]
         *     |         |      |       |         |
         *     |         |      |       |         +---max shape parameter 
         *     |         |      |       |             deviation
         *     |         |      |       +---number of components
         *     |         |      +---PDM if necessary
         *     |         +---boolean: true if PDM is stored as well
         *     +---PDM pointer                     
         */
        uint64_t pdmPointer;
        uint8_t shouldPdmHaveBeenStored;
        w->dumbRead(ifs, pdmPointer);
        w->dumbRead(ifs, shouldPdmHaveBeenStored);

        if (shouldPdmHaveBeenStored == w->hasNewSharedPtr(pdmPointer))
          throw IOException("Corrupt dump file: A PDM that should not exist "
                            "does indeed exist, or should exist but does not");
        if (shouldPdmHaveBeenStored)
          w->undumpSharedPointer<Pdm>(ifs, pdmPointer);
        assert(w->hasNewSharedPtr(pdmPointer));
        std::shared_ptr<Pdm> pdm = w->getNewSharedPtr<Pdm>(pdmPointer);
        uint64_t componentCount;
        w->dumbRead(ifs, componentCount);
        double maxShapeParameterDeviation;
        w->dumbRead(ifs, maxShapeParameterDeviation);
        return Asm(pdm, componentCount, maxShapeParameterDeviation);
      });

  }

  boost::program_options::options_description AsmTracker::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("AsmTracker.nlandmarks", value<int>()->default_value(60),
       "The number of landmark points that are placed evenly around the object when creating the point distribution model.")
      ("AsmTracker.alignthreshold", value<double>()->default_value(0),
       "When aligning vectors during Point Distribution Model creation, use this threshold for determining convergence.")
      ("AsmTracker.sobelaperturesize", value<string>()->default_value("scharr"),
       "Possible values for this option include <int> and 'schar'. Sets "
       "the aperture size for the Sobel operator when computing intensity "
       "gradients for the image. Valid values are 1, 3, 5, and 7. Setting "
       "this option to 'scharr' will use a 3x3 Sobel operator with a "
       "Scharr filter.")
      ("AsmTracker.maxlookdistance", value<int>()->default_value(30),
       "Sets the maximum distance for target landmarks in relation to the "
       "mean shape in pixels.")
      ("AsmTracker.convergencethreshold", value<double>()->default_value(1.0),
       "Sets a threshold for deciding if the ASM has converged. Convergence"
       " is decided by the sum of absolute values of elements in vector dbt"
       " such that the shape of the PDM x is approximately mean(x) + "
       "Pt*(bt + dbt) where pt is a 2N×t PCA matrix)")
      ("AsmTracker.intensityfactor", value<double>()->default_value(5.0),
       "When doing ASM fitting and scanning for new target landmarks, the "
       "gradient value must exceed the value at the old landmark by the "
       "given factor, or the target is not considered clear enough.")
      ("AsmTracker.minintensity", value<string>()->default_value("mean"),
       "When doing ASM fitting and scanning for new target landmarks, the "
       "absolute value of the gradient at the new target must exceed this "
       "figure. If no such point is found, then a new target is not set. "
       "Alternatively, you can use 'mean', which will set the minimum "
       "intensity to the mean absolute value, computed over the whole "
       "gradient map.")
      ("AsmTracker.pcacomponentcount", value<int>()->default_value(3),
       "Number of PCA components to use in the fitting")
      ("AsmTracker.maxiterations", value<int>()->default_value(20),
       "Maximum number of iterations allowed when performing ASM fitting")
      ("AsmTracker.scaleapproximationfactor", value<string>()->default_value("1.0:1.0"),
       "A pair of numbers a:b which indicate the amount by which the body "
       "part may exceed the dimensions of its blobs in its approximate "
       "initial pose, i.e. setting this to 1.2:1.4 would mean that the body"
       " part may exceed the width of the blob by a factor of 1.2 and "
       "height by 1.4. Setting either of these zero will disable scaling "
       "approximation altogether.")
      ("AsmTracker.maxshapeparameterdeviation", value<double>()->default_value(3.0),
       "The shape of the body part is determined by a t-vector so that the "
       "shape is determined as a sum of the mean shape and the product of "
       "the transformation matrix and the shape vector. Each component in "
       "the shape vector is associated with an eigenvalue, computed from "
       "the covariance matrix of original sample PDMs. This parameter sets "
       "upper and lower bounds for these components. More precisely, the "
       "absolute values of the components in the particular shape vector "
       "may not exceed this parameter * the square root of their associated"
       " eigenvalues. E.g. if the parameter were set to 3.0, and a "
       "component had an eigenvalue, say, 16, the corresponding component "
       "in the shape vector could vary between -12 and 12.")
      ("AsmTracker.shapehistorysize", value<size_t>()->default_value(0),
       "When constructing the mean PDM, use this many previous shapes as " 
       "reference. Setting this 0 will use as many shapes as possible.")
      ("AsmTracker.blackout", value<bool>()->default_value(true),
       "If enabled, black out parts of the image deemed unimportant when "
       "performing ASM fitting.")
      ("AsmTracker.initialposeestimationmethod", value<string>()->default_value("poseandshapehistory"),
       "Sets the method used for selecting the initial pose. Possible "
       "values are listed below:\n"
       "- probeonly           : Always probe for the pose\n"
       "- posehistory         : Use pose values from previous frame (given "
       "that their goodness exceeds the set threshold)\n"
       "- poseandshapehistory : Like above, but also set the initial value "
       "of the particular shape vector to its previous value\n\n"
       "If shapehistorysize > 0, setting this option to anything other than"
       " probeonly may result in unexpected behaviour.")
      ("AsmTracker.initialposeestimationgoodnessthreshold", value<double>()->default_value(0.7),
       "If posehistory or poseandshapehistory is enabled above, their "
       "goodness is checked against a binary mask for the body part being "
       "fitted. Goodness is defined as a fraction of (count of pixels in "
       "the intersection of the pixels enclosed within a polygon limited by"
       " landmarks of the old shape in its old pose and the pixels in the "
       "reference mask) / max(number of pixels in the polygonm, number of "
       "pixels in the reference frame). The denominator has the max there "
       "to prevent huge predicted shapes from overcoming smaller reference "
       "shapes (the intersection could be perfect, yet the shape would "
       "cover a lot more area than it should)")
      ("AsmTracker.equalisehistogram", value<bool>()->default_value(false),
       "Whether to perform histogram equalisation on the grayscale image "
       "before computing intensity gradients.")
      ("AsmTracker.targettype", value<string>()->default_value("absolute"),
       "When choosing new target landmarks, choose those pixels whose "
       "intensity gradients:\n"
       "- absolute    : have the greatest absolute value\n"
       "- negative    : have the most negative value (ignoring any positive"
       "pixels)\n"
       "- positive    : have the most positive value\n")
      ("AsmTracker.targetmethod", value<string>()->default_value("directional"),
       "Selects the method used for choosing new target landmarks. Possible"
       "values include:\n"
       "- gradient    : use the magnitude of the intensity gradient\n"
       "- directional : computes intensity gradients separately for x and y"
       "axes and projects the gradient vector on the normal. Then, it is "
       "assumed that intensity decreases as we move away from the body "
       "part, so pick the pixel with highest intensity (as decided above)\n"
       "- canny       : uses the Canny edge detector\n")
      ("AsmTracker.gradientmethod", value<string>()->default_value("sobel"),
       "Sets the method used for approximating the intensity gradient. "
       "Possible values include:\n"
       "- sobel      : The Sobel operator (default), also has a special "
       "option to set the aperture size (see above). Can be used always.\n"
       "- laplacian  : Uses the OpenCV laplacian operator (which uses the "
       "Sobel operator) with the aperture size set according to the Sobel "
       "operator aperture size option. Laplacian operators cannot be used "
       "with directional targets.\n"
       "- laplacian4 : A 4-neighbourhood Laplacian\n"
       "- laplacian8 : An 8-neighbourhood Laplacian\n"
       "- roberts    : Uses the Roberts operator\n"
       "- prewitt    : The Prewitt operator\n"
       "- robinson   : The Robinson operator\n"
       "- kirsch     : The Kirsch operator")
      ("AsmTracker.cannythreshold1", value<double>()->default_value(150),
       "Canny hysteresis parameter 1")
      ("AsmTracker.cannythreshold2", value<double>()->default_value(100),
       "Canny hysteresis parameter 2")
      ("AsmTracker.alignmaxiterations", value<unsigned int>()->default_value(10),
       "Maximum number of iterations allowed when aligning vectors during "
       "Point Distribution Model creation");

    return opts;
 // return configuration::ConfigurationFileOptionsDescription {
 //      opts, "This section controls the behaviour of the point distribution "
 //        "models, how they are created, and how they are used when "
 //        "performing ASM fitting" };
  }

  Component* AsmTracker::createComponentImpl(const boost::program_options::variables_map& configuration, 
                                             BlackBoard* blackBoard, FrameSource* frameSource) const {
    AsmTracker at(blackBoard, frameSource);
    if (configuration.count("AsmTracker.nlandmarks"))
      at.setNLandmarks(configuration["AsmTracker.nlandmarks"].as<int>());

    if (configuration.count("AsmTracker.pcacomponentcount"))
      at.setPcaComponentCount(configuration["AsmTracker.pcacomponentcount"].as<int>());

    if (configuration.count("AsmTracker.maxshapeparameterdeviation"))
      at.setMaxShapeParameterDeviation(configuration["AsmTracker.maxshapeparameterdeviation"].as<double>());

    return new AsmTracker(at);
  }

  
  bool AsmTracker::train(size_t firstFrame, size_t lastFrame, 
                         UiCallback* uiCallback) {
    list<BodyPart> leftHands, rightHands, heads;
    std::vector<Mat> leftHandLandmarks; 
    std::vector<Mat> rightHandLandmarks;
    std::vector<Mat> headLandmarks;

    // Pdm headPdm; ///< Model for ASM fitting
    // Pdm leftHandPdm; ///< Model for ASM fitting
    // Pdm rightHandPdm; ///< Model for ASM fitting

    if (slmotion::debug > 1)
      cerr << "Grouping body parts, removing unwanted ones." << endl;

    for (size_t frnumber = firstFrame; frnumber < lastFrame; ++frnumber) {

      BlackBoardPointer<list<BodyPart> > historyParts = getBlackBoard().get<list<BodyPart>>(frnumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY);

      for (list<BodyPart>::const_iterator it = historyParts->cbegin();
           it != historyParts->cend(); it++) {
        switch(it->getIdentity()) {

        case BodyPart::LEFT_HAND:
          leftHands.push_back(*it);
          if (slmotion::debug > 2)
            cerr << "Found a left hand (" << leftHands.size() << ")" << endl;
          
        case BodyPart::RIGHT_HAND:
          rightHands.push_back(*it);
          
          if (slmotion::debug > 2)
            cerr << "Found a right hand (" << rightHands.size() << ")" << endl;
	
        case BodyPart::HEAD:
          heads.push_back(*it);
          
          if (slmotion::debug > 2)
            cerr << "Found a head (" << heads.size() << ")" << endl;

          break;

        default:
          break; // ignore others
        }        
      }

      if (uiCallback != nullptr &&
          ((*uiCallback)(50./4.*(frnumber-firstFrame+1)/(lastFrame-firstFrame))) == false)
        return false;
    }

    // else if (slmotion::debug > 1) {
    //   cerr << "Done! Found " << historyParts.size() << " body parts in total." << endl;

    if (slmotion::debug > 1) {
      cerr << "Tally of individual body parts: " << endl;
      cerr << heads.size() << " heads, " << leftHands.size() <<
        " left hands, and " << rightHands.size() << " right hands." << endl;
    }
    

    // Assign landmark points at even distances around the objects

    // Set estimates to where they are likely at the start of the video
    if (leftHands.size() > 0)
      getBlackBoard().set(firstFrame, 
                          ASM_TRACKER_LEFT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY,
                          leftHands.front().getBlob().getBottomMost());
    if (rightHands.size() > 0)
      getBlackBoard().set(firstFrame, 
                          ASM_TRACKER_RIGHT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY,
                          rightHands.front().getBlob().getBottomMost());
    if (heads.size() > 0)
      getBlackBoard().set(firstFrame, 
                          ASM_TRACKER_HEAD_ANCHOR_GUESS_BLACKBOARD_ENTRY,
                          heads.front().getBlob().getBottomMost());

    // for (list<BodyPart>::const_iterator it = leftHands.begin();
    //      it != leftHands.end(); ++it) 
    //   leftHandPDMs.push_back(PDM(it->getBlob().toMatrix(),
    //                              it->getBlob().getBottomMost(),
    //                              nlandmarks));

    // for (list<BodyPart>::const_iterator it = rightHands.begin();
    //      it != rightHands.end(); ++it) 
    //   rightHandPDMs.push_back(PDM(it->getBlob().toMatrix(),
    //                               it->getBlob().getBottomMost(),
    //                               nlandmarks));

    // for (list<BodyPart>::const_iterator it = heads.begin();
    //      it != heads.end(); ++it) 
    //   headPDMs.push_back(PDM(it->getBlob().toMatrix(),
    //                          it->getBlob().getBottomMost(),
    //                          nlandmarks));

    std::vector<Blob> leftHandBlobs;
    std::vector<Blob> rightHandBlobs;
    std::vector<Blob> headBlobs;
    for (auto it = leftHands.cbegin(); it != leftHands.cend(); ++it)
      leftHandBlobs.push_back(it->getBlob());
    for (auto it = rightHands.cbegin(); it != rightHands.cend(); ++it)
      rightHandBlobs.push_back(it->getBlob());
    for (auto it = heads.cbegin(); it != heads.cend(); ++it)
      headBlobs.push_back(it->getBlob());

    getBlobLandmarks(leftHandBlobs, nlandmarks, leftHandLandmarks);
    getBlobLandmarks(rightHandBlobs, nlandmarks, rightHandLandmarks);
    getBlobLandmarks(headBlobs, nlandmarks, headLandmarks);


    // PDM::approximateAlign(headPDMs, pdmAlignThreshold,
    //                       pdmAlignMaxIterations, meanHead);

    // PDM::approximateAlign(leftHandPDMs, pdmAlignThreshold, 
    //                       pdmAlignMaxIterations, meanLeftHand);

    // PDM::approximateAlign(rightHandPDMs, pdmAlignThreshold, 
    //                       pdmAlignMaxIterations, meanRightHand);
    if (debug > 1)
      std::cout << "Training head ASM...";
    if (headBlobs.size() > 0)
      getBlackBoard().set(ASM_TRACKER_HEAD_ASM, 
                          Asm(headBlobs, nlandmarks, pcaComponentCount, 
                              maxShapeParameterDeviation));
    if (debug > 1)
      std::cout << " OK!" << std::endl;

    if (uiCallback != nullptr &&
        (*uiCallback)(2.*50./4.) == false)
      return false;

    if (debug > 1)
      std::cerr << "Training left hand ASM...";
    if (rightHandBlobs.size() > 0)
      getBlackBoard().set(ASM_TRACKER_LEFT_HAND_ASM, 
                          Asm(rightHandBlobs, nlandmarks, pcaComponentCount,
                              maxShapeParameterDeviation));
    if (debug > 1)
      std::cerr << " OK!" << std::endl;

    if (uiCallback != nullptr &&
        (*uiCallback)(3.*50./4.) == false)
      return false;

    if (debug > 1)
      std::cerr << "Training right hand ASM...";
    if (leftHandBlobs.size() > 0)
      getBlackBoard().set(ASM_TRACKER_RIGHT_HAND_ASM, 
                          Asm(leftHandBlobs, nlandmarks, pcaComponentCount,
                              maxShapeParameterDeviation));
    if (debug > 1)
      std::cerr << " OK!" << std::endl;

    if (uiCallback != nullptr &&
        (*uiCallback)(50.) == false)
      return false;

    // const Pdm& pdm = getBlackBoard().get<Asm>(ASM_TRACKER_RIGHT_HAND_ASM)->getPdm();
    // Mat eigenValuesSqrt(pdm.getEigenValues().size(),
    //                     pdm.getEigenValues().type());
    // cv::sqrt(pdm.getEigenValues(), eigenValuesSqrt);
    // const int t = pcaComponentCount;
    // double d = -maxShapeParameterDeviation;
    // for (int k = 0; k < 10; ++k) {
    //   Mat bt(t, 1, CV_64FC1);
    //   for (int j = 0; j < t; ++j)
    //     bt.at<double>(j) = d * eigenValuesSqrt.at<double>(j);
    //   Mat shape = pdm.generateShape(bt);
    //   Mat temp(576, 720, CV_8UC1, Scalar::all(0));
    //   for (int i = 0; i < shape.rows; i += 2)
    //     circle(temp, cv::Point2d(shape.at<double>(i), shape.at<double>(i+1)) +
    //            Point2d(200, 200), 3,
    //            Scalar::all(255));
    //   imshow("", temp);
    //   waitKey(0);
    //   d += maxShapeParameterDeviation / 5.;
    // }

    isTrained = true;

    return true;
  }



  void AsmTracker::process(frame_number_t frnumber) {
    assert(isTrained);
    // const Mat& frameCopy = getFrameSource()[frnumber];

    BlackBoardPointer<list<BodyPart> > currentBodyParts = getBlackBoard().get<list<BodyPart>>(frnumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY);

    Mat headGuessMask, rightHandGuessMask, leftHandGuessMask;
    const Mat& frame = getFrameSource()[frnumber];
    createGuessMasks(*currentBodyParts, leftHandGuessMask, 
                     rightHandGuessMask, headGuessMask,
                     frame.size());
    
    auto fitInstance = [&frame, &frnumber](BlackBoard& blackBoard,
                                           const std::string& asmString,
                                           const std::string& instanceString,
                                           const cv::Mat& guessMask,
                                           const Asm& asmi) {
      if (blackBoard.has(asmString)) {
        Asm::Instance* oldInstance = NULL;
        if (frnumber > 0 && 
            blackBoard.has(frnumber, instanceString))
          oldInstance = 
            &*blackBoard.get<Asm::Instance>(frnumber, 
                                           instanceString);
        Asm::Instance instance(oldInstance ? 
                               asmi.fit(frame, guessMask, 
                                        *oldInstance) :
                               asmi.fit(frame, guessMask));
        blackBoard.set(frnumber, instanceString, instance);
        // Mat temp(frame.clone());
        // instance.drawLine(temp, 1, cv::Scalar(0,255,0));
        // cv::imshow("", temp);
        // cv::waitKey(0);
      }
    };

    if (slmotion::debug > 1)
      cerr << "Fitting head ASM" << endl;
    
    if (getBlackBoard().has(ASM_TRACKER_HEAD_ASM))
      fitInstance(getBlackBoard(), ASM_TRACKER_HEAD_ASM,
                  ASM_TRACKER_HEAD_ASM_INSTANCE, headGuessMask,
                  *getBlackBoard().get<Asm>(ASM_TRACKER_HEAD_ASM));

    if (getBlackBoard().has(ASM_TRACKER_LEFT_HAND_ASM))
      fitInstance(getBlackBoard(), ASM_TRACKER_LEFT_HAND_ASM,
                  ASM_TRACKER_LEFT_HAND_ASM_INSTANCE, leftHandGuessMask, 
                  *getBlackBoard().get<Asm>(ASM_TRACKER_LEFT_HAND_ASM));
    
    if (getBlackBoard().has(ASM_TRACKER_RIGHT_HAND_ASM))
      fitInstance(getBlackBoard(), ASM_TRACKER_RIGHT_HAND_ASM,
                  ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE, rightHandGuessMask, 
                  *getBlackBoard().get<Asm>(ASM_TRACKER_RIGHT_HAND_ASM));
  }
  
  

  bool AsmTracker::processRangeImplementation(frame_number_t first, 
                                              frame_number_t last,
                                              UiCallback* uiCallback) {
    reset();
    if (!train(first, last, uiCallback))
      return false;

    for (frame_number_t frnumber = first; frnumber < last; ++frnumber) {
      if (debug > 1)
        std::cerr << frnumber - first + 1 << '/' 
                  << last - first;
      process(frnumber);
      if (uiCallback != NULL && !(*uiCallback)(50. + 50.*(frnumber-first+1)/(last-first)))
        return false;
    }

    return true;
  }



  void AsmTracker::reset() {
    isTrained = false;
  }



  void createGuessMasks(const std::list<BodyPart>& bodyParts,
                        cv::Mat& outLeftHandGuessMask, 
                        cv::Mat& outRightHandGuessMask,
                        cv::Mat& outHeadGuessMask,
                        const cv::Size& frameSize) {
    size_t lhCount, rhCount, headCount;
    lhCount = rhCount = headCount = 0;
    const BodyPart *lh, *rh, *head; // body part pointers
    lh = rh = head = NULL;

    for (auto it = bodyParts.cbegin(); it != bodyParts.cend(); ++it) {
      if (!it->isA(BodyPart::HEAD) &&
          !it->isA(BodyPart::LEFT_HAND) &&
          !it->isA(BodyPart::RIGHT_HAND))
        throw AsmException("Invalid body part!");
        
      if (it->isA(BodyPart::HEAD)) {
        ++headCount;
        head = &*it;
      }

      if (it->isA(BodyPart::LEFT_HAND)) {
        ++lhCount;
        lh = &*it;
      }

      if (it->isA(BodyPart::RIGHT_HAND)) {
        ++rhCount;
        rh = &*it;
      }
    }

    if (lhCount != 1 || rhCount != 1 || headCount != 1)
      throw AsmException("Invalid body part count: one entry for each body part was expected; three in total. However, " + 
                         boost::lexical_cast<string>(lhCount) + " left hands, " +
                         boost::lexical_cast<string>(rhCount) + 
                         " right hands, and " +
                         boost::lexical_cast<string>(headCount) +
                         " heads were found.");

    outLeftHandGuessMask = lh->getBlob().toMatrix(frameSize);
    outRightHandGuessMask = rh->getBlob().toMatrix(frameSize);
    outHeadGuessMask = head->getBlob().toMatrix(frameSize);

    // centroid locations
    Point lhCent = lh->getBlob().getCentroid();
    Point rhCent = rh->getBlob().getCentroid();
    Point headCent = head->getBlob().getCentroid();

    // split masks where necessary
    //
    // 1. in case the head is merged with either of the hands, chop it off
    if (lh->isA(BodyPart::HEAD)) 
      outLeftHandGuessMask(Rect(Point(0, 0), 
                                Point(frameSize.width, lhCent.y))) =
        Scalar::all(0);

    if (rh->isA(BodyPart::HEAD)) 
      outRightHandGuessMask(Rect(Point(0, 0), 
                                 Point(frameSize.width, rhCent.y)))
        = Scalar::all(0);

    // 2. In case two hands are merged, chop the other one off
    if (lh->isA(BodyPart::RIGHT_HAND)) 
      outLeftHandGuessMask(Rect(Point(lhCent.x, 0), 
                                 Point(frameSize.width, frameSize.height)))
        = Scalar::all(0);

    if (rh->isA(BodyPart::LEFT_HAND)) 
      outRightHandGuessMask(Rect(Point(0, 0), 
                                 Point(rhCent.x, frameSize.height))) = 
        Scalar::all(0);

    // 3. In case the head is merged with a hand or two, chop the hands off
    if (head->isA(BodyPart::LEFT_HAND) || head->isA(BodyPart::RIGHT_HAND))
      outHeadGuessMask(Rect(Point(0, headCent.y), 
                            Point(frameSize.width, frameSize.height))) = 
        Scalar::all(0);
  }
}

