#ifndef SLMOTION_ASM_TRACKER
#define SLMOTION_ASM_TRACKER

#include <string>
#include "Component.hpp"
#include "BodyPartCollector.hpp"

namespace slmotion {
  // cv::Points
  const std::string ASM_TRACKER_LEFT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY =
    "lefthandanchorguess"; 
  const std::string ASM_TRACKER_RIGHT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY =
    "righthandanchorguess";
  const std::string ASM_TRACKER_HEAD_ANCHOR_GUESS_BLACKBOARD_ENTRY =
    "headanchorguess";

  // Asms
  const std::string ASM_TRACKER_HEAD_ASM = "headasm";
  const std::string ASM_TRACKER_LEFT_HAND_ASM = "lefthandasm";
  const std::string ASM_TRACKER_RIGHT_HAND_ASM = "righthandasm";

  // Asm::Instances
  const std::string ASM_TRACKER_HEAD_ASM_INSTANCE = "headasminstance";
  const std::string ASM_TRACKER_LEFT_HAND_ASM_INSTANCE = "lefthandasminstance";
  const std::string ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE = "righthandasminstance";


  /**
   * This component tracks relevant body parts over the video.
   */
  class AsmTracker : public Component {
  public:
    /**
     * The nice constructor
     */
    AsmTracker(BlackBoard* blackBoard, FrameSource* frameSource) :
      Component(blackBoard, frameSource), isTrained(false),
      // pdmAlignThreshold(0.01),
      pdmAlignMaxIterations(10),
      nlandmarks(30),
      pcaComponentCount(3),
      maxShapeParameterDeviation(3.0) { }

    /**
     * Fits the models to the given frame. Beware, the model must have been
     * trained first, so calling this function manually may not be desired.
     */
    void process(frame_number_t frameNumber);

    /**
     * Resets the models
     */
    void reset();



    /**
     * Returns a set of property keys which must be found on the blackboard 
     * for each frame that shall be processed by the component (if 
     * available)
     */
    inline virtual property_set_t getRequirements() const {
      return property_set_t { FACEDETECTOR_BLACKBOARD_ENTRY,
          BODYPARTCOLLECTOR_BLACKBOARD_ENTRY };
    }



    /**
     * Returns a set of property keys which will be created or updated (i.e.
     * provided) by this component on the blackboard after processing has 
     * been performed
     */
    inline virtual property_set_t getProvided() const {
      return property_set_t {
        ASM_TRACKER_LEFT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY,
          ASM_TRACKER_RIGHT_HAND_ANCHOR_GUESS_BLACKBOARD_ENTRY,
          ASM_TRACKER_HEAD_ANCHOR_GUESS_BLACKBOARD_ENTRY,
          ASM_TRACKER_HEAD_ASM_INSTANCE,
          ASM_TRACKER_LEFT_HAND_ASM_INSTANCE,
          ASM_TRACKER_RIGHT_HAND_ASM_INSTANCE
          };
    }



    // inline void setPdmAlignThreshold(double d) {
    //   pdmAlignThreshold = d;
    // }



    inline void setPdmAlignMaxIterations(unsigned int i) {
      pdmAlignMaxIterations = i;
    }



    inline void setNLandmarks(int i) {
      nlandmarks = i;
    }



    // inline void setAsmFittingContext(const PDM::AsmFittingContext& src) {
    //   asmFittingContext = src;
    // }



    /**
     * Returns the name of the component
     */
    virtual std::string getComponentName() const {
      return "ASM Tracker";
    }

    virtual std::string getShortName() const {
      return "AsmTracker";
    }

    virtual std::string getShortDescription() const {
      return "Tracks the silhouettes of the hands and the head by the means of Active Shape Models, learned from individual, separate body parts located during a first pass, and then fits the models in the second pass";
    }



    virtual std::string getLongDescription() const {
      return "Tracks the silhouettes of the hands and the head by the means of Active Shape Models, learned from individual, separate body parts located during a first pass, and then fits the models in the second pass";
    }



    AsmTracker(bool);



    // virtual configuration::ConfigurationFileOptionsDescription getConfigurationFileOptionsDescription() const;
    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    /**
     * Resets models, goes through the range of frames and trains the 
     * models, then passes through the video again, and fits the models
     * 
     * @param uiCallback Possible user interface callback. The
     * implementation function should call this function every now and
     * then to notify the parent program of its status (current frame)
     * if applicable. This may also be ignored, and the function
     * should be able to handle a NULL pointer.  If the callback
     * function returns false, execution should be terminated
     * prematurely.
     *
     * @return True on successful termination, false otherwise (including 
     * false from callback)
     */
    bool processRangeImplementation(frame_number_t first = 0, 
                                    frame_number_t last = SIZE_MAX,
                                    UiCallback* uiCallback = NULL);



    /**
     * Goes through the video, and trains the model
     *
     * @return Returns false if the process should be cancelled
     * prematurely (from uicallback if nonzero and user requests
     * termination)
     */
    bool train(size_t firstFrame, size_t lastFrame, UiCallback* uiCallback);



    inline void setPcaComponentCount(int i) {
      pcaComponentCount = i;
    }

    inline void setMaxShapeParameterDeviation(double d) {
      maxShapeParameterDeviation = d;
    }



    bool isTrained; //< Set true when the models have been trained
    // double pdmAlignThreshold; ///< During PDM creation, when aligning vectors, use this value as criterion when testing for convergence

    unsigned int pdmAlignMaxIterations; ///< Maximum number of iterations allowed for vector alignment approximation (useful if threshold is rather low; the mean may get stuck to alter between two different shapes and threshold criterion is never met even though convergence has occurred)

    int nlandmarks; ///< The number of landmarks that are placed evenly around the object when creating the point distribution model

    int pcaComponentCount; ///< Number of PCA components
    double maxShapeParameterDeviation; ///< Determines the range of shape parameter values. Given component b_i, with a mean at 0, and associated eigenvalue lambda_i, the value is allowed to vary between -maxShapeParameterDeviation sqrt(lambda_i) and maxShapeParameterDeviation sqrt(lambda_i)
  };



  /**
   * Creates 'butchered' guess masks from the given set of body parts
   * The list must contain one and exactly one entry corresponding to
   * each body part, e.g. two entries: LEFTHAND_HEAD + RIGHTHAND.
   * However, LEFTHAND_HEAD, LEFT_RIGHT_HAND would be invalid since
   * there would be two entries corresponding to the left hand.
   *
   * @param frameSize Size of the input frame (also size of the output 
   * frame)
   */
  void createGuessMasks(const std::list<BodyPart>& bodyParts,
                        cv::Mat& outLeftHandGuessMask, 
                        cv::Mat& outRightHandGuessMask,
                        cv::Mat& outHeadGuessMask,
                        const cv::Size& frameSize);
}

#endif
