#ifndef SLMOTION_BLACKBOARD_DUMP_READER
#define SLMOTION_BLACKBOARD_DUMP_READER

#include "Component.hpp"
#include "exceptions.hpp"
#include <type_traits>

namespace slmotion {
  /**
   * Represents simple black board dump entry metadata
   */
  struct DumpEntry {
    uint64_t frameNumber;
    uint64_t dataType; // new version: type_info hash
    char key[33]; // the \0 at the end is omitted on writing
    uint64_t dataPointer;
  };

  /**
   * This component reads data written by the BlackBoardDumpWriter. Please 
   * see that class for more detailed description.
   */
  class BlackBoardDumpReader : public Component {
  public:
    BlackBoardDumpReader(bool);

    BlackBoardDumpReader(BlackBoard* blackBoard, FrameSource* frameSource) :
      Component(blackBoard, frameSource) { }

    virtual void process(frame_number_t frameNumber);

    virtual std::string getShortDescription() const {
      return "Black Board dump file reader";
    }

    virtual std::string getLongDescription() const {
      return "Black Board dump file reader. Reads data from dump files and stores it on the blac kboard.";
    }

    virtual property_set_t getRequirements() const {
      return property_set_t();
    }



    /**
     * Reads through all headers and lists everything
     */
    virtual property_set_t getProvided() const;




    virtual std::string getShortName() const {
      return "BlackBoardDumpReader";
    }

    virtual std::string getComponentName() const {
      return "BlackBoard Dump Reader";
    }

    virtual void reset() {
    }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const {
      return boost::program_options::options_description();
    }

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const;

    inline void setFilenames(const std::vector<std::string>& filenames) {
      this->filenames = filenames;
    }



    /**
     * Registers the type so that they can be undumped automagically as an any object
     */
    template<typename T>
    static void registerAnyUnDumper() {
      if (!anyUndumpFunctions)
        anyUndumpFunctions = new std::unordered_map<size_t, boost::any(*)(BlackBoardDumpReader* const, std::istream&) >();

      if (!anyUndumpFunctions->count(typeid(T).hash_code())) 
        (*anyUndumpFunctions)[typeid(T).hash_code()] = [](BlackBoardDumpReader* const w, std::istream& is)->boost::any {
          T t = w->undump<T>(is);
          return boost::any(t);
        };
    }

    /**
     * Registers the type so that they can be undumped automagically
     */
    template<typename T>
    static void registerUnDumper(T(*func)(BlackBoardDumpReader* const, std::istream&)) {
      if (!undumpFunctions)
        undumpFunctions = new std::unordered_map<size_t, std::shared_ptr<UnDumperBase> >();

      if (!undumpFunctions->count(typeid(T).hash_code())) 
        (*undumpFunctions)[typeid(T).hash_code()] = std::shared_ptr<UnDumperBase>(new UnDumper<T>(func));
    }

    
    /**
     * Simply reads the data in a dumb manner from the stream (to an array)
     */
    template<typename T, size_t N>
    void dumbRead(std::istream& ifs, T (&outData)[N] ) {
      if (undumpFunctions && undumpFunctions->count(typeid(T).hash_code())) 
        for (size_t i = 0; i < N; ++i)
          outData[i] = undump<T>(ifs);
      else if (!std::is_pod<T>::value) 
        throw IOException("The default implementation supports only POD types, "
                          "but " + string_demangle(typeid(T).name()) + 
                          " is not POD!");
      else
        ifs.read(reinterpret_cast<char*>(&outData), sizeof(outData));
    }



    /**
     * Simply reads the data in a dumb manner from the stream
     */
    template<typename T>
    void dumbRead(std::istream& ifs, T& outData) {
      if (undumpFunctions && undumpFunctions->count(typeid(T).hash_code()))
        outData = undump<T>(ifs);
      else if (!std::is_pod<T>::value) 
        throw IOException("The default implementation supports only POD types, "
                          "but " + string_demangle(typeid(T).name()) + 
                          " is not POD!");
      else
        ifs.read(reinterpret_cast<char*>(&outData), sizeof(outData));
    }
    
    template<typename T>
    void dumbRead(std::istream& ifs, std::list<T>& outData) {
      uint64_t s[2];
      dumbRead(ifs, s);
      outData = std::list<T>();
      for (uint64_t i = 0; i < s[1]; ++i)
        outData.push_back(undump<T>(ifs));
    }

    template<typename T>
    void dumbRead(std::istream& ifs, std::vector<T>& outData) {
      uint64_t s[2];
      dumbRead(ifs, s);
      // if T is not default_constructible, we must do this the slow way
      outData = std::vector<T>();
      for (uint64_t i = 0; i < s[1]; ++i) 
        outData.push_back(undump<T>(ifs));
    }
    
    template<typename T>
    void dumbRead(std::istream& ifs, std::set<T>& outSet) {
      uint64_t r = 0;
      dumbRead(ifs, r); // this value is ignored
      dumbRead(ifs, r);
      outSet = std::set<T>();
      for (uint64_t i = 0; i < r; ++i) 
        outSet.insert(boost::any_cast<T>(undump<T>(ifs)));
    }

    template<typename K, typename V>
    void dumbRead(std::istream& ifs, std::map<K,V>& outMap) {
      // std::maps are stored as follows:
      // [uint64_t][uint64_t]([key_i][value_i])*
      //     |         |        |
      //     |         |        +---Each (k,v) pair written after one another 
      //     |         +---Number of elements
      //     +---Size of data (not necessarily proportional to the number of
      //         elements)
      uint64_t nElems;
      dumbRead(ifs, nElems); // size information may be discarded
      dumbRead(ifs, nElems);

      outMap = std::map<K,V>();
      for (size_t i = 0; i < nElems; ++i) {
        K key = undump<K>(ifs);
        V value = undump<V>(ifs);
        outMap.insert(std::make_pair(key,value));
      }
    }

    inline bool hasNewSharedPtr(uintptr_t ptrKey) const {
      return oldToNewSharedPtrMap.count(ptrKey);
    }

    template<typename T>
    std::shared_ptr<T> getNewSharedPtr(uintptr_t ptrKey) {
      return boost::any_cast<std::shared_ptr<T> >(oldToNewSharedPtrMap[ptrKey]);
    }

    template<typename T>
    void undumpSharedPointer(std::istream& ifs, uintptr_t ptrKey) {
      uint64_t ptr;
      dumbRead(ifs, ptr);
      assert(ptr == ptrKey);
      boost::any d(undump<T>(ifs));
      std::shared_ptr<T> p(new T(boost::any_cast<T>(d)));
      oldToNewSharedPtrMap[ptrKey] = p;
    }
    
  private:
    /**
     * Simply reads the data in a dumb manner from the stream and returns it as an any object
     */
    template<typename T>
    static boost::any dumbReadAny(std::istream& ifs) {
      if (!std::is_pod<T>::value)
        throw IOException("Only POD types can be dumbly read! " + string_demangle(typeid(T).name()) +
                          " is not POD!");
      char outData[sizeof(T)];
      ifs.read(reinterpret_cast<char*>(outData), sizeof(outData));
      return boost::any(*reinterpret_cast<T*>(outData));
    }



    class UnDumperBase {
    public:
      virtual boost::any undumpAny(BlackBoardDumpReader* const w, std::istream& is) const = 0;
      virtual ~UnDumperBase() {}
    };
      
    template<typename T>
    class UnDumper : public UnDumperBase {
    public:
      typedef T(*undump_function_t)(BlackBoardDumpReader* const, std::istream&);
      explicit UnDumper(undump_function_t func) : func(func) {}

      virtual boost::any undumpAny(BlackBoardDumpReader* const w, std::istream& is) const {
        return boost::any(func(w,is));
      }

      T undump(BlackBoardDumpReader* const w, std::istream& is) const {
        return func(w, is);
      }

      virtual ~UnDumper() {}

    private:
      undump_function_t func;
    };

    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    // a writer struct for partial template specialisation
    // can only read POD
    template<typename T, bool IS_POD = std::is_pod<T>::value>
    struct DefaultReader {
      static boost::any read(std::istream& ifs);
    };


    // for partial specialisation
    template<typename T>
    class UndumpFunctor {
    public:
      static T undump(BlackBoardDumpReader * const w, std::istream& ifs) {
        if (undumpFunctions && undumpFunctions->count(typeid(T).hash_code()))
          return dynamic_cast<UnDumper<T>*>((*undumpFunctions)[typeid(T).hash_code()].get())->undump(w, ifs);
      
        if (!std::is_pod<T>::value) 
          throw IOException("No undump function was found for type " + 
                            string_demangle(typeid(T).name()) + " and the "
                            "default implementation only applicable to POD types");
        
        return boost::any_cast<T>(DefaultReader<T>::read(ifs));        
      }
    };



    template<typename T>
    T undump(std::istream& ifs) {
      return UndumpFunctor<T>::undump(this, ifs);
    }



    template<typename T>
    boost::any undumpAny(std::istream& is) const {
      return boost::any(undump<T>(is));
    }

    boost::any undump(std::istream& ifs, uint64_t dataType);


    /**
     * Dumps the content to a file
     */
    virtual bool processRangeImplementation(frame_number_t first, 
                                            frame_number_t last, 
                                            UiCallback* uiCallback);

    std::vector<DumpEntry> readHeaders(std::istream& ifs) const;



    static std::unordered_map<size_t, std::shared_ptr<UnDumperBase> >* undumpFunctions;
    static std::unordered_map<size_t, boost::any(*)(BlackBoardDumpReader* const, std::istream& )>* anyUndumpFunctions;

    // this is here to just free the variable above. do not use for anything else
    static struct Foo {
      ~Foo() {
        delete undumpFunctions;
        delete anyUndumpFunctions;
      }
    } undumpFunctionsDeleter;

    /**
     * The keys of this map are the original pointers stored in the dump 
     * file; the values are the corresponding new std::shared_ptrs
     */
    std::map<uintptr_t, boost::any> oldToNewSharedPtrMap;
    std::vector<std::string> filenames;
  };

  template<typename T>
  struct BlackBoardDumpReader::DefaultReader<T,true> {
    static boost::any read(std::istream& ifs) {
      T t;
      ifs.read(reinterpret_cast<char*>(&t), sizeof(t));
      return boost::any(t);
    }
  };

  template<typename T>
  struct BlackBoardDumpReader::DefaultReader<T,false> {
    static boost::any read(std::istream&) {
      throw IOException("Default reader can only read POD types and " + 
                        string_demangle(typeid(T).name()) + " is not POD");
      return boost::any();
    }
  };

  template<typename T>
  class BlackBoardDumpReader::UndumpFunctor<std::list<T> > {
  public:
    static std::list<T> undump(BlackBoardDumpReader * const w, std::istream& ifs) {
      std::list<T> l;
      w->dumbRead(ifs, l);
      return l;
    }
  };

  template<typename T>
  class BlackBoardDumpReader::UndumpFunctor<std::vector<T> > {
  public:
    static std::vector<T> undump(BlackBoardDumpReader * const w, std::istream& ifs) {
      std::vector<T> l;
      w->dumbRead(ifs, l);
      return l;
    }
  };

  template<typename T>
  class BlackBoardDumpReader::UndumpFunctor<std::set<T> > {
  public:
    static std::set<T> undump(BlackBoardDumpReader * const w, std::istream& ifs) {
      std::set<T> l;
      w->dumbRead(ifs, l);
      return l;
    }
  };

  template<typename K, typename V>
  class BlackBoardDumpReader::UndumpFunctor<std::map<K, V> > {
  public:
    static std::map<K,V> undump(BlackBoardDumpReader * const w, std::istream& ifs) {
      std::map<K,V> m;
      w->dumbRead(ifs, m);
      return m;
    }
  };
}

#endif

