#include "BlackBoardDumpWriter.hpp"
#include "BlackBoardDumpReader.hpp"
#include <fstream>
#include "Blob.hpp"
#include "BodyPart.hpp"
#include "TrackedPoint.hpp"
#include "gzstream.hpp"

namespace slmotion {
  extern int debug;

  static BlackBoardDumpWriter DUMMY(true);

  BlackBoardDumpWriter::BlackBoardDumpWriter(bool) : Component(true) { 
    registerAnyWriter<double>();
    registerAnyWriter<int>();
    registerAnyWriter<long>();
    registerAnyWriter<cv::Rect>();
    registerAnyWriter<cv::Mat>();
    registerAnyWriter<cv::Point>();
    registerDumbWriter<cv::Rect>([](std::ostream& ofs, const cv::Rect& data) {
        int32_t r[4] = { data.x, data.y, data.width, data.height };
        ofs.write(reinterpret_cast<const char*>(r), sizeof(r));
      });
    registerSizeComputer<cv::Rect>([](const cv::Rect&) { 
        return 4*sizeof(int32_t); 
      });
    registerSizeComputer<cv::Point>([](const cv::Point&) {
        return 2*sizeof(int32_t);
      });
    registerSizeComputer<cv::Point2d>([](const cv::Point2d&) {
        return 2*sizeof(double);
      });
    registerSizeComputer<cv::Point2f>([](const cv::Point2f&) {
        return 2*sizeof(float);
      });
    registerSizeComputer<std::string>([](const std::string& s) {
        // format:
        // [uint64_t][char-array]
        //     |          |
        //     |          +--- content (NO terminating zero)
        //     +--- number of chars
        return s.length() + sizeof(uint64_t);
      });  
    registerDumbWriter<std::string>([](std::ostream& ofs, const std::string& data) {
        // format:
        // [uint64_t][char-array]
        //     |          |
        //     |          +--- content (NO terminating zero)
        //     +--- number of chars
        uint64_t len = data.length();
        ofs.write(reinterpret_cast<const char*>(&len), sizeof(len));
        ofs.write(data.c_str(), len);
      });

    registerSizeComputer<cv::Mat>([](const cv::Mat& m) { 
        return m.total()*m.elemSize() + 3*sizeof(int32_t);
      });
    registerDumbWriter<cv::Point>([](std::ostream& ofs, const cv::Point& data) {
        int32_t x[2] = { data.x, data.y };
        ofs.write(reinterpret_cast<const char*>(x), sizeof(x));
      });
    registerDumbWriter<cv::Point2d>([](std::ostream& ofs, const cv::Point2d& data) {
        double x[2] = { data.x, data.y };
        ofs.write(reinterpret_cast<const char*>(x), sizeof(x));
      });
    registerDumbWriter<cv::Point2f>([](std::ostream& ofs, const cv::Point2f& data) {
        float x[2] = { data.x, data.y };
        ofs.write(reinterpret_cast<const char*>(x), sizeof(x));
      });
    // Matrices are stored as follows:
    // [int32_t][int32_t][int32_t][data]
    //     |        |        |
    //     |        |        +---Rows
    //     |        +---Cols
    //     +---Type
    registerDumbWriter<cv::Mat>([](std::ostream& ofs, const cv::Mat& m) {
        assert(m.isContinuous());
        int32_t t = m.type();
        ofs.write(reinterpret_cast<const char*>(&t), sizeof(t));
        t = m.cols;
        ofs.write(reinterpret_cast<const char*>(&t), sizeof(t));
        t = m.rows;
        ofs.write(reinterpret_cast<const char*>(&t), sizeof(t));
        ofs.write(reinterpret_cast<const char*>(m.data), m.total()*m.elemSize());
      });
    
    registerSizeComputer<boost::any>([](const BlackBoardDumpWriter* const w, const boost::any& data)->size_t {
        uint64_t dataType = data.type().hash_code();
        if (sizeComputers->count(dataType))
          return (*sizeComputers)[dataType]->getSize(w, data);
        else if (anySizeComputers->count(dataType))
          return (*anySizeComputers)[dataType](w, data);

        throw IOException("Cannot handle data of this type: " +
                          string_demangle(data.type().name()));
        assert(false);
        return 0;
      });
  }

  std::unordered_map<size_t, std::shared_ptr<BlackBoardDumpWriter::AnyDumbWriterBase> >* BlackBoardDumpWriter::anyWriterFunctions = nullptr;

  std::unordered_map<size_t, std::shared_ptr<BlackBoardDumpWriter::DumbWriterBase> >* BlackBoardDumpWriter::dumbWriterFunctions = nullptr;

  std::unordered_map<size_t, std::shared_ptr<BlackBoardDumpWriter::SizeComputerBase> >* BlackBoardDumpWriter::sizeComputers = nullptr;

  std::unordered_map<size_t, size_t(*)(const BlackBoardDumpWriter* const, const boost::any&) >* BlackBoardDumpWriter::anySizeComputers = nullptr;

  std::unordered_map<size_t, std::shared_ptr<BlackBoardDumpWriter::AnyPointerSetProviderBase> >* BlackBoardDumpWriter::pointerStoreFunctions = nullptr;

  void BlackBoardDumpWriter::process(frame_number_t) {
    assert(false && "NOT SUPPORTED");
  }

  Component* BlackBoardDumpWriter::createComponentImpl(const boost::program_options::variables_map& opts, BlackBoard* blackBoard, FrameSource* frameSource) const {
    BlackBoardDumpWriter b(blackBoard, frameSource);
    if (opts.count("out-dump-file"))
      b.filename = opts["out-dump-file"].as<std::string>();
    return new BlackBoardDumpWriter(b);
  }

  boost::program_options::options_description BlackBoardDumpWriter::getCommandLineOptionsDescription() const {
    boost::program_options::options_description opts("BlackBoard Dump Writer options");
    opts.add_options()("out-dump-file", 
                       boost::program_options::value<std::string>(),
                       "file to dump black board content to, file name ending in .gz will be written using gzip compression");
    return opts;
  }



  void BlackBoardDumpWriter::dump(std::ostream& ofs, const boost::any& data) {
    size_t dataType = data.type().hash_code();
    if (anyWriterFunctions && anyWriterFunctions->count(dataType))
      (*anyWriterFunctions)[dataType]->write(this, ofs, data);
    else
      throw IOException("Cannot handle data of this type: " +
                        string_demangle(data.type().name()));
  }


      
  DumpEntry BlackBoardDumpWriter::constructDumpEntry(uint64_t frameNumber,
                                                     const std::string& key,
                                                     uint64_t& dataPointer,
                                                     const boost::any& data) {
    uint64_t dataType = data.type().hash_code();
    assert(key.length() < 33);
    DumpEntry d { frameNumber, dataType, "", dataPointer };
    strcpy(d.key, key.c_str());
    dataPointer += getSize(data);
    if (pointerStoreFunctions->count(data.type().hash_code())) {
      auto newPointers = (*(*pointerStoreFunctions)[data.type().hash_code()])(this, data);
      storedPointers.insert(newPointers.cbegin(), newPointers.cend());
    }
    return d;
  }



  bool BlackBoardDumpWriter::processRangeImplementation(frame_number_t,
                                                        frame_number_t,
                                                        UiCallback* callback) {
    assert(storedPointers.size() == 0);
    unsigned int i = 1;
    uint64_t totalSize = getBlackBoard().getFrameBoard().size() +
      getBlackBoard().getGlobalBoard().size();
    if (debug > 1) 
      std::cerr << "BlackBoard Dump Writer:" << std::endl
                << "Opening output file: \"" << filename << "\"...";

    std::ofstream *ofs=NULL;
    ogzstream *ogzs=NULL;

    std::ostream *osptr;
    
    if(filename.substr(filename.size()-3)==".gz"){
      ogzs=new ogzstream(filename.c_str());
      osptr=ogzs;
    } else{
      ofs= new std::ofstream(filename);
      osptr=ofs;
    }



    if (!osptr->good())
      throw IOException("Could not open \"" + filename + "\" for writing!");
    if (debug > 1)
      std::cerr << "OK!" << std::endl
                << "Writing header..." << std::endl;

    dumbWrite(*osptr, totalSize);
    if (debug > 1)
      std::cerr << "Total number of entries: " << totalSize << std::endl;

    uint64_t dataPointer = sizeof(uint64_t); // move by the number of 
                                             // entries
    // then adjust by considering the headers
    dataPointer += totalSize*sizeof(DumpEntry);

    std::vector<DumpEntry> entryHeaders;

    // construct headers

    for (auto it = getBlackBoard().getFrameBoard().cbegin(); 
         it != getBlackBoard().getFrameBoard().cend();
         ++it) {
      BlackBoardPointer<boost::any> bbp = it->second->referenceAny(it->second);
      entryHeaders.push_back(constructDumpEntry(it->first.first,
                                                it->first.second,
                                                dataPointer,
                                                *bbp));
      if (callback != nullptr)
        (*callback)(i*(100./3.)/totalSize);
      ++i;
    }

    for (auto it = getBlackBoard().getGlobalBoard().cbegin(); 
         it != getBlackBoard().getGlobalBoard().cend();
         ++it) {
      BlackBoardPointer<boost::any> bbp = it->second->referenceAny(it->second);
      entryHeaders.push_back(constructDumpEntry(-1,
                                                it->first,
                                                dataPointer,
                                                *bbp));
      if (callback != nullptr)
        (*callback)(i*(100./3.)/totalSize);
      ++i;
    }

    i = 1;
    for (auto it = entryHeaders.cbegin(); it != entryHeaders.cend(); ++it) {
      if (debug > 1) {
        std::cerr << "Dumping header " << it->key;
        if (it->frameNumber < uint64_t(-1))
          std::cerr << '[' << it->frameNumber << ']';
        std::cerr << '(' << i << "/" << totalSize  << ')' << std::endl;
      }
      // else if (debug == 1)
      //   std::cerr << "\rDumping header " << i++ << "/" << totalSize;
      dumbWrite(*osptr, *it);
      if (callback != nullptr)
        (*callback)(100./3. + i*(100./3.)/totalSize);
      ++i;
    }
    // if (debug == 1)
    //   std::cerr << std::endl;

    // the getSize function stores some pointers in the set to avoid 
    // redundant copies of pointer-members.
    storedPointers.clear();

    i = 1;
    if (debug > 1)
      std::cerr << "Beginning data dump..." << std::endl;
    for (auto it = entryHeaders.cbegin(); it != entryHeaders.cend(); ++it) {
      if (debug > 1) {
        std::cerr << "Dumping " << it->key;
        if (it->frameNumber < uint64_t(-1))
          std::cerr << '[' << it->frameNumber << ']';
        std::cerr << '(' << i << "/" << totalSize  << ')' << std::endl;
      }
      if(ofs!=NULL)
	if (ofs->tellp() != static_cast<int64_t>(it->dataPointer))
	  throw IOException("Fatal error: data pointer mismatch. Tellp: " + 
                            boost::lexical_cast<std::string>(ofs->tellp()) + 
                            "; expected: " + 
                            boost::lexical_cast<std::string>(it->dataPointer));

      if (it->frameNumber < uint64_t(-1))
        dump(*osptr, *getBlackBoard().get(it->frameNumber, it->key));
      else
        dump(*osptr, *getBlackBoard().get(it->key));

      if (callback != nullptr)
        (*callback)(200./3. + i*(100./3.)/totalSize);

      ++i;
    }

    // if (debug == 1)
    //   std::cerr << std::endl;

    storedPointers.clear();

    if(ofs)
      delete ofs;

    if(ogzs)
      delete ogzs;

    return true;
  }
}
