#ifndef SLMOTION_BLACKBOARD_ENTRY
#define SLMOTION_BLACKBOARD_ENTRY

#include "BlackBoardException.hpp"
#include <memory>
#include <boost/any.hpp>
#include <iostream>
#include <vector>

// this define is here temporarily; in the future, implicit
// conversions will be disabled as they are unsafe
#define BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS false

namespace slmotion {
  // forward declaration
  template <typename T>
  class BlackBoardPointer;

  // forward declaration (util.cpp depends on this file, and including
  // util.hpp here would cause a mess)
  std::string string_demangle(const char* name);

  /**
   * This class represents a single entry on the black
   * board. Basically, it implements reference-counting and knows how
   * to compress matrices when they are not being referred to (unless
   * compression is forbidden).
   */
  class BlackBoardEntry {
  public:
    // only the pointer class should be able to decrease the reference
    // count; hence friendship
    template <typename T>
    friend class BlackBoardPointer;

    /**
     * These are attributes that affect the behaviour of the entry;
     * they should be rather self-explanatory
     */
    enum BlackBoardEntryAttribute {
      NONE = 0, // also: never compress
      COMPRESS_WITHOUT_REFERENCES
    };

    /**
     * Create an arbitrary entry
     */
    template <typename T>
    BlackBoardEntry(const T& value, BlackBoardEntryAttribute attribs) :
      value(new boost::any(value)), attribs(attribs), refCount(0), status(UNCOMPRESSED)
    {
      compressIfPossible();
    }

    /**
     * Create an arbitrary entry
     */
    template <typename T>
    BlackBoardEntry(T&& value, BlackBoardEntryAttribute attribs) :
      value(new boost::any(value)), attribs(attribs), refCount(0), status(UNCOMPRESSED)
    {
      compressIfPossible();
    }

#ifdef BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS
#if BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS == true
#warning Implicit conversions are unsafe! If you really need them, please address your code.
    template<typename T>
    operator T&() {
      return boost::any_cast<T&>(*value);
    }

    inline const boost::any& getAnyReference() const {
      return *value;
    }
#endif // BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS == true
#endif // BLACKBOARD_ENTRY_ALLOW_IMPLICIT_CONVERSIONS



    /**
     * Returns the type of the value
     */
    inline const std::type_info& type() const {
      return value->type();
    }

    /**
     * Increases the reference count by one and returns a pointer
     */
    template <typename T>
    BlackBoardPointer<T> reference(std::shared_ptr<BlackBoardEntry> thisPointer) {
      if (status == COMPRESSED && !decompress())
        throw BlackBoardException("Decompression failed!");
      

      T* v = boost::any_cast<T>(value.get());
      if (value.get() != nullptr &&
          v == nullptr)
        throw BlackBoardException("Conversion from " + 
                                  string_demangle(value->type().name()) +
                                  " to " +
                                  string_demangle(typeid(T).name()) +
                                  " failed.");
      ++refCount;
      return BlackBoardPointer<T>(v, thisPointer);
    }

    /**
     * Increases the reference count by one and returns a pointer to
     * the any object
     */
    BlackBoardPointer<boost::any> referenceAny(std::shared_ptr<BlackBoardEntry> thisPointer); 



    /**
     * Checks if the entry points to the same value. This can be used
     * to prevent self-assignment.
     */
    inline bool pointsTo(const void* ptr) const {
      return value.get() == ptr;
    }

  private:
    // only the pointer class should be able to decrease the reference
    // count; hence friendship
    void unReference();

    /**
     * Attempts to compress the value if the attributes and the
     * reference count permit, and the value is of such type that it
     * can be compressed.
     *
     * @return True if the value was compressed
     */
    bool compressIfPossible();

    /**
     * Attempts to decompress the compressed data.
     *
     * @return True if the data was decompressed. False if
     * decompression failed, or the data was not compressed to begin
     * with.
     */
    bool decompress();

    enum BlackBoardEntryStatus {
      UNCOMPRESSED = 0,
      COMPRESSED
    };

    std::unique_ptr<boost::any> value;
    BlackBoardEntryAttribute attribs;
    int refCount;
    BlackBoardEntryStatus status;
    std::vector<unsigned char> compressedData; ///< holds the compressed data
  };
}

#endif
