#include "BlackBoardExplorer.hpp"
#include "util.hpp"



#ifdef SLMOTION_BOOST_REGEX
#include <boost/regex.hpp>
namespace std {
  using boost::regex;
  namespace regex_constants = boost::regex_constants;
  using boost::smatch;
  using boost::regex_match;
}
#else
#include <regex> // Not properly implemented as of GCC 4.5.1
#endif

using std::string;
using boost::lexical_cast;
using std::regex;
using std::smatch;

namespace slmotion {
  static BlackBoardExplorer DUMMY(true);

  class Printer {
  public:
    void createLog(const std::string& filename) {
      logstream.open(filename);
    }

    /**
     * Prints a message to any open streams
     */
    void operator()(const std::string& message) {
      if (logstream.is_open())
        logstream << message;
      std::cout << message;
    }

    void operator()(const cv::Rect& r) {
      (*this)("(" + lexical_cast<std::string>(r.x) + "," +
              lexical_cast<std::string>(r.y) + ")->(" +
              lexical_cast<std::string>(r.x+r.width-1) + "," +
              lexical_cast<std::string>(r.y+r.height-1) + ")");
    }    

    /**
     * Prints the message to a log if the log stream is open
     */
    void log(const std::string& message) {
      if (logstream.is_open())
        logstream << message;
    }

  private:
    std::ofstream logstream; ///< possible log file
  };


  static void printHelp(Printer& print) {
    print("Known commands:\n"
          "help                 shows this screen\n"
          "open_log <filename>  logs everything to the file\n"
          "list                 lists everything on the blackboard\n"
          "print <int> <str>    prints the given property\n"
          "imshow <int> <str>   shows the given image property\n"
          "erase <int> <str>    erases the given property from the blackboard\n"
          "erase <str>          erases the given global property from the\n"
          "                     blackboard\n"
          "eraseall <str>       erases all instances of the given property\n"
          "quit                 terminates the interactive session\n");
  }

  

  static void printAll(Printer& print, const BlackBoard& blackboard) {
    auto keys = blackboard.getAllFrameboardKeys();
    for (auto it = keys.cbegin(); it != keys.cend(); ++it) {
      print(lexical_cast<string>(it->first) + (" " + it->second + " "));
      const std::type_info& type = blackboard.typeOf(it->first, it->second);
      if (type == typeid(cv::Rect))
        print("Rectangle");
      else if (type == typeid(cv::Mat)) {
        print("Matrix (");
        BlackBoardPointer<cv::Mat> m = blackboard.get<cv::Mat>(it->first, it->second);
        print(boost::lexical_cast<std::string>(m->cols));
        print("x");
        print(boost::lexical_cast<std::string>(m->rows));
        print(", ");
        switch(m->type()) {
        case CV_8UC1:
          print("CV_8UC1");
          break;
        case CV_8UC3:
          print("CV_8UC3");
          break;
        case CV_32FC1:
          print("CV_32FC1");
          break;
        case CV_64FC1:
          print("CV_64FC1");
          break;
        default:
          print("UNKNOWN");
        }
        print(")");
      }
      else
        print("Unknown type (" + string_demangle(type.name()) + ")");
      print("\n");
    }
  }



  static bool getUserInput(Printer& print, std::istream& inStream, std::string& outInput,
                           const std::string& prefix = "") {
    print(prefix);
    return !getline(inStream, outInput).fail();
  }



  static void printProperty(Printer& print, const BlackBoard& bb,
                            BlackBoard::frame_number_t frameNumber,
                            const std::string& property) {
    if (!bb.has(frameNumber, property)) {
      print("No such element on the board!\n");
      return;
    }
    
    if (bb.typeOf(frameNumber, property) == typeid(cv::Rect))
      print(*bb.get<cv::Rect>(frameNumber, property));
    else
      print("Unrecognised object type.");

    print("\n");
  }



  void BlackBoardExplorer::goInteractive() {
    Printer print;

    print("Interactive BlackBoard explorer. Type 'help' for help or 'quit' to quit.\n");

    std::string userInput;
    smatch m;
    while (getUserInput(print, std::cin, userInput, ">")) {
      print.log(userInput + "\n");

      try {
        if (userInput == "quit")
          break;
        else if (userInput == "help")
          printHelp(print);
        else if (regex_match(userInput, m, regex("open_log (.*)")))
          print.createLog(m[1]);
        else if (userInput == "list")
          printAll(print, getBlackBoard());
        else if (regex_match(userInput, m, regex("print ([0-9]+) (.*)")))
          printProperty(print, getBlackBoard(), 
                        lexical_cast<BlackBoard::frame_number_t>(m[1]), m[2]);
        else if (regex_match(userInput, m, regex("imshow ([0-9]+) (.*)"))) {
          BlackBoardPointer<cv::Mat> mtx = getBlackBoard().get<cv::Mat>(lexical_cast<BlackBoard::frame_number_t>(m[1]), m[2]);
          cv::imshow("", *mtx);
          cv::waitKey(0);
        }
        else if (regex_match(userInput, m, regex("erase ([0-9]+) (.*)")))
          getBlackBoard().erase(lexical_cast<BlackBoard::frame_number_t>(m[1]), 
                                m[2]);
        else if (regex_match(userInput, m, regex("erase (.*)")))
          getBlackBoard().erase(m[1]);
        else if (regex_match(userInput, m, regex("eraseall (.*)")))
          getBlackBoard().eraseAll(m[1]);
        else 
          print("Unrecognised command\n");
      }
      catch (BlackBoardException& e) {
        print("ERROR: " + std::string(e.what()) + "\n");
      }
    }
  }
}
