#include "Blob.hpp"
#include "util.hpp"

using cv::Point;
using cv::Mat;
using cv::Scalar;
using std::cerr;
using std::endl;
using std::vector;
using cv::Size;



namespace {
  // This is a simple functor that is used to remove blobs whose size is 
  // below the given threshold
  struct IsSizeLessThan {
    IsSizeLessThan(size_t minSize) :
      minSize(minSize) { }

    bool operator()(const slmotion::Blob& b) {
      return b.size() < minSize;
    }

    size_t minSize;
  };
}


namespace slmotion {
  extern int debug;



  const cv::Point Blob::INVALID_POINT(INT_MIN,INT_MIN);



  vector<Blob> Blob::getContourBlobs(const cv::Mat& inMask) {
    Mat temp = inMask.clone();

    vector<vector<Point>> contours;
    findContours(temp, contours, CV_RETR_EXTERNAL, CV_CHAIN_APPROX_SIMPLE);

    vector<Blob> blobs(contours.size());
    for (size_t i = 0; i < blobs.size(); ++i)
      blobs[i] = Blob(contours[i]);
    
    return blobs;
  }



  void Blob::computeCachedValues() {
    // initialise extreme points to bogus values so that they inevitably get updated
    // (except e.g. in the case where the blob is utterly located on the top pixel
    // row, or the extreme left pixel column. This case isn't very relevant,
    // though.
    topMost = Point(0, INT_MAX);
    bottomMost = Point(0, 0);
    leftMost = Point(INT_MAX, 0);
    rightMost = Point(0, 0);

    for (auto i = points.cbegin(); i != points.cend(); i++) {
      if (i->x < leftMost.x)
        leftMost = *i;
      if (i->x > rightMost.x)
        rightMost = *i;
      if (i->y < topMost.y)
        topMost = *i;
      if (i->y > bottomMost.y)
        bottomMost = *i;
    }

    int x = 0, y = 0;
    blobSize = 0;
    Mat temp = toMatrix(Size(rightMost.x+1,bottomMost.y+1));
    for (int i = 0; i < temp.rows; ++i) {
      for (int j = 0; j < temp.cols; ++j) {
        if (temp.at<uchar>(i,j)) {
          x += j;
          y += i;
          ++blobSize;
        }
      }
    }

    centroid = points.size() > 0 ?
      Point(x / blobSize, y / blobSize) :
      INVALID_POINT;
  }



  void slmotion::Blob::draw(Mat& img, const Scalar& colour) const {
    drawPolygon(img, points, colour);
  }

#if (__GNUC__ >= 4 && __GNUC_MINOR__ < 5)
  namespace {
    // functor for replacing the lambda function (for GCC < 4.5)
    struct blobSmallerThan {
      int minSize;
      blobSmallerThan(int minSize) :
        minSize(minSize) {}
      bool operator()(const Blob& b) { 
        return b.size() < minSize; 
      }
    };
  }
#endif // (__GNUC__ >= 4 && __GNUC_MINOR__ < 5)

  // Extracts a vector of blobs whose size exceeds the minimum size from the
  // internal binary mask temporary
  vector<slmotion::Blob> slmotion::Blob::extractBlobsFromMask(const Mat& mask, size_t minSize, int maxBlobs) {

    vector<Blob> blobs = getContourBlobs(mask); //getFourConnectedBlobs(mask);

    // Finally, make sure that the number of blobs does not exceed the maximum
    // number
    if (maxBlobs > 0 && static_cast<int>(blobs.size()) > maxBlobs) {
      std::sort(blobs.begin(), blobs.end(), slmotion::Blob::compare);
      blobs.resize(maxBlobs);
    }

    // go through blobs and only return those blobs whose size exceeds the minimum
#if (__GNUC__ >= 4 && __GNUC_MINOR__ >= 5)
    blobs.erase(std::remove_if(blobs.begin(), blobs.end(), 
                               [&minSize](const Blob& b) { 
                                 return b.size() < minSize; 
                               }), blobs.end());
#else // (__GNUC__ >= 4 && __GNUC_MINOR__ >= 5)
    blobs.erase(std::remove_if(blobs.begin(), blobs.end(), 
                               blobSmallerThan(minSize)), blobs.end());
#endif // (__GNUC__ >= 4 && __GNUC_MINOR__ >= 5)

    if (slmotion::debug > 1)
      cerr << "Found " << blobs.size() << " blobs\n";

    return blobs;
  }



  void slmotion::Blob::cutPointsOutside(const cv::Rect& box) {
    const Point& br = box.br();
    Mat temp2 = toMatrix(Size(std::max(rightMost.x,br.x)+2, 
                              std::max(bottomMost.y,br.y)+2));
    Mat temp(temp2.size(), CV_8UC1, Scalar::all(0));
    Mat mask(temp2.size(), CV_8UC1, Scalar::all(0));
    rectangle(mask, box.tl(), box.br(), Scalar::all(255), CV_FILLED);
    temp2.copyTo(temp, mask);

    vector<Blob> blobs = getContourBlobs(temp);
    this->points = std::move(std::max_element(blobs.begin(), blobs.end(), 
                                              &Blob::compare)->points);
      
    computeCachedValues();
  }



  Mat slmotion::Blob::toMatrix(const cv::Size& size,
                               bool invert,
                               const Point& tr) const {
    // make sure the matrix will be large enough
    assert((leftMost+tr).x >= 0);
    assert((topMost+tr).y >= 0);
    assert(size.width > (rightMost+tr).x);
    assert(size.height > (bottomMost+tr).y);

    Mat newMat(size, CV_8UC1, invert ? Scalar::all(255) : Scalar::all(0));
    drawPolygon(newMat, points, invert ? Scalar::all(0) : Scalar::all(255),
                tr);

    return newMat;
  }



  Blob& Blob::operator=(const Blob& another) {
    if (this != &another) {
      vector<Point> tempPoints = another.points;
      Point tempCentroid = another.centroid;
      Point tempLeftMost = another.leftMost;
      Point tempRightMost = another.rightMost;
      Point tempTopMost = another.topMost;
      Point tempBottomMost = another.bottomMost;
      std::swap(points, tempPoints);
      std::swap(leftMost, tempLeftMost);
      std::swap(rightMost, tempRightMost);
      std::swap(topMost, tempTopMost);
      std::swap(bottomMost, tempBottomMost);
      std::swap(centroid, tempCentroid);
      blobSize = another.blobSize;
    }
    return *this;
  }



  Blob& Blob::operator=(Blob&& another) {
    if (this != &another) {
      points = std::move(another.points);
      centroid = std::move(another.centroid);
      leftMost = std::move(another.leftMost);
      rightMost = std::move(another.rightMost);
      topMost = std::move(another.topMost);
      bottomMost = std::move(another.bottomMost);
      blobSize = std::move(another.blobSize);
      another.blobSize = 0;
      another.points = vector<Point>();
      another.centroid = another.leftMost = another.rightMost = 
        another.topMost = another.bottomMost = INVALID_POINT;
    }
    return *this;
  }



  bool Blob::operator==(const Blob& b) const {
    return centroid == b.centroid && points == b.points &&
      leftMost == b.leftMost && rightMost == b.rightMost &&
      topMost == b.topMost && bottomMost == b.bottomMost &&
      blobSize == b.blobSize;
  }



  std::vector<cv::Point> Blob::getContour() const {
    return chainApproxSimpleToChainApproxNone(points);
  }



  cv::Rect Blob::getBoundingBox() const {
    auto lp = getLeftMost();
    auto rp = getRightMost();
    auto bp = getBottomMost();
    auto tp = getTopMost();
    return cv::Rect(lp.x, tp.y, rp.x-lp.x, bp.y-tp.y);  
  }
}

