#include "BlobExtractor.hpp"
#include "Blob.hpp"
#include "configuration.hpp"
#include "BlackBoardDumpWriter.hpp"

namespace slmotion {
  static BlobExtractor DUMMY(true);

  BlobExtractor::BlobExtractor(bool) :
      Component(true) {
    BlackBoardDumpWriter::registerAnyWriter<std::vector<Blob> >();

    BlackBoardDumpWriter::registerSizeComputer<Blob>([](const BlackBoardDumpWriter * const w, const Blob& b) {
        return w->getSize(b.getChainCodedContour()) + sizeof(uint64_t) +
          5*w->getSize(b.getCentroid());
      });

    BlackBoardDumpWriter::registerDumbWriter<Blob>([](BlackBoardDumpWriter* const w, std::ostream& ofs, const Blob& data) {
        // Stored as follows:
        // [point][vectorpoint][4*point][uint64_t]
        //    |       |           |         |
        //    |       |           |         +---blobSize
        //    |       |           +---leftMost,RightMost,
        //    |       |               topMost, bottomMost
        //    |       +---points in order 
        //    +---centroid                            
        w->dumbWrite(ofs, data.getCentroid());
        w->dumbWrite(ofs, data.getChainCodedContour());
        w->dumbWrite(ofs, data.getLeftMost());
        w->dumbWrite(ofs, data.getRightMost());
        w->dumbWrite(ofs, data.getTopMost());
        w->dumbWrite(ofs, data.getBottomMost());
        uint64_t r = data.size();
        w->dumbWrite(ofs, r);
      });

    BlackBoardDumpReader::registerUnDumper<Blob>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        cv::Point centroid, leftMost, rightMost, topMost, bottomMost;
        std::vector<cv::Point> v;
        uint64_t s;
    
        w->dumbRead(ifs, centroid);
        w->dumbRead(ifs, v);
        w->dumbRead(ifs, leftMost);
        w->dumbRead(ifs, rightMost);
        w->dumbRead(ifs, topMost);
        w->dumbRead(ifs, bottomMost);
        w->dumbRead(ifs, s);
        return Blob(centroid, v, leftMost, rightMost, topMost, bottomMost, s);
      });
  }


  boost::program_options::options_description BlobExtractor::getConfigurationFileOptionsDescription() const
  {
    boost::program_options::options_description opts;
    opts.add_options()
      ("BlobExtractor.blobremoval", boost::program_options::value<bool>()->default_value(true), "Whether to remove small blobs or not.")
      ("BlobExtractor.blobs", boost::program_options::value<int>()->default_value(3), "Only leave the <int> largest blobs, and remove others.")
      ("BlobExtractor.minblobsize", boost::program_options::value<int>()->default_value(3300), "Sets the minimum number of pixels that a blob must cover. Any smaller blobs will be removed.");
    return opts; //, "This section controls the blob extractor component"
  }



  void BlobExtractor::process(size_t framenr) {
    BlackBoardPointer<cv::Mat> skinMask = getBlackBoard().get<cv::Mat>(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);

    // extract blobs
    std::vector<Blob> outBlobs = Blob::extractBlobsFromMask(*skinMask, 
                                                            minBlobSize, 
                                                            nBlobs);
            
    if (removeSmallBlobs) {
      *skinMask = cv::Scalar::all(0);
      for (std::vector<Blob>::const_iterator it = outBlobs.begin(); 
           it != outBlobs.end(); ++it) 
        it->draw(*skinMask, cv::Scalar::all(255));

      getBlackBoard().set(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY,
                          *skinMask);
    }
    getBlackBoard().set(framenr, BLACKBOARD_BLOBS_ENTRY, outBlobs);
  }



  Component* BlobExtractor::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    return new BlobExtractor(blackBoard, frameSource, 
                             configuration.count("BlobExtractor.blobremoval") ?
                             configuration["BlobExtractor.blobremoval"].as<bool>() : true, 
                             configuration.count("BlobExtractor.blobs") ?
                             std::max(configuration["BlobExtractor.blobs"].as<int>(),0) : 3, 
                             configuration.count("BlobExtractor.minblobsize") ? 
                             std::max(configuration["BlobExtractor.minblobsize"].as<int>(),0) : 3300);
  }
}
