#ifndef SLMOTION_BLOB_EXTRACTOR
#define SLMOTION_BLOB_EXTRACTOR

#include <string>
#include "Component.hpp"
#include "SkinDetector.hpp"

namespace slmotion {
  // type: std::vector<Blob>
  const std::string BLACKBOARD_BLOBS_ENTRY = "skinblobs";



  /**
   * This component extracts blobs from frames and, optionally, also updates
   * the skin mask in such a way that areas representing blobs matching 
   * certain criteria, such as a minimum size, are removed.
   */ 
  class BlobExtractor : public Component {
  public:
    virtual std::string getShortDescription() const { 
      return "Extracts contiguous 4-connected blobs from binary images";
    };



    virtual std::string getLongDescription() const { 
      return "The Blob Extractor Extracts contiguous 4-connected blobs from binary images using a basic sequential algorithm."; 
    };



    /**
     * Returns the component name
     */
    inline std::string getComponentName() {
      return "Blob extractor";
    }



    /**
     * Processes a frame. This may also include pre or postprocessing
     * stages.
     *
     * @param inFrame Input frame
     * @param outMask Output binary mask
     * @param outBlobs Optional output parameter for blobs
     */
    void process(size_t framenr);



    /**
     * A simple constructor
     *
     * @param blackBoard The Black Board
     * @param frameSource The associated frame source (unimportant)
     * @param removeSmallBlobs If true, small blobs are removed from the 
     * mask (see the next parameters for criteria)
     * @param nBlobs Maximum number of blobs to return. Only the greatest 
     * blobs are returned.
     * @param minBlobSize Minimum blob size. Any blobs with fewer pixels 
     * than specified here shall be lost.
     */
    explicit BlobExtractor(BlackBoard* blackBoard, 
                           FrameSource* frameSource,
                           bool removeSmallBlobs = true,
                           size_t nBlobs = 3, size_t minBlobSize = 3300) :
      Component(blackBoard, frameSource), 
      removeSmallBlobs(removeSmallBlobs), nBlobs(nBlobs), 
      minBlobSize(minBlobSize) {
    }



    // dummy registration constructor
    BlobExtractor(bool);



    // virtual configuration::ConfigurationFileOptionsDescription getConfigurationFileOptionsDescription() const;
    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;



    inline virtual property_set_t getProvided() const {
      return property_set_t { BLACKBOARD_BLOBS_ENTRY };
    }



    inline virtual property_set_t getRequirements() const {
      return property_set_t { SKINDETECTOR_BLACKBOARD_MASK_ENTRY };
    }



    virtual void reset() {}
    virtual std::string getComponentName() const {
      return "Blob Extractor";
    }
    virtual std::string getShortName() const {
      return "BlobExtractor";
    }

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    bool removeSmallBlobs; ///< Whether small blobs should be removed
    size_t nBlobs; ///< Number of blobs to extract
    size_t minBlobSize; ///< Minimum size for a blob
  };
}

#endif
