#include "BodyPart.hpp"
#include "exceptions.hpp"

using std::cerr;
using std::endl;
using std::vector;
using cv::Point;



namespace slmotion {
  extern int debug;



  void BodyPart::createDistanceMatrix(const cv::Size& s) const {
    if (blob.size() == 0)
      throw SLMotionException("Cannot compute distance to a blob with no points!");
    cv::distanceTransform(blob.toMatrix(s, true), distanceMatrix,
                          CV_DIST_L2, 3);
  }



  double BodyPart::distanceTo(const Point& p) const {
    LOCK_THIS_MUTEX_WHEN_THREADING();

    if (p.x < 0 || p.y < 0)
      return INFINITY;

    if (p.x >= distanceMatrix.cols || p.y >= distanceMatrix.rows)
      createDistanceMatrix(cv::Size(std::max(p.x+1,blob.getRightMost().x+1),
                                    std::max(p.y+1,blob.getBottomMost().y+1)));
    return distanceMatrix.at<float>(p);
  }



  /**
   * Default copy constructor
   */
  BodyPart::BodyPart(const BodyPart& other) {
    LOCK_THIS_AND_OTHER(*this, other);
    this->identity = other.identity;
    this->blob = other.blob;
    this->ambiguous = other.ambiguous;
    this->distanceMatrix = other.distanceMatrix.clone();
  }



  /**
   * Default copy assignment
   */
  BodyPart& BodyPart::operator=(const BodyPart& other) {
    if (&other != this) {
      LOCK_THIS_MUTEX_WHEN_THREADING();
      BodyPart temp { other };
      std::swap(this->identity, temp.identity);
      std::swap(this->blob, temp.blob);
      std::swap(this->ambiguous, temp.ambiguous);
      std::swap(this->distanceMatrix, temp.distanceMatrix);
    }
    return *this;
  }



  /**
   * Default move constructor
   */
  BodyPart::BodyPart(BodyPart&& other) {
    LOCK_THIS_AND_OTHER(*this, other);
    this->identity = std::move(other.identity);
    this->blob = std::move(other.blob);
    this->ambiguous = std::move(other.ambiguous);
    this->distanceMatrix = std::move(other.distanceMatrix.clone());
  }



  /**
   * Default move assignment
   */
  BodyPart& BodyPart::operator=(BodyPart&& other) {
    if (&other != this) {
      LOCK_THIS_AND_OTHER(*this, other);
      std::swap(this->blob, other.blob);
      std::swap(this->distanceMatrix, other.distanceMatrix);
      std::swap(this->identity, other.identity);
      std::swap(this->ambiguous, other.ambiguous);
    }
    return *this;
  }



  void BodyPart::associateWithBetterIdentity(const BodyPart::PartName& previous,
                                             BodyPart::PartName& target) {
    switch(previous) {

    case BodyPart::HEAD:
      switch(target) {
      case BodyPart::LEFT_HAND_HEAD:
      case BodyPart::RIGHT_HAND_HEAD:
      case BodyPart::LEFT_RIGHT_HAND_HEAD:
      case BodyPart::UNKNOWN:
        target = BodyPart::HEAD;
      case BodyPart::HEAD:
        break;
      
      case BodyPart::LEFT_HAND:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association: "
               << "HEAD-->LEFT_HAND" << endl;
        break;

      case BodyPart::RIGHT_HAND:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association: "
               << "HEAD-->RIGHT_HAND" << endl;
        break;

      case BodyPart::LEFT_RIGHT_HAND:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association: "
               << "HEAD-->LEFT_RIGHT_HAND" << endl;
        break;

      case BodyPart::OTHER:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "HEAD-->OTHER" << endl;
        break;
      }
      break;

    case BodyPart::LEFT_HAND:
      switch (target) {
      case BodyPart::LEFT_HAND_HEAD:
      case BodyPart::LEFT_RIGHT_HAND_HEAD:
      case BodyPart::LEFT_RIGHT_HAND:
      case BodyPart::UNKNOWN:
        target = BodyPart::LEFT_HAND;
      case BodyPart::LEFT_HAND:
        break;

      case BodyPart::HEAD:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_HAND-->HEAD" << endl;
        break;

      case BodyPart::RIGHT_HAND:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_HAND-->RIGHT_HAND" << endl;
        break;

      case BodyPart::RIGHT_HAND_HEAD:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_HAND-->RIGHT_HAND_HEAD" << endl;
        break;

      case BodyPart::OTHER:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_HAND-->OTHER" << endl;
        break;
      }
      break;

    case BodyPart::RIGHT_HAND:
      switch(target) {
      case BodyPart::RIGHT_HAND_HEAD:
      case BodyPart::LEFT_RIGHT_HAND:
      case BodyPart::LEFT_RIGHT_HAND_HEAD:
      case BodyPart::UNKNOWN:
        target = BodyPart::RIGHT_HAND;
      case BodyPart::RIGHT_HAND:
        break;

      case BodyPart::LEFT_HAND:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "RIGHT_HAND-->LEFT_HAND" << endl;
        break;

      case BodyPart::LEFT_HAND_HEAD:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "RIGHT_HAND-->LEFT_HAND_HEAD" << endl;
        break;

      case BodyPart::HEAD:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "RIGHT_HAND-->HEAD" << endl;
        break;

      case BodyPart::OTHER:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "RIGHT_HAND-->OTHER" << endl;
        break;
      }
      break;

    case BodyPart::LEFT_RIGHT_HAND:
      switch(target) {

      case BodyPart::LEFT_HAND_HEAD:
        target = BodyPart::LEFT_HAND;
        break;

      case BodyPart::RIGHT_HAND_HEAD:
        target = BodyPart::RIGHT_HAND;
        break;

      case BodyPart::UNKNOWN:
      case BodyPart::LEFT_RIGHT_HAND_HEAD:
        target = BodyPart::LEFT_RIGHT_HAND;
        break;

      case BodyPart::LEFT_HAND:
      case BodyPart::RIGHT_HAND:
      case BodyPart::LEFT_RIGHT_HAND:
        break;

      case BodyPart::HEAD:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_RIGHT_HAND-->HEAD" << endl;
        break;

      case BodyPart::OTHER:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_RIGHT_HAND-->OTHER" << endl;
        break;
      }
      break;

    case BodyPart::LEFT_HAND_HEAD:
      switch(target) {
      case BodyPart::RIGHT_HAND_HEAD:
        target = BodyPart::HEAD;
        break;

      case BodyPart::LEFT_HAND:
      case BodyPart::HEAD:
      case BodyPart::LEFT_HAND_HEAD:
        break;

      case BodyPart::LEFT_RIGHT_HAND:
        target = BodyPart::LEFT_HAND;
        break;

      case BodyPart::UNKNOWN:
      case BodyPart::LEFT_RIGHT_HAND_HEAD:
        target = previous;
        break;

      case BodyPart::OTHER:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_HAND_HEAD-->OTHER" << endl;
        break;

      case BodyPart::RIGHT_HAND:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_HAND_HEAD-->RIGHT_HAND" << endl;
        break;
      }
      break;

    case BodyPart::RIGHT_HAND_HEAD:
      switch(target) {
      case BodyPart::LEFT_HAND_HEAD:
        target = BodyPart::HEAD;
        break;

      case BodyPart::RIGHT_HAND:
      case BodyPart::HEAD:
      case BodyPart::RIGHT_HAND_HEAD:
        break;

      case BodyPart::LEFT_RIGHT_HAND:
        target = BodyPart::RIGHT_HAND;
        break;

      case BodyPart::UNKNOWN:
      case BodyPart::LEFT_RIGHT_HAND_HEAD:
        target = previous;
        break;

      case BodyPart::OTHER:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "RIGHT_HAND_HEAD-->OTHER" << endl;
        break;

      case BodyPart::LEFT_HAND:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "RIGHT_HAND_HEAD-->LEFT_HAND" << endl;
        break;
      }
      break;


    case BodyPart::LEFT_RIGHT_HAND_HEAD:
      switch(target) {
      case BodyPart::LEFT_HAND:
      case BodyPart::RIGHT_HAND:
      case BodyPart::HEAD:
      case BodyPart::LEFT_RIGHT_HAND:
      case BodyPart::LEFT_RIGHT_HAND_HEAD:
      case BodyPart::RIGHT_HAND_HEAD:
      case BodyPart::LEFT_HAND_HEAD:
        break;

      case BodyPart::UNKNOWN:
        target = BodyPart::LEFT_RIGHT_HAND_HEAD;
        break;

      case BodyPart::OTHER:
        if (slmotion::debug > 1) 
          cerr << "Warning: Possible ambiguous case detected when using previous frame information to correct feature body part association. " 
               << "LEFT_RIGHT_HAND_HEAD-->OTHER" << endl;
        break;
      }

    default:
      break;
    }
  }
}
