#ifndef SLMOTION_BODYPART
#define SLMOTION_BODYPART
#include "Blob.hpp"
#include "Thread.hpp"

namespace slmotion {
  // forward declaration
  class Blob;

  /**
   * A simple class that represents body parts. Each body part is
   * associated with an identity that may span over several "real life"
   * body parts, and a boolean value called "ambiguous" which is set for
   * those body parts that were obtained from ambiguous or not-good frames.
   */
  class BodyPart {
  public:
    /**
     * PartNames can be used as identity descriptors with BodyPart objects.
     * Combined names represent body parts whose blobs have merged.
     */
    enum PartName : char {
      UNKNOWN = 0, // can also be used as an "any" part
      HEAD = 1,
      LEFT_HAND = 2,
      LEFT_HAND_HEAD = 3, // = left hand + head
      RIGHT_HAND = 4,
      RIGHT_HAND_HEAD = 5,
      LEFT_RIGHT_HAND = 6,
      LEFT_RIGHT_HAND_HEAD = 7, // = all together
      OTHER = 8
    };



    /**
     * Compares the identities between the previous and the target feature
     * identities (body part names), and sets the target to match the previous
     * one, if the previous one is somehow 'stronger' (i.e. HEAD is stronger
     * than LEFT_HAND_HEAD, which is stronger than LEFT_RIGHT_HAND_HEAD etc.)
     *
     * @param previous Previous identity
     * @param target Target identity
     */
    static void associateWithBetterIdentity(const PartName& previous,
                                            PartName& target);



    /**
     * Returns true if the current part is a part of the other (e.g.
     * a HEAD is a HEAD, a HEAD is and LEFT_HAND_HEAD and so on)
     */
    inline bool isA(const BodyPart& other) const {
      return isA(other.identity);
    }



    /**
     * Returns true if the current part is a part of the other (e.g.
     * a HEAD is a HEAD, a HEAD is and LEFT_HAND_HEAD and so on)
     */
    inline bool isA(PartName p) const {
      return (p & this->identity) == p;
    }



    /**
     * Returns true if the LHS part is a part of the RHS part (e.g.
     * a HEAD is a HEAD, a HEAD is and LEFT_HAND_HEAD and so on)
     */
    static inline bool isA(PartName lhs, PartName rhs) {
      return (lhs & rhs) == rhs;
    }

    inline bool operator==(const BodyPart& other) const {
      return this->ambiguous == other.ambiguous &&
        this->identity == other.identity &&
        this->blob == other.blob;
    }



    /**
     * Changes the identity so that the new identity is anything it used to
     * be, and will also be whatever it is set to be.
     *
     * E.g. if the old identity was LEFT_HAND_HEAD, and it is made a 
     * RIGHT_HAND, the new identity would be LEFT_RIGHT_HAND_HEAD
     */
    inline void makeA(PartName b) {
      LOCK_THIS_MUTEX_WHEN_THREADING();
      this->identity = static_cast<PartName>(this->identity | static_cast<char>(b));
    }



    /**
     * Simple constructor:
     * 
     * @param p Identity
     * @param ambiguous Whether the body part is ambiguous
     * @param b Blob associated witht the body part
     */
    BodyPart(PartName p, bool ambiguous, const Blob& b) :
      identity(p),
      blob(b),
      ambiguous(ambiguous) {}



    /**
     * Returns a reference to the blob associated with the body part
     *
     * @return The blob
     */
    inline const Blob& getBlob() const {
      LOCK_THIS_MUTEX_WHEN_THREADING();
      return blob;
    }



    /**
     * Sets a new blob for the body part
     */
    inline void setBlob(const Blob& b) {
      LOCK_THIS_MUTEX_WHEN_THREADING();
      blob = b;
      distanceMatrix = cv::Mat();
    }



    /**
     * Sets the ambiguous value
     */
    inline void setAmbiguous(bool b) {
      ambiguous = b;
    }



    /**
     * Checks if the body part is ambiguous
     * 
     * @return True if ambiguous
     */
    inline bool isAmbiguous() const {
      return ambiguous;
    }



    /**
     * Returns the identity information
     *
     * @return The body part identity
     */
    inline PartName getIdentity() const {
      return identity;
    }



    /**
     * Sets the identity, no questions asked.
     */
    inline void setIdentity(PartName p) {
      identity = p;
    }



    /**
     * Returns the shortest distance (L2 norm) between the blob and the
     * point
     *
     * @param p Point of reference
     * @return If coordinates are non-negative, returns the L2 norm or if 
     * the blob is empty, throw up. If the coordinates are negative, returns
     * positive infinity.
     */
    double distanceTo(const cv::Point& p) const;



    /**
     * Default copy constructor
     */
    BodyPart(const BodyPart&);

    /**
     * Default copy assignment
     */
    BodyPart& operator=(const BodyPart&);

    /**
     * Default move constructor
     */
    BodyPart(BodyPart&&);

    /**
     * Default move assignment
     */
    BodyPart& operator=(BodyPart&&);

    /**
     * Creates an empty body part
     */
    BodyPart() :
      identity(PartName::UNKNOWN),
      ambiguous(true) {}

  private:
    /**
     * Performs distance transform for a matrix of size s, and stores the 
     * result
     */
    void createDistanceMatrix(const cv::Size& s) const;



    PartName identity; ///< Sets the body part -- or combination thereof -- that the object is associated with

    Blob blob; ///< A blob that makes up the pixels the BodyPart is made of

    /// if set true, the part will be considered to have
    /// originated from an ambiguous frame, and care should be
    /// taken when using the object as a reference
    bool ambiguous; 

    mutable cv::Mat distanceMatrix; ///< Matrix of points for distance computation (a cache matrix)
#ifdef SLMOTION_THREADING
    mutable std::mutex mutex; ///< For protecting the distanceMatrix
#endif
  };
}
#endif
