#include "BodyPartCollector.hpp"
#include "FaceDetector.hpp"
#include "SkinDetector.hpp"
#include "BlackBoardDumpWriter.hpp"

using std::cerr;
using std::endl;
using cv::Mat;
using cv::Rect;
using std::vector;
using std::list;
using cv::Point;

namespace slmotion {
  static BodyPartCollector DUMMY(true);

  BodyPartCollector::BodyPartCollector(bool) : Component(true) {
    BlackBoardDumpWriter::registerAnyWriter<std::list<BodyPart> >();

    BlackBoardDumpWriter::registerSizeComputer<BodyPart>([](const BlackBoardDumpWriter * const w, 
                                                            const BodyPart& b) {
                                                           return w->getSize(b.getBlob()) + 2*sizeof(char);
                                                         });
    BlackBoardDumpWriter::registerDumbWriter<BodyPart>([](BlackBoardDumpWriter * const w, std::ostream& ofs, const BodyPart& data) {
        // Stored as follows:
        // [char][blob][char]
        //    |     |     |
        //    |     |     |
        //    |     |     +---bool ambiguous
        //    |     |               
        //    |     +---Associated blob
        //    +---PartName identity                           
        char b = data.getIdentity();
        w->dumbWrite(ofs, b);
        w->dumbWrite(ofs, data.getBlob());
        b = data.isAmbiguous();
        w->dumbWrite(ofs, b);
      });

    BlackBoardDumpReader::registerUnDumper<BodyPart>([](BlackBoardDumpReader* const w, std::istream& ifs) {
        char c;
        w->dumbRead(ifs, c);
        BodyPart outData;
        outData.setIdentity(static_cast<BodyPart::PartName>(c));
        Blob b;
        w->dumbRead(ifs, b);
        outData.setBlob(b);
        w->dumbRead(ifs, c);
        outData.setAmbiguous(c);
        return outData;
      });
  }

  extern int debug;

  void BodyPartCollector::process(frame_number_t frameNumber) {
    if (slmotion::debug > 1)
      cerr << "Detecting body parts, and grouping them, removing unwanted ones." << endl;

    Rect face = *getBlackBoard().get<Rect>(frameNumber, 
                                           FACEDETECTOR_BLACKBOARD_ENTRY);
    BlackBoardPointer<vector<Blob> > blobs = getBlackBoard().get<vector<Blob>>(frameNumber, BLACKBOARD_BLOBS_ENTRY);
    list<BodyPart> historyParts;
    classifyBodyParts(*blobs, face, historyParts);

    getBlackBoard().set<list<BodyPart>>(frameNumber, BODYPARTCOLLECTOR_BLACKBOARD_ENTRY, historyParts);
  }



  size_t BodyPartCollector::classifyBodyParts(const vector<Blob>& blobs, const Rect& faceLocation, list<BodyPart>& bodyParts) {
    // No data can be gathered if no information about the face is available
    if (faceLocation.x == 0 && faceLocation.y == 0 && faceLocation.width == 0
        && faceLocation.height == 0) {

      return 0;
    }

    if (blobs.size() == 0) {
      if (debug > 0)
        cerr << "WARNING: no blobs detected!" << endl;

      return 0;
    }

    Point idealHead = Point(faceLocation.x + faceLocation.width / 2,
                            faceLocation.y + faceLocation.height / 2);
    int headIndex = -1;
    int lhIndex = -1;
    int rhIndex = -1;
    double bestHeadDisplacement = DBL_MAX;
    double displacement;

    bool isGood = false;

    // Begin by assessing if the frame is good (can we rely on what we deduce
    // from the content?)

    // Criteria for goodness:
    // - 3 blobs
    // - One blob (the head) has its centroid close to the centre of the face
    //   rectangle
    // - The two other blobs have their centroids considerably below that of the
    //   head
    // - The hands are horizontally sufficiently far apart (to avoid the case
    //   where one hand is split in two separate blobs, and the hand is then
    //   regarded as two separate hands)

    if (blobs.size() == 3) {
      for (int i = 0; i < (int)blobs.size(); i++) {
        displacement = norm(idealHead - Point(blobs[i].getCentroid()));

        if (displacement < goodFrameMaxHeadDisplacement &&
            displacement < bestHeadDisplacement) {
          headIndex = i;
          bestHeadDisplacement = displacement;
        }
      }

      if (headIndex > -1) {
        lhIndex = (headIndex == 0 ? 1 : 0); // Anything but the same
        rhIndex = 0;
        while(rhIndex == lhIndex || rhIndex == headIndex)
          rhIndex++;
        // Now all three indices should be different
        // Then check the order and reverse if necessary

        // Note that what is called a left hand here is the hand that can be seen
        // on the left-hand side of the screen (usually the right hand because of
        // mirroring)

        if (blobs[lhIndex].getCentroid().x > blobs[rhIndex].getCentroid().x) {
          int t = lhIndex;
          lhIndex = rhIndex;
          rhIndex = t;
        }

        // then verify their vertical distance from the head
        double lhToHead = blobs[lhIndex].getCentroid().y - blobs[headIndex].getCentroid().y;
        double rhToHead = blobs[rhIndex].getCentroid().y - blobs[headIndex].getCentroid().y;
        if (slmotion::debug > 3)
          std::cerr << "slmotion: lh->head blob vertical distance: "
                    << lhToHead << std::endl
                    << "slmotion: rh->head blob vertical distance: "
                    << rhToHead << std::endl;
        if (lhToHead > goodFrameMinHandHeadVDist &&
            rhToHead > goodFrameMinHandHeadVDist) {
          // And finally, the horizontal distance between hands
          if (blobs[rhIndex].getCentroid().x - blobs[lhIndex].getCentroid().x >
              goodFrameMinHandHDist)
            isGood = true;
          else {
            if (slmotion::debug > 2)
              cerr << "Among three distinct blobs, a head was found above the " <<
                "hands, but the hands were too close horizontally." << endl;
          }
        }
        else if (slmotion::debug > 2)
          cerr << "3 blobs were found, but one of the hand blobs is too near " <<
            "the head head (vertically)" << endl;
      }
      else if (slmotion::debug > 2)
        cerr << "A frame with 3 blobs encountered, but no reliable head blob" <<
          " could be found." << endl;
	
    }

    if (slmotion::debug > 1 && !isGood)
      cerr << "The frame is not good (blobwise)." << endl;
  
    if (isGood) {
      // This part of analysis could be done separately as a preprocessing
      // stage

      if (slmotion::debug > 1)
        cerr << "The frame is good (blobwise)" << endl << "Updating body "<<
          "parts, and using these parts as past reference in the future." <<
          endl;
      // oldHead.setBlob(blobs[headIndex]);
      // oldHead.setAmbiguous(false);

      bodyParts.push_back(BodyPart(BodyPart::HEAD, false, blobs[headIndex]));
      bodyParts.push_back(BodyPart(BodyPart::LEFT_HAND, false, blobs[lhIndex]));
      bodyParts.push_back(BodyPart(BodyPart::RIGHT_HAND, false, blobs[rhIndex]));

      return 3;
    }
    else if (blobs.size() > 3) {
      cerr << "WARNING: a blob with more than three blobs detected" << endl;
      cerr << "Doing nothing." << endl;
      return 0;
    }
    else if (blobs.size() == 3) {
      cerr << "WARNING: a bad frame with three blobs detected. Doing nothing." <<
        endl;
      return 0;
    } // We now have either 1 or 2 blobs
    else if (blobs.size() == 1) {
      // We know that everything MUST be together
      bodyParts.push_back(BodyPart(BodyPart::LEFT_RIGHT_HAND_HEAD, true, blobs[0]));

      // if (opts.useBodyPartCache)
      //   bodyPartCache.back().push_back(BodyPart(BodyPart::LEFT_RIGHT_HAND_HEAD, true, blobs[0]));

      return 1;
    }
    else if (blobs.size() == 2) {
      // We must determine (somehow) if the head is separate or not
      const Blob& b1 = blobs[0];
      const Blob& b2 = blobs[1];
    
      // Steps: select the blob whose centroid is closer to the head, then
      // consider its size. If it is too big, the head must be merged. Let us
      // keep 20 per cent increase in size as the criterion.

      const Blob& hcand = norm(Point(blobs[0].getCentroid()) - idealHead) <
        norm(Point(blobs[1].getCentroid()) - idealHead) ? blobs[0] : blobs[1];

      if (slmotion::debug > 3) {
        cerr << "Two blobs. Blob " << (&hcand == &b1 ? "1" : 
                                       &hcand == &b2 ? "2" : "ERROR") <<
          " is closer to the head." << endl <<
          "Testing if the head is separate." << endl;
      }

      bool separateHead;
      // if (oldHead.isAmbiguous()) {
        // if (slmotion::debug > 3)
        //   cerr << "Old head blob is ambiguous, using face detection heuristics" << endl;

        separateHead = hcand.size() > classifyHeadSizeFactor * faceLocation.width * faceLocation.height ? false : true;
        if (!separateHead) {
          if (slmotion::debug > 3)
            cerr << "The blob is too great. Head cannot be separate." << endl;
        }
        else {
          separateHead = norm(Point(hcand.getCentroid()) - idealHead) >
            classifyHeadMaxDistance ? false : true;
          if (!separateHead && slmotion::debug > 3)
            cerr << "The centroid is too far away. The head cannot be separate." << endl;
        }	
      // }
      // else {
      //   separateHead = hcand.size() > oldHead.getBlob().size() * 1.2 ? false : true;
      // }

      if (separateHead) {
        if (slmotion::debug > 3)
          cerr << "The head appears to be separate." << endl;
        bodyParts.push_back(BodyPart(BodyPart::HEAD, true, hcand));
        bodyParts.push_back(BodyPart(BodyPart::LEFT_RIGHT_HAND, true,
                                     &hcand == &b1 ? b2 : b1));

        return 2;
      }
      else {
        if (slmotion::debug > 3)
          cerr << "The head appears to be merged with a hand." << endl;
        const Blob& other = &hcand == &b1 ? b2 : b1;

        if (hcand.getCentroid().x < other.getCentroid().x) {
          bodyParts.push_back(BodyPart(BodyPart::LEFT_HAND_HEAD,true,hcand));
          bodyParts.push_back(BodyPart(BodyPart::RIGHT_HAND, true, other));

          return 2;
        }
        else {
          bodyParts.push_back(BodyPart(BodyPart::RIGHT_HAND_HEAD, true, hcand));
          bodyParts.push_back(BodyPart(BodyPart::LEFT_HAND, true, other));

          return 2;
        }
      }
    }
    else {
      cerr << "Something totally unexpected happened" << endl;
      return 0;
    }
  }



  Component* BodyPartCollector::createComponentImpl(const boost::program_options::variables_map&, BlackBoard* blackBoard, FrameSource* frameSource) const {
    return new BodyPartCollector(blackBoard, frameSource);
  }
}
