#ifndef SLMOTION_BODY_PART_COLLECTOR
#define SLMOTION_BODY_PART_COLLECTOR

#include "Component.hpp"
#include "Blob.hpp"
#include "BodyPart.hpp"
#include "FaceDetector.hpp"
#include "SkinDetector.hpp"
#include "BlobExtractor.hpp"

namespace slmotion {
  const std::string BODYPARTCOLLECTOR_BLACKBOARD_ENTRY = "bodyparts";
  // std::list<BodyPart>
  // contains the detected body parts in an unspecified order



  /**
   * This component expects to find blobs, and then assigns them identities
   * as body parts in some fashion
   */
  class BodyPartCollector : public Component {
  public:
    explicit BodyPartCollector(bool);



    /**
     * Rather a trivial constructor
     */
    BodyPartCollector(BlackBoard* blackBoard, FrameSource* frameSource) :
      Component(blackBoard, frameSource),
      goodFrameMaxHeadDisplacement(1000), goodFrameMinHandHeadVDist(-1000),
      goodFrameMinHandHDist(0), classifyHeadMaxDistance(50),
      classifyHeadSizeFactor(3.4) { }



    virtual std::string getShortDescription() const { 
      return "Labels blobs as body parts";
    };



    virtual std::string getLongDescription() const { 
      return "Labels blobs as body parts";
    };



    virtual std::string getShortName() const {
      return "BodyPartCollector";
    }



    virtual std::string getComponentName() const {
      return "Body Part Collector";
    }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const {
      return boost::program_options::options_description();
    }

    virtual void reset() { }

    virtual void process(frame_number_t frameNumber);



    inline virtual property_set_t getProvided() const {
      return property_set_t { BODYPARTCOLLECTOR_BLACKBOARD_ENTRY } ;
    }



    inline virtual property_set_t getRequirements() const {
      return property_set_t { FACEDETECTOR_BLACKBOARD_ENTRY,
          BLACKBOARD_BLOBS_ENTRY };
    }

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;

    /**
     * Go through blobs in the vector, and assign them identities as 
     * BodyParts
     * 
     * @param blobs Blobs that need classifying
     * @param faceLocation the face as found by the face detector
     * @param bodyParts An output vector for classified body parts. The 
     * vector will be cleared.
     * @param frnumber The frame number. This is only significant if body 
     * part caching is enabled.
     * @param clear If set, bodyParts will be cleared. Otherwise, body parts
     * found are merely appended
     *
     * @return The total number of body parts classified
     */
    size_t classifyBodyParts(const std::vector<Blob>& blobs, 
                             const cv::Rect& faceLocation, 
                             std::list<BodyPart>& bodyParts);



    int goodFrameMaxHeadDisplacement; ///< the maximum number of pixels that the head blob may be displaced from its ideal location (i.e. where the face is according to the HAAR classifier)
    int goodFrameMinHandHeadVDist; ///< The minimum vertical distance between the centroids of hand and head blobs that allows a frame to be considered good
    int goodFrameMinHandHDist; ///< The same but for the horizontal distance between the hands
    int classifyHeadMaxDistance; ///< When determining body parts, the maximum distance allowed for the centroid of the separate head blob from the center of the detected face
    double classifyHeadSizeFactor; ///< When determining body parts, use this factor to determine if the possible head blob is too large with respect to the detected face to be considered a valid head
  };
}

#endif

