#ifndef SLMOTION_COLOUR_SPACE
#define SLMOTION_COLOUR_SPACE

#include <memory>

#include "opencv.hpp"

#include "Clonable.hpp"

namespace slmotion {
  /**
   * This class represents a 'colour space', mainly used by the skin 
   * detector. A colour space used for classification may consist of 
   * independent components from several different colourspaces, so the 
   * simple cv::cvtColor function will not suffice.
   *
   * This is an abstract base class. Subclasses ought to be instantiated.
   */
  class ColourSpace : public Clonable {
  public:
    // Some default colour spaces
    const static std::unique_ptr<const ColourSpace> YCRCB;
    const static std::unique_ptr<const ColourSpace> RGB;
    const static std::unique_ptr<const ColourSpace> BGR;
    const static std::unique_ptr<const ColourSpace> HSV;
    const static std::unique_ptr<const ColourSpace> GREY;
    const static std::unique_ptr<const ColourSpace> XYZ;
    // const static std::unique_ptr<const ColourSpace> CGZ;
    const static std::unique_ptr<const ColourSpace> IHLS;

    enum ColourComponent {
      R, // Red
      G, // Green
      B, // Blue
      Y, // Luma
      CR, // Red difference
      CB, // Blue difference
      H, // Hue
      S, // Saturation
      V, // Value
      X, // XYZ components
      XYZ_Y,
      Z,
      //CHONG, // These are placeholders for the CGZ colour space
      //GORTLER,
      //ZICKLER,
      IHLS_Y, // these are for IHLS; luminance
      IHLS_S, // saturation
      IHLS_H // hue
    };


    
    ColourSpace(std::initializer_list<ColourComponent> init) :
      components(init) {}



    ColourSpace() {}



    virtual ~ColourSpace() = 0;



    inline void addComponent(ColourComponent add) {
      components.push_back(add);
    }



    /**
     * Converts a matrix to the colour space.
     * Subclasses may override this function with a special, faster 
     * implementation for a specific colour space.
     *
     * @param in Input matrix (assumed to be 3-Channel BGR)
     * @param out Output matrix. Channel number matches that of 
     * components. Type is not guaranteed to be preserved.
     */
    virtual void convertColour(const cv::Mat& in, cv::Mat& out) const;



    inline bool operator==(const ColourSpace& other) const {
      return components == other.components;
    }



    virtual ColourSpace* clone() const = 0;



  private:
    std::vector<ColourComponent> components;
  };



  // specialised sub colour spaces, these are rather unimportant from the 
  // user's point of view, since they can be used in a Singleton-like manner
  // through the static member pointers in ColourSpace class


  class CustomColourSpace : public ColourSpace {
  public:
    CustomColourSpace(std::initializer_list<ColourComponent> init) :
      ColourSpace(init) {}

    CustomColourSpace() {}

    inline CustomColourSpace* clone() const {
      return new CustomColourSpace(*this);
    }
  };



  class YCrCbColourSpace : public ColourSpace {
  public:
    YCrCbColourSpace() :
      ColourSpace({ColourSpace::Y, ColourSpace::CR, ColourSpace::CB}) {}

    inline void convertColour(const cv::Mat& in, cv::Mat& out) const {
      assert(in.data != NULL && in.data != out.data);
      cv::cvtColor(in, out, CV_BGR2YCrCb);
    }

    inline YCrCbColourSpace* clone() const {
      return new YCrCbColourSpace(*this);
    }
  };



  class RGBColourSpace : public ColourSpace {
  public:
    RGBColourSpace() :
      ColourSpace({ColourSpace::R, ColourSpace::G, ColourSpace::B}) {}

    inline void convertColour(const cv::Mat& in, cv::Mat& out) const {
      assert(in.data != NULL && in.data != out.data);
      cv::cvtColor(in, out, CV_BGR2RGB);
    }

    inline RGBColourSpace* clone() const {
      return new RGBColourSpace(*this);
    }
  };



  class BGRColourSpace : public ColourSpace {
  public:
    BGRColourSpace() :
      ColourSpace({ColourSpace::B, ColourSpace::G, ColourSpace::R}) {}

    inline void convertColour(const cv::Mat& in, cv::Mat& out) const {
      assert(in.data != NULL && in.data != out.data); 
      // no conversion needed
      in.copyTo(out);
    }

    inline BGRColourSpace* clone() const {
      return new BGRColourSpace(*this);
    }
  };



  class HSVColourSpace : public ColourSpace {
  public:
    HSVColourSpace() :
      ColourSpace({ColourSpace::H, ColourSpace::S, ColourSpace::V}) {}

    inline void convertColour(const cv::Mat& in, cv::Mat& out) const {
      assert(in.data != NULL && in.data != out.data);
      cv::cvtColor(in, out, CV_BGR2HSV);
    }

    inline HSVColourSpace* clone() const {
      return new HSVColourSpace(*this);
    }
  };



  class GreyColourSpace : public ColourSpace {
  public:
    GreyColourSpace() :
      ColourSpace({ColourSpace::Y}) {}

    inline void convertColour(const cv::Mat& in, cv::Mat& out) const {
      assert(in.data != NULL && in.data != out.data);
      cv::cvtColor(in, out, CV_BGR2GRAY);
    }

    inline GreyColourSpace* clone() const {
      return new GreyColourSpace(*this);
    }
  };



  class XYZColourSpace : public ColourSpace {
  public:
    XYZColourSpace() :
      ColourSpace({ColourSpace::X, ColourSpace::XYZ_Y, ColourSpace::Z}) {}

    inline void convertColour(const cv::Mat& in, cv::Mat& out) const {
      assert(in.data != NULL && in.data != out.data);
      cv::cvtColor(in, out, CV_BGR2XYZ);
    }

    inline XYZColourSpace* clone() const {
      return new XYZColourSpace(*this);
    }
  };



  /**
   * Perception-based illumination invariant colour space from 
   *   Chong, Gortler, Zickler 2008
   */
#if 0
  // current implementation is incompatible with clang
  class PerceptualColourSpace : public ColourSpace {
  public:
    PerceptualColourSpace() :
      ColourSpace({ColourSpace::CHONG, ColourSpace::GORTLER,
            ColourSpace::ZICKLER}) {}

    void convertColour(const cv::Mat& in, cv::Mat& out) const;

    PerceptualColourSpace* clone() const {
      return new PerceptualColourSpace(*this);
    }
  };
#endif



  /**
   * Improved HLS colour space, description can be found in Allan Hanbury, 
   *  Jean Serra. A 3D-polar Coordinate Colour Representation Suitable for
   *  Image Analysis.
   */
  class IHLSColourSpace : public ColourSpace {
  public:
    IHLSColourSpace() :
      ColourSpace({ColourSpace::IHLS_H, ColourSpace::IHLS_Y, 
            ColourSpace::IHLS_S}) {}



    /**
     * Assuming 24 bit BGR input, the transformation is defined as follows:
     *  (B,G,R) -> (Y,S,H)
     *
     * Y  = 0.2126R + 0.7152G + 0.0722B
     * S  = max(R,G,B) - min(R,G,B)
     *             [           R - 1/2 G - 1/2 B          ]
     * H' = arccos [ ------------------------------------ ]
     *             [ sqrt(R^2 + G^2 + B^2 - RG - RB - BG) ]
     *
     *      { 360° - H'  if B > G
     * H  = { H'         otherwise
     *
     * Naturally, H (and other components) will be scaled to the range of 
     * the output element type, i.e. in case of integral elements, H will be
     * scaled to [-MIN,MAX] for the respective type, and [0,1] for floating
     * point types.
     */
    void convertColour(const cv::Mat& in, cv::Mat& out) const;



    inline IHLSColourSpace* clone() const {
      return new IHLSColourSpace(*this);
    }
  };



  /**
   * Creates a colour space of the correct subclass, when appropriate. 
   * Otherwise, a generic colour space is created.
   *
   * It is up to the caller to call delete on the created object.
   *
   * @param comps Colour components
   */
  ColourSpace* createColourSpace(const std::vector<ColourSpace::ColourComponent>& comps);
}

#endif
