#ifndef SLMOTION_COMPONENT
#define SLMOTION_COMPONENT

#include "UiCallback.hpp"
#include <set>
#include <boost/program_options.hpp>
#include "BlackBoard.hpp"
#include "FrameSource.hpp"

namespace slmotion {
  /**
   * This class represents the interface of a generic analysis component
   */
  class Component {
  public:
    typedef size_t frame_number_t;



    /**
     * Returns the names of all registered components and their descriptions
     * @param maxLineWidth Maximum width of a line
     */
    static std::string getRegisteredComponents(size_t maxLineWidth);



    /**
     * Returns the description of component configuration file options for
     * all components
     */
    static boost::program_options::options_description getCompleteConfigurationFileOptionsDescription();

    /**
     * Returns the description of component command line options for
     * all components
     */
    static boost::program_options::options_description getCompleteCommandLineOptionsDescription();


    /**
     * Returns the description of component configuration file options for
     * the given component
     */
    static boost::program_options::options_description getConfigurationFileOptionsDescriptionByComponent(const std::string& name);



    /**
     * Returns the long description for the given component
     */
    static std::string getLongDescriptionForComponent(const std::string& name);



    /**
     * Constructs a new component. It is up to the creator of the object to
     * ensure that the blackboard and the frameSource objects outlive this
     * instance. This class does NOT take ownership of the pointers.
     */
    Component(BlackBoard* blackBoard, FrameSource* frameSource) :
      nextComponent(NULL), blackBoard(blackBoard), frameSource(frameSource)
    {}



    /**
     * This is a very special constructor, intended to be used ONLY
     * when registering new components, called by the static member in
     * the translation unit.
     *
     * DO NOT USE FOR ANYTHING ELSE. Components initialised using this
     * constructor WILL BE INVALID.
     */
    Component(bool);



    /**
     * Default copy-constructor. No need for resetting (no internal state 
     * yet!)
     */
    Component(const Component&) = default;



    /**
     * Perform some processing, then store results in the black board.
     */
    virtual void process(frame_number_t frameNumber) = 0;



    /**
     * Process a range of frames. By default, just call process to each of 
     * these frames. However, this behaviour may be overridden, and some 
     * components may do extra processing (e.g. interpolation)
     *
     * Last frame is exclusive, first inclusive, i.e. do something for 
     * frames in [first,last)
     *
     * If first < last, an exception is thrown.
     * If last > the total number of frames, the total number of frames is 
     * passed instead.
     *
     * Implementation is provided by a separate function that may be 
     * overridden by any subclass.
     * 
     * @param uiCallback Possible user interface callback. The
     * implementation function should call this function every now and
     * then to notify the parent program of its status (current frame)
     * if applicable. This may also be ignored, and the function
     * should be able to handle a NULL pointer.  If the callback
     * function returns false, execution should be terminated
     * prematurely.
     *
     * @return True on successful termination, false otherwise (including 
     * false from callback)
     */
    bool processRange(frame_number_t first = 0, 
                      frame_number_t last = SIZE_MAX,
                      UiCallback* uiCallback = nullptr);



    /**
     * Standard copy-assignment. Resets this object (except in the case of
     * self-assignment, of course)
     */
    Component& operator=(const Component& another); 



    virtual ~Component() = 0;



    inline const BlackBoard& getBlackBoard() const {
      return *blackBoard;
    }



    inline const FrameSource& getFrameSource() const {
      return *frameSource;
    }



    inline BlackBoard& getBlackBoard() {
      return *blackBoard;
    }



    inline FrameSource& getFrameSource() {
      return *frameSource;
    }

   
    
    typedef std::set<BlackBoard::property_key_t> property_set_t;



    /**
     * Constructs the component using the variables and returns a
     * pointer to the new instance.
     *
     * @param componentName Component identifier i.e. getShortName()
     * of the desired component
     * @param configuration Configuration file configuration used to set up
     * the new component
     * @param blackBoard The associated black board
     * @param frameSource The associated frame source
     *
     * @return A pointer to the new component instance, or NULL if no such
     * component can be constructed.
     */
    Component* createComponent(const std::string& componentName,
                               const boost::program_options::variables_map& configuration, 
                               BlackBoard* blackBoard, FrameSource* frameSource) const;



    /**
     * Constructs the component using the variables and returns a
     * pointer to the new instance. This version is used in particular
     * when instantiating components from Python side.
     *
     * @param componentName Component identifier i.e. getShortName()
     * of the desired component
     * @param configuration Configuration for the component. The keys should not
     * be prefixed.
     * @param blackBoard The associated black board
     * @param frameSource The associated frame source
     *
     * @return A pointer to the new component instance, or NULL if no such
     * component can be constructed.
     */
    Component* createComponent(const std::string& componentName,
                               const std::map<std::string, boost::any>& configuration, 
                               const boost::program_options::variables_map* commandLineOptions,
                               BlackBoard* blackBoard, FrameSource* frameSource) const;



#define SLMOTION_DISABLE_DEFAULT_COMPONENT_MEMBER_DEFINITIONS 1
#if SLMOTION_DISABLE_DEFAULT_COMPONENT_MEMBER_DEFINITIONS
    /**
     * Returns a short description of the component that can be used e.g. in the
     * command line component list
     */
    virtual std::string getShortDescription() const = 0;



    /**
     * Returns a long and detailed component description that can be
     * used e.g. to generate the default configuration file.
     */
    virtual std::string getLongDescription() const = 0;



    /**
     * Returns a set of property keys which must be found on the blackboard 
     * for each frame that shall be processed by the component (if 
     * available)
     */
    virtual property_set_t getRequirements() const = 0;



    /**
     * Returns a set of property keys which will be created or updated (i.e.
     * provided) by this component on the blackboard after processing has 
     * been performed
     */
    virtual property_set_t getProvided() const = 0;



    /**
     * Returns a short UNIQUE name that identifies the component (and
     * can be used as a command line argument, determining the
     * components in the process chain)
     */
    virtual std::string getShortName() const = 0;



    /**
     * Returns the long human-readable name of the component
     * (e.g. Face Detector)
     */
    virtual std::string getComponentName() const = 0;



    /**
     * Cleans up any internal state the component may or may not have (e.g.
     * internal caches). This function should be called upon assignment.
     */
    virtual void reset() = 0;



    /**
     * Returns a description of options for the configuration file used by
     * the individual component (with default values)
     */
    // virtual configuration::ConfigurationFileOptionsDescription getConfigurationFileOptionsDescription() const = 0;
    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const = 0;

    /**
     * Returns a description of options for the command line used by
     * the individual component (with default values)
     */
    virtual boost::program_options::options_description getCommandLineOptionsDescription() const = 0;



    /**
     * Returns the names of all components that provide the desired property
     */
    static std::vector<std::string> findProviders(const std::string& property);



  private:
    /**
     * Constructs the component using the variables and returns a
     * pointer to the new instance. This is the actual implementation; this 
     * is where the magic happens.
     *
     * @param configuration Configuration file configuration used to set up
     * the new component
     *
     * @return A pointer to the new component instance, or NULL if no such
     * component can be constructed.
     */
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const = 0;



    /**
     * Constructs the component using the variables and returns a
     * pointer to the new instance. This is the actual implementation; this 
     * is where the magic happens.
     *
     * @param configuration Configuration file configuration used to set up
     * the new component
     *
     * @return A pointer to the new component instance, or NULL if no such
     * component can be constructed.
     */
    virtual Component* createComponentImpl(const std::map<std::string, boost::any>& configuration, 
                                           const boost::program_options::variables_map* commandLineOptions,
                                           BlackBoard* blackBoard, FrameSource* frameSource) const;
#else
    /**
     * Returns a short description of the component that can be used e.g. in the
     * command line component list
     */
    virtual std::string getShortDescription() const { return ""; };



    /**
     * Returns a long and detailed component description that can be used e.g. 
     * to generate the default configuration file.
     */
    virtual std::string getLongDescription() const { return ""; };



    /**
     * Returns a set of property keys which must be found on the blackboard 
     * for each frame that shall be processed by the component (if 
     * available)
     */
    virtual property_set_t getRequirements() const {
      return property_set_t();
    }



    /**
     * Returns a set of property keys which will be created or updated (i.e.
     * provided) by this component on the blackboard after processing has 
     * been performed
     */
    virtual property_set_t getProvided() const {
      return property_set_t();
    }



    /**
     * Returns a short UNIQUE name that identifies the component (and
     * can be used as a command line argument, determining the
     * components in the process chain)
     */
    virtual std::string getShortName() const {
      return "";
    }



    /**
     * Returns the long human-readable name of the component
     * (e.g. Face Detector)
     */
    virtual std::string getComponentName() const {
      return "Component";
    }

    /**
     * Cleans up any internal state the component may or may not have (e.g.
     * internal caches). This function should be called upon assignment.
     */
    virtual void reset() { }



    /**
     * Returns a description of options for the configuration file used by
     * the individual component (with default values)
     */
    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;



  private:
    /**
     * Constructs the component using the variables and returns a
     * pointer to the new instance. This is the actual implementation; this 
     * is where the magic happens.
     *
     * @param configuration Configuration file configuration used to set up
     * the new component
     *
     * @return A pointer to the new component instance, or NULL if no such
     * component can be constructed.
     */
    virtual Component* createComponentImpl(const boost::program_options::variables_map&, BlackBoard*, FrameSource*) const {
      return NULL;
    }

#endif

  protected:
    /**
     * Implementation for the processRange. ProcessRange is guaranteed to 
     * call this function with a valid range with
     *   first in [0,frameSource->size())
     *   last in (first,frameSource->size()]
     *
     * Performs some processing for frames in [first,last), stores results 
     * on the black board.
     *
     * Default action: call process for each frame, separately. May be 
     * overridden in any way.
     *
     * NOTE: If multithreading is enabled, this default implementation will
     * assume that calling process(i) for any frame number i is INDEPENDENT
     * of any other calls with any other i, i.e. the result ought to be the
     * same no matter what order calls are made and which frames the call is
     * made upon. If this is not the case, this function will need to be 
     * overridden, as the default implementation may not guarantee the order
     * in which calls are made..
     *
     * @param first First frame to process (inclusive)
     * @param last Last frame to process (exclusive
     * @param uiCallback Possible user interface callback. The function 
     * should call this function every now and then to notify the parent 
     * program of its status (current frame) if applicable. This may also 
     * be ignored, and the function should be able to handle a NULL pointer.
     * If the callback function returns false, execution should be 
     * terminated prematurely.
     *
     * @return True on successful termination, false otherwise (including 
     * false from callback)
     */
    virtual bool processRangeImplementation(frame_number_t first, 
                                            frame_number_t last, 
                                            UiCallback* uiCallback);



  private:
    /**
     * This is a very special member only to be used when registering
     * components. DO NOT USE FOR ANYTHING ELSE.
     */
    const Component* nextComponent;
    BlackBoard* blackBoard;
    FrameSource* frameSource;
  };



  /**
   * This is a list of all registered components
   */
  extern const Component* COMPONENT_LIST;
}

#endif
