#ifndef SLMOTION_ELM_CLASSIFIER
#define SLMOTION_ELM_CLASSIFIER

#include "BinaryClassifier.hpp"
#include <map>
#include <vector>

namespace slmotion {
  inline float sigmoid(float x) {
    return 1./(1.+exp(-x));
  }



  /**
   * This class represents a classifier based on an Extreme Learning
   * Machine (ELM). Implementation is based on work by Huang et
   * al. (see: Guang-Bin Huang & Qin-Yu Zhu & Chee-Kheong
   * Siew. Extreme learning machine: Theory and
   * applications. 2006. Neurocomputing 70. pp. 489--501). The class
   * attempts to implement a vanilla version of the ELM. It is similar
   * to the one below but simpler and is written using OpenCV routines
   * only.
   */
  class CvElmClassifier {
  public:
    /**
     * Constructs an empty classifier.
     */
    CvElmClassifier() : g(&sigmoid) {}



    /**
     * Loads a model from a file (in the format produced by toString())
     */
    CvElmClassifier(const std::string& filename);
    


    /**
     * Trains the ELM.
     *
     * @param samples Samples as row vectors (CV_32FC1) 
     * @param desiredOutputs Desired outputs (scalars, CV_32FC1),
     * number of rows must match samples
     * @param N The number of neurons in the hidden layer
     * @param activationFunction float --> float activation function
     */
    void train(const cv::Mat& samples, const cv::Mat& desiredOutputs, size_t N, 
               float(*activationFunction)(float));


    
    /**
     * Returns the class number (either 1 or 0) and also the ratio of
     * the class values (somehow predicting how certain the prediction
     * is)
     */
    float predict(const cv::Vec3b& sample) const; 

    /**
     * Forms a textual representation of the trained ELM (e.g., for
     * saving to a file)
     */
    std::string toString() const;



    /**
     * Basic copy constructor. Deep copies of member matrices are made.
     */
    CvElmClassifier(const CvElmClassifier& that) : W(that.W.clone()),
                                                   b(that.b.clone()),
                                                   betas(that.betas.clone()),
                                                   g(that.g),
                                                   means(that.means.clone()),
                                                   stddevs(that.stddevs.clone())
    {}



    /**
     * Basic move constructor. Shallow copies are made (moved)
     */
    CvElmClassifier(CvElmClassifier&& that) : W(std::move(that.W)),
                                              b(std::move(that.b)),
                                              betas(std::move(that.betas)),
                                              g(that.g),
                                              means(std::move(that.means)),
                                              stddevs(std::move(that.stddevs))
    {}



    /**
     * Basic copy-assignment. Deep copies of member matrices are made.
     */
    CvElmClassifier& operator=(const CvElmClassifier& that);



    /**
     * Basic move-assignment. Shallow moves.
     */
    CvElmClassifier& operator=(CvElmClassifier&& that);



  private:
    cv::Mat W; // neurons in the hidden layer (N x 3)
    cv::Mat b; // bias terms (N x 1)
    cv::Mat betas; // output neuron weights (1 x N)
    float(*g)(float); // the activation function
    cv::Mat means; // means for computing zscore
    cv::Mat stddevs; // stddevs for computing zscore
  };


  typedef std::vector<std::vector<float> > regressionPath;
  

  struct trainedELM{
    floatMatrix weightM; // size (Nneur_sigmoid*(dim+1)) (weight vectors of the sigmoidal neurons)
    floatMatrix RBFcenters; // size (Nneur_RBF*dim)
    floatMatrix RBFwidths; // size NNeur_RBF * 1 (RBF kernel widths)
    floatMatrix chisqcenters; // size (Nneur_chisq*dim)
    floatMatrix chisqwidths; // size NNeur_chisq * 1 (chisq kernel widths)
    floatMatrix beta; // size (Nneur_sigmoid+Nneur_RBF+Nneur_chisq) * 1
  };

  class ELMClassifier :  public BinaryClassifier {

  public:

    virtual void train(const std::vector<std::vector<float> > &negSamples,
		       const std::vector<std::vector<float> > &posSamples,
		       const std::vector<bool> *dimMask=NULL,
		       const std::vector<float> *negWeights=NULL,
		       const std::vector<float> *posWeights=NULL);

    virtual void train(const floatMatrix &datamat);

    virtual float predict(const std::vector<float>  &sample,
			  const std::vector<bool> *dimMask=NULL);

    virtual float predict_lean(const std::vector<float>  &sample);

    virtual bool readModelFromFile(const std::string &fn);

    virtual void writeModelToFile(const std::string &fn);

    virtual void showParam();

    ELMClassifier(){
      // weightM=beta=NULL;
      maxcachedim=-1;

      useznorm=true;

      hardlimitresponses=true;

      useOPELMpruning=false;

      Nneur_sigmoid=150;

      Nneur_RBF=0;

      Nneur_chisq=0;

      ensemblesize=1;

      fusionoperator="mean";

      transferfunction_spec="tanh";

    }

    ~ELMClassifier(){
      //  delete[] weightM;
      // delete[] beta;
    }


    

    std::vector<trainedELM> trainedparam;

    //    float *weightM; // after training: Nneur * (dim+1) matrix in column major order
    // float *beta; // after training: Nneur * 1 matrix in column major order
    // int dim;
    // int Nneur;

    bool useznorm;

    bool hardlimitresponses;
    
    bool useOPELMpruning;

    int Nneur_sigmoid;

    int Nneur_RBF;

    int Nneur_chisq;

    int ensemblesize;

    std::string fusionoperator;

    std::string transferfunction_spec;


    float (*transferfunction)(float);

    std::vector<float> invsqrtvariance;

    // means stored in parent class


    int maxcachedim; // value >=0 signals the use of caching

    std::map<int,float> predictioncache;
  };

  bool readText(FILE *f,const char *t);

  bool readFloatVectorFromFile(FILE *f, std::vector<float> &v);
  void writeFloatVectorToFile(FILE *f, const std::vector<float> &v);
  int uniform(int m);
  float unifloat();
  float	evalEuclDist(const float *m1, int r1,int rows1,const float *m2, int r2,int rows2,int dim);
  float	evalChisqDist(const float *m1, int r1,int rows1,const float *m2, int r2,int rows2,int dim);

  void stagewise(const floatMatrix &regressors, const floatMatrix &tgt, float steplen, regressionPath &p);

  void lars(const floatMatrix &regressors, const floatMatrix &tgt, regressionPath &p);

  void normaliseColumns(floatMatrix &m,bool unitlen=false);

 void findBestNeurons(float *KM, float *tgt,int nsamples, int nneur, 
		      std::vector<int> &bestNeurons);


}
#endif
