#include "EafDocument.hpp"
#include "configuration.hpp"
#include "exceptions.hpp"
#include "xmlutil.hpp"
#include "util.hpp"
#include <cstring>
#include <boost/lexical_cast.hpp>
#include <cassert>

using std::string;
using std::vector;
using std::map;

namespace slmotion {
  std::string getEafSchemaFile() {
    return getInstallPrefix() + "/share/slmotion/EAFv2.6.xsd";
  }



  void EafDocument::initialise(xml::XmlNode node) {  
    try {
      // check the document format, and create internal representations of timecodes
      if (node.getName() != "ANNOTATION_DOCUMENT")
        throw ConvertException("ANNOTATION_DOCUMENT was expected to be the root node.");

      // Add timecodes
      // node = xmlFirstElementChild(node);

      // while(node && strcmp(toChar(node->name), "TIME_ORDER") != 0)
      //   node = xmlNextElementSibling(node);
      node = node.getChild("TIME_ORDER");

      // if (!node)
      //   throw ConvertException("TIME_ORDER element could not be found.");

      // node = xmlFirstElementChild(node2.getInternalNode());

      // while (node) {
      for (auto it = node.begin(); it != node.end(); ++it) {
        if (it->getName() != "TIME_SLOT")  
          throw ConvertException(it->getName() + 
                                 " was unexpected. TIME_SLOT was expected.");
      
        string id = it->getProperty("TIME_SLOT_ID");

        if (timeCodes.count(id))
          throw ConvertException(("Duplicate timecodes with the id " + id + " detected.").c_str());

        timeCodes[id] = it->getProperty("TIME_VALUE");
      }

      // Add annotation and tier IDs
      for (auto tierNode = getRoot().begin(); tierNode != getRoot().end(); ++tierNode) {
        if (tierNode->getName() ==  "TIER") {
          string tierId = tierNode->getProperty("TIER_ID");
          if (tierIds.count(tierId))
            throw ConvertException("Multiple tiers with the same ID encountered!");
          else
            tierIds.insert(tierId);

          for (auto annNode = tierNode->begin(); annNode != tierNode->end(); ++annNode) {
            if (annNode->getName() == "ANNOTATION") {
              for (auto alAnnNode = annNode->begin(); alAnnNode != annNode->end(); 
                   ++alAnnNode) {
                if (alAnnNode->getName() == "ALIGNABLE_ANNOTATION") {
                  string id = alAnnNode->getProperty("ANNOTATION_ID");
                  if (annotationIds.count(id))
                    throw ConvertException("Multiple annotations with the same "
                                           "ID encountered!");
                  annotationIds.insert(id);
                }
              }
            }
          }
        }
      }
    }
    catch(ConvertException& e) {
      throw ConvertException(string("An error occurred on line ") +
                             boost::lexical_cast<string>(node.getLineNumber()) + " while initialising an internal document from an EAF file: " + e.what());
      
    }
  }



  void EafDocument::initialise() {
    xml::XmlNode root("ANNOTATION_DOCUMENT");
    xml::XmlNamespace schemaInstanceNamespace =
      root.newNamespace("http://www.w3.org/2001/XMLSchema-instance", "xsi");

    setRootElement(root);

    schemaInstanceNamespace.newProperty("noNamespaceSchemaLocation", getEafSchemaFile());

    // Set required attributes

    time_t t = time(NULL); // timezone offset in minutes
    struct tm* cal = localtime(&t); // calendar;
    t = cal->tm_gmtoff / 60;
    char timestring[128]; 
  
    sprintf(timestring, "%4d-%02d-%02dT%02d:%02d:%02d%c%02d:%02d",
            cal->tm_year + 1900, cal->tm_mon + 1, cal->tm_mday, cal->tm_hour,
            cal->tm_min, cal->tm_sec, t < 0 ? '-' : '+',
            static_cast<int>(t / 60), static_cast<int>(t - (t/60)*60));

    root.newProperty("DATE", timestring);
    root.newProperty("AUTHOR", "");
    root.newProperty("VERSION", "2.6");

    // Create the header
    xml::XmlNode  header("HEADER");
    root.addChild(header);

    header.newProperty("TIME_UNITS", "milliseconds");

    xml::XmlNode timeOrder("TIME_ORDER");
    root.addChild(timeOrder);
 
    if (!validate())
      throw ConvertException("Tried to create an invalid ELAN file.");
  }



  string EafDocument::addTimeCode(unsigned int value) {
    return addTimeCode(boost::lexical_cast<string>(value));
  }



  string EafDocument::addTimeCode(const string& value) {
    string id;
    do 
      id = "sltc" + boost::lexical_cast<string>(timeCodeNameIndexCounter++);
    while (timeCodes.count(id));

    timeCodes[id] = value;
    return id;
  }



  void EafDocument::save(const string& filename) {
    sync();
    XmlDocument::save(filename);
  }




  bool EafDocument::addMediaDescriptor(const string& filename, const string& mime) {
    xml::XmlNode header;
    try {
      header = getHeader();
    }
    catch (ConvertException&) {
      return false;
    }

    /*
     * WARNING
     *
     * This code is potentially unsafe (see man 3 realpath)
     */
    vector<char> canonicalPath(4096); // 4096 here is a Stetson-Harrison constant
    if (!realpath(filename.c_str(), &canonicalPath[0]))
      return false;

    xml::XmlNode mdNode("MEDIA_DESCRIPTOR");
    string mediaUrl = "file://";
    mediaUrl += &canonicalPath[0];
    mdNode.newProperty("MEDIA_URL", mediaUrl);
    mdNode.newProperty("RELATIVE_MEDIA_URL", filename);
    mdNode.newProperty("MIME_TYPE", mime);

    // Check if the header contains other child elements before adding. If 
    // it does, add the MD element as the first in the list
    if (header.hasChildren())
      header.getFirstChild().addPreviousSibling(mdNode);
    else
      header.addChild(mdNode);
    return true;
  }



  xml::XmlNode EafDocument::getHeader() {
    return getRoot().getChild("HEADER");
  }

  void EafDocument::sync() const {
    // Delete current time codes from the tree
    xml::XmlNode timeOrder = getRoot().getChild("TIME_ORDER");
    while(timeOrder.hasChildren())
      timeOrder.begin()->unlink();

    for (map < string, string >::const_iterator it = timeCodes.begin();
         it != timeCodes.end(); it++) {
      xml::XmlNode node("TIME_SLOT");
      node.newProperty("TIME_SLOT_ID", it->first);
      node.newProperty("TIME_VALUE", it->second);
      timeOrder.addChild(node);
    }

    // Add the default linguistic type in case it does not exist
    addLingType();

    if (!validate())
      throw ConvertException("Internal validation failure");
  }



  void EafDocument::addLingType(const std::string& name) const {
    if (getRoot().hasChild("LINGUISTIC_TYPE")) {      
      for (auto lingTypeNode = getRoot().getChild("LINGUISTIC_TYPE").getSiblingIterator();
           lingTypeNode != getRoot().end() && lingTypeNode->getName() == "LINGUISTIC_TYPE";
           ++lingTypeNode) 
        if (lingTypeNode->getProperty("LINGUISTIC_TYPE_ID") == name)
          return;
    }

    // Add the new type. The correct location is just after the last tier.
    // If there are no tiers, then add just after TIME_ORDER.
    xml::XmlNode linguisticTypePlaceNode;
    if (getRoot().hasChild("TIER"))
      linguisticTypePlaceNode = getRoot().getLastChild("TIER");
    else if (getRoot().hasChild("TIME_ORDER")) 
      linguisticTypePlaceNode = getRoot().getLastChild("TIME_ORDER");
    else 
      throw ConvertException("Internal XML creation error: Could not add LINGUISTIC_TYPE element: neither were any TIER nodes nor a TIME_ORDER node found.");

    xml::XmlNode lingTypeNode("LINGUISTIC_TYPE");
    lingTypeNode.newProperty("LINGUISTIC_TYPE_ID", name);
    lingTypeNode.newProperty("TIME_ALIGNABLE", "true");
    lingTypeNode.newProperty("GRAPHIC_REFERENCES", "false");
    linguisticTypePlaceNode.addNextSibling(lingTypeNode);
  }



  tier_deque_t EafDocument::getTiers() {
    xml::XmlNode node = getRoot().getChild("TIER");

    tier_deque_t td;
    
    if (getRoot().hasChild("TIER")) {
      for (auto node = getRoot().getChild("TIER").getSiblingIterator(); 
           node != getRoot().end() && node->getName() == "TIER"; ++node) {
        tier_t newTier;
        newTier.first = node->getProperty("TIER_ID");

        for (auto annNode = node->begin(); annNode != node->end() && 
               annNode->getName() ==  "ANNOTATION"; ++annNode) {
          xml::XmlNode annNode2 = annNode->getFirstChild();
          assert(annNode2.getName() == "ALIGNABLE_ANNOTATION");

          xml::XmlNode annNode3 = annNode2.getFirstChild();
          assert(annNode3.getName() == "ANNOTATION_VALUE");

          string name = annNode3.getContent();
          string timeRef1 = annNode2.getProperty("TIME_SLOT_REF1");
          string timeRef2 = annNode2.getProperty("TIME_SLOT_REF2");

          Annotation ann(name, boost::lexical_cast<int>(timeCodes[timeRef1]),
                         boost::lexical_cast<int>(timeCodes[timeRef2]));
          newTier.second.push_back(ann);
        }

        td.push_back(newTier);
      }
    }
    return td;
  }



  int EafDocument::getLastTimecode() {
    int temp = 0;
    for (std::map<std::string,std::string>::const_iterator it =
           timeCodes.begin(); it != timeCodes.end(); ++it)
      if (boost::lexical_cast<int>(it->second) > temp) 
        temp = boost::lexical_cast<int>(it->second);
      
    return temp;
  }



  void EafDocument::addTier(const tier_t& tier) {
    xml::XmlNode root = getRoot();
    xml::XmlNode tierNode("TIER");

    // Add basic attributes
    tierNode.newProperty("ANNOTATOR", "SLMotion");
    tierNode.newProperty("LINGUISTIC_TYPE_REF", "default-lt");

    // Ensure uniqueness of the ID
    string tierId = tier.first;
    if (tierIds.count(tierId)) {
      std::cerr << "slconvert: Warning: A tier with the ID '" << tierId
                << "' already exists in the document." << std::endl;
      size_t counter = 2;
      string tierId2;
      do 
        tierId2 = tierId + boost::lexical_cast<string>(counter++);
      while (tierIds.count(tierId2));
      tierId = tierId2;
      std::cerr << "The tier will be added as '" << tierId << "'" 
                << std::endl;
    }

    tierNode.newProperty("TIER_ID", tierId);
    tierIds.insert(tierId);

    // The tier should be added as a sibling with respect to other tier nodes
    // (order must be preserved), so attempt to locate the TIME_ORDER node,
    // and attach as its sibling (the other tiers should follow directly afterwards)
    if (!root.hasChild("TIME_ORDER"))
      throw ConvertException("No TIME_ORDER element was found.");
    xml::XmlNode timeOrder =  root.getChild("TIME_ORDER");

    timeOrder.addNextSibling(tierNode);

    // Add annotations
    for (std::deque<Annotation>::const_iterator it = tier.second.begin();
         it != tier.second.end(); it++) {
      // The structure is as follows:
      // <TIER>
      //   <ANNOTATION>
      //     <ALIGNABLE_ANNOTATION>
      //       <ANNOTATION_VALUE>foo</ANNOTATION_VALUE>
      //     </ALIGNABLE_ANNOTATION>
      //   </ANNOTATION>
      // </TIER>

      xml::XmlNode annNode("ANNOTATION");
      tierNode.addChild(annNode);

      xml::XmlNode alAnnNode("ALIGNABLE_ANNOTATION");
      annNode.addChild(alAnnNode);

      alAnnNode.newTextChild("ANNOTATION_VALUE", it->text);

      // Assign a unique ID to the annotation
      string id;
      do 
        id = "slann" + boost::lexical_cast<string>(annotationIdIndexCounter++);
      while(annotationIds.count(id));

      annotationIds.insert(id);

      alAnnNode.newProperty("ANNOTATION_ID", id);

      alAnnNode.newProperty("TIME_SLOT_REF1", addTimeCode(it->timerefs.first));
      alAnnNode.newProperty("TIME_SLOT_REF2", addTimeCode(it->timerefs.second));
    }
  }

  std::ostream& EafDocument::print(std::ostream& os) const {
    sync();
    return xml::XmlDocument::print(os);
  }
}
