#include "ElmSkinDetector.hpp"
#include <boost/lexical_cast.hpp>
#include "exceptions.hpp"
#include "BlackBoardExplorer.hpp"
#include "PropagationSkinDetector.hpp"
#include "Blob.hpp"

using cv::Mat;
using cv::TermCriteria;
using cv::Rect;
using cv::Size;
using cv::Scalar;
using boost::lexical_cast;
using boost::program_options::value;
using std::string;
using std::vector;
using std::cerr;
using std::endl;



namespace slmotion {
  static ElmSkinDetector DUMMY(true);

 bool ElmSkinDetector::processRangeImplementation(frame_number_t first,
						  frame_number_t last,
						  UiCallback* /*uiCallback*/){

   for(int framenr=first;framenr<(int)last;framenr++){

     cerr << framenr << " ";

     Mat inMask;
     bool inMaskDefined=false;

     if (getBlackBoard().has(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)){
       inMask = getBlackBoard().get<Mat>(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)->clone();
       inMaskDefined=true;
     } else
       getBlackBoard().set(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY,
                           Mat());
     
     BlackBoardPointer<Mat> outMask = getBlackBoard().get<Mat>(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY);
    const Mat& inFrame = getFrameSource()[framenr];

    // Make sure that the mask is not empty
    *outMask = Mat(inFrame.size(), CV_8UC1, cv::Scalar::all(0));

    doDetect(inFrame, *outMask,inMaskDefined ? &inMask : NULL);

    // post processing
    postprocess(inFrame, *outMask);

    //cv::morphologyEx(outMask, outMask, cv::MORPH_OPEN, 
    //		     cv::getStructuringElement(cv::MORPH_ELLIPSE, cv::Size(2, 2)), 
    //		     cv::Point(-1,-1), 1);

    getBlackBoard().set(framenr, SKINDETECTOR_BLACKBOARD_MASK_ENTRY, 
                        *outMask);

//            cv::imshow("post-processed output",outMask);


//          cv::waitKey(0);   


   }

   cerr << endl;

   return true;
 }



  void ElmSkinDetector::detect(const cv::Mat& /*inFrame*/,
			       cv::Mat& /*outMask*/) {
    throw string("ElmSkinDetector::detect should not be called for a single frame");
  }

  void ElmSkinDetector::doDetect(const cv::Mat& inFrame,
				 cv::Mat& outMask, cv::Mat *inputmask)
  {

    if(!trained)
      trainByFaceDetector();


    assert(inFrame.size() == outMask.size());
    assert(inFrame.depth() == CV_8U);
    assert(outMask.type()  == CV_8UC1);

    if(inputmask){
      int maskexpand=30;
      cv::morphologyEx(*inputmask, *inputmask, cv::MORPH_DILATE, 
		      cv::getStructuringElement(cv::MORPH_ELLIPSE, cv::Size(maskexpand,maskexpand)), 
		      cv::Point(-1,-1), 1);
    }
  
     localBlackBoardType localbb;

     int dummyframenr=6666;

    localbb[std::pair<int,std::string>(dummyframenr,"frame")]=inFrame;

    cv::Mat pmap(inFrame.rows,inFrame.cols,
		 cv::DataType<float>::type,cv::Scalar::all(0));

    outMask=Mat(inFrame.rows,inFrame.cols,CV_8UC1,cv::Scalar::all(0));

    for(int i = 0; i < inFrame.rows; i++)
      for(int j = 0; j < inFrame.cols; j++){

	if(inputmask && inputmask->at<uchar>(i,j)==0){
	  outMask.at<uchar>(i,j)= 0;
	} else{
	  vector<float> samplevec;
	  fe.extractFeatures(samplevec,i,j,dummyframenr,localbb);
	  pmap.at<float>(i,j)=getPixelP(samplevec);
	  
	  outMask.at<uchar>(i,j)= pmap.at<float>(i,j) >= skinthresholdp ? 255 : 0;
	}
      }

    // expand the detections 

    // idea: dilate the detected mask to its watershed
    // (up to a fixed number of pixels)

    // reconsider the probability of the pixels
    // in the dilated region with a looser threshold

    int maxexpand=35;
    //int refineexpand=10;

    //    Mat watershed;
    Mat dilated;

    cv::dilate(outMask, dilated, 
	       cv::getStructuringElement(cv::MORPH_ELLIPSE, 
					 Size(maxexpand,maxexpand)));

    // ensure existence of pmap values within the dilated mask


    for(int i = 0; i < inFrame.rows; i++)
      for(int j = 0; j < inFrame.cols; j++){
	if(dilated.at<uchar>(i,j)&&pmap.at<float>(i,j)==0){
	  vector<float> samplevec;
	  fe.extractFeatures(samplevec,i,j,dummyframenr,localbb);
	  pmap.at<float>(i,j)=getPixelP(samplevec);
	}
      }

    // apply smoothing to pmap

    Mat smoothedpmap;

    if(blurmagnitude>0){

      cv::GaussianBlur(pmap,smoothedpmap,cv::Size(0,0),blurmagnitude);
    } else
      smoothedpmap=pmap.clone();

    //findWatershed(outMask,inFrame,watershed);

   // morplogically fill small gaps in the expansion
   //cv::morphologyEx(watershed, watershed, cv::MORPH_CLOSE, 
    //		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(4, 4)), 
    //		    cv::Point(-1,-1), 1);
   

    for(int i = 0; i < inFrame.rows; i++)
      for(int j = 0; j < inFrame.cols; j++)
	if(smoothedpmap.at<float>(i,j)>skinthresholdp)
	  outMask.at<uchar>(i,j)= 255;

    // then reconsider a small area around mask without any watershed requirements

  //   cv::dilate(outMask, dilated, 
// 	       cv::getStructuringElement(cv::MORPH_ELLIPSE, 
// 					 Size(refineexpand,refineexpand)));
//     for(int i = 0; i < inFrame.rows; i++)
//       for(int j = 0; j < inFrame.cols; j++)
// 	if(dilated.at<uchar>(i,j) && pmap.at<float>(i,j)>0.6)
// 	  outMask.at<uchar>(i,j)= 255;



//    cv::Mat skinFrame=inFrame.clone();
//     for(int x=0;x<inFrame.cols;x++)
//       for(int y=0;y<inFrame.rows;y++)
// 	if(outMask.at<uchar>(y,x)){
// 	  skinFrame.at<cv::Vec3b>(y,x)=cv::Vec3b(255,50,50);
//        	 }
 


//       cv::imshow("input frame",inFrame);
//       if(inputmask)
// 	cv::imshow("incoming skin mask",*inputmask);
//        cv::imshow("expanded pmap",pmap);
//        cv::imshow("smoothed pmap",smoothedpmap);
//        cv::imshow("frame with detected skin",skinFrame);


//          cv::waitKey(0);   





//  // morplogically fill small gaps in the mask
//    cv::morphologyEx(outMask, outMask, cv::MORPH_CLOSE, 
// 		    cv::getStructuringElement(cv::MORPH_ELLIPSE, Size(4, 4)), 
// 		    cv::Point(-1,-1), 1);

   cv::dilate(outMask, outMask, 
	   cv::getStructuringElement(cv::MORPH_ELLIPSE, 
					 Size(2,2)));
   
   
  }


  bool ElmSkinDetector::trainByFaceDetector() {

    localBlackBoardType localbb;

    delete cls;
    cls=new ELMClassifier;

    //cls->maxcachedim=2;
    cls->Nneur_sigmoid=101;
    cls->Nneur_RBF=0;

    vector<size_t> sampledframes;

    // sample the nrFramesToSample 
    // first frames with a succesful face detection

    size_t nrFramesToSample=20;

    vector<vector<float> > negSamples,posSamples;
    vector<float> negWeights,posWeights;
       
    FeatureExtractor fe;



    size_t framenr = 0; // this is a bit bad, should take the frames only in the
                        // slice that we're currently working with

    for (FrameSource::const_iterator it = getFrameSource().cbegin();
	 sampledframes.size()<nrFramesToSample &&
	   it < getFrameSource().cend(); ++framenr, ++it) {
      // read frames until successful detection

    if (!getBlackBoard().has(framenr, FACEDETECTOR_BLACKBOARD_ENTRY)) 
      continue;

    BlackBoardPointer<cv::Rect> faceLocation = getBlackBoard().get<Rect>(framenr, FACEDETECTOR_BLACKBOARD_ENTRY);
      if (*faceLocation != FaceDetector::INVALID_FACE) {         
	   sampledframes.push_back(framenr);	
	   localbb[std::pair<int,std::string>(framenr,"frame")]=getFrameSource()[framenr];
	   localbb[std::pair<int,std::string>(framenr,"facerect")]=faceLocation;
      }
    }

    assert(sampledframes.size()>1);

    sampleFrames(localbb,"dummyid","",false,sampledframes,negSamples,posSamples,negWeights,posWeights,fe,false,false,true,false,false,true,true);

    size_t samplesize=15000;

    if(posSamples.size()>samplesize) posSamples.resize(samplesize);
    if(negSamples.size()>samplesize) negSamples.resize(samplesize);

    // at the moment this truncation is a way to prevent numerical
    // problems in ELM training (should eventually fix properly)

    // hard coded: cls uses feature 0

    vector<bool> dimMask=fe.getFeatureMask(0);
    cls->train(negSamples,posSamples,&dimMask,&negWeights,
	       &posWeights);

    //    cerr << "...classifier trained" << endl;

    trained=true;

    
    return true;
  }

  void ElmSkinDetector::reset() {
    SkinDetector::reset();
    trained=false;
    pixelPCache.clear();
  }



  boost::program_options::options_description ElmSkinDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add(SkinDetector::getConfigurationFileOptionsDescription());
    opts.add_options()
      ("ElmSkinDetector.pthreshold", 
       value<double>()->default_value(0.75),
       "Sets the threshold for the value of estimated "
       "skin probability that a pixel should have"
       " to be interpreted as a skin pixel. Pixels whose "
       "value exceeds this threshold will be classified as "
       "skin pixels.")
      ("ElmSkinDetector.blurmagnitude", value<double>()->default_value(0),
       "Sets the amount of Gaussian blur applied to the skin probability "
       "distribution before thresholding.")
;
    return opts;
  }



  Component* ElmSkinDetector::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    return new ElmSkinDetector(configuration, blackBoard, frameSource);
  }



  ElmSkinDetector::ElmSkinDetector(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) :
    SkinDetector(configuration, blackBoard, frameSource),
    cls(NULL),
    trained(false),
    skinthresholdp(configuration["ElmSkinDetector.pthreshold"].as<double>()),
    blurmagnitude(configuration["ElmSkinDetector.blurmagnitude"].as<double>())

      {
	
      }


  void findWatershed(Mat &mask,const Mat &frame, Mat &output){


    // prepare seeds for the watershed expansion
   
    output=mask.clone();

    vector<Blob> outBlobs = 
      Blob::extractBlobsFromMask(output,0,0);
    
    for (size_t blobidx=0;blobidx<outBlobs.size();blobidx++)
      outBlobs[blobidx].draw(output, Scalar::all(blobidx+1));
   

   int backgroundmargin = 40;
   
   Mat flippedMask=output.clone();
   
   for(int i = 0; i < frame.rows;i++)
     for(int j = 0; j < frame.cols;j++)
       flippedMask.at<uchar>(i,j)=(flippedMask.at<uchar>(i,j)==0)?255:0;
   
   Mat distMat;
   distanceTransform(flippedMask, distMat, CV_DIST_L2,3);
   
   for(int i = 0; i < frame.rows; i+=6)
     for(int j = 0; j < frame.cols; j+=6)
       if(distMat.at<float>(i,j)>backgroundmargin)
	 output.at<uchar>(i,j)=255;

   
   Mat blobmask=Mat::zeros(frame.size(), CV_32SC1);
   
   for(int i = 0; i < frame.rows; i++)
     for(int j = 0; j < frame.cols; j++)
       blobmask.at<int>(i,j)=(int)output.at<uchar>(i,j);
   
   // perform watershed
   
   cv::watershed(frame,blobmask);
   
   for(int i = 0; i < frame.rows; i++)
     for(int j = 0; j < frame.cols; j++)
       if(blobmask.at<int>(i,j)>0&&blobmask.at<int>(i,j)!=255)
	 output.at<uchar>(i,j)= 255;
       else
	 output.at<uchar>(i,j)= 0;

   // output now holds the watershed expansion

  }


}

