#include "FaceDetector.hpp"

using cv::Rect;
using std::string;

namespace slmotion {
  extern int debug;
  static FaceDetector DUMMY(true);



  static cv::Point2d getCentroid(Rect* x) {
    // printRect(x);
  
    return cv::Point2d(x->x + x->width/2.0,
                       x->y + x->height/2.0);
  }



  /**
   * Returns centroid distance between the two rectangles
   */
  static double distance(Rect* a, Rect* b) {
    return norm(getCentroid(a) - getCentroid(b));
  }



  /**
   * Proportional, undirected size difference normalised to [0,1]
   */
  static double sizeDifference(Rect* a, Rect* b) {
    double aa = a->size().area();
    double ba = b->size().area();
  
    return std::abs(aa-ba)/std::max(aa,ba);
  }



  /**
   * Performs simple linear interpolation for the rects between the first 
   * (inclusive) index and the last (exclusive). First index must not be 0,
   * and first and second must be unequal.
   */
  static void interpolateGap(std::vector<Rect*>& rects, const std::pair<size_t,size_t>& gap) {
    assert(gap.first > 0);
    assert(gap.first < gap.second);
    const Rect& first = *rects[gap.first-1];
    const Rect& last = *rects[gap.second];
    size_t len = gap.second - gap.first;
    for (size_t i = 0; i < len; ++i) {
      double ratio = (i+1) / static_cast<double>(len+1);
      int newX = first.x + (last.x - first.x)*ratio;
      int newY = first.y + (last.y - first.y)*ratio;
      int newW = first.width + (last.width - first.width)*ratio;
      int newH = first.height + (last.height - first.height)*ratio;
    
      *rects[gap.first + i] = Rect(newX, newY, newW, newH);
    }
  }



  /**
   * Locates a gap, beginning at the firstIndex.
   *
   * Return value: first = first invalid detection in a gap,
   * last = first valid
   */
  static std::pair<size_t,size_t> findGap(const std::vector<Rect*>& rects, 
                                          size_t firstIndex) {
    std::pair<size_t,size_t> indices(firstIndex,rects.size());
    while(indices.first < rects.size() &&
          *rects[indices.first] != ViolaJonesFaceDetector::INVALID_FACE)
      ++indices.first;
    
    indices.second = indices.first;
    while(indices.second < rects.size() &&
          *rects[indices.second] == ViolaJonesFaceDetector::INVALID_FACE)
      ++indices.second;
    return indices;
  }



  // static void printRect(Rect* r) {
  //   std::cout << r->tl() << "->" << r->br() << std::endl;
  // }



  /**
   * Returns true if distance and size difference conditions are satisfied
   */
  static bool goodPair(Rect* a, Rect* b) {
    if (*a == ViolaJonesFaceDetector::INVALID_FACE || *b == ViolaJonesFaceDetector::INVALID_FACE) {
      return false;
    }
  
    if (distance(a, b) > slmotion::FACEDETECTOR_DEFAULT_MAX_MOVE) {
      return false;
    }
  
    if (sizeDifference(a, b) > slmotion::FACEDETECTOR_DEFAULT_MAX_AREA_CHANGE) {
      return false;
    }
  
    return true;
  }



  /**
   * Remove bad detections from the given range
   *
   * Must be at least 3-long
   */
  static void removeBadDetections(std::vector<Rect*>& rects) {
    assert(rects.size() > 2);
    auto next = rects.begin();
    auto prev = next++; 
    auto cur = next++;
    
    // deal with the first detection as a special case
    if (!goodPair(*prev, *cur)) {
      **prev = ViolaJonesFaceDetector::INVALID_FACE;
    }
  
    for (; next < rects.end(); ++next, ++cur, ++prev)
      if (!goodPair(*prev, *cur) && !goodPair(*cur, *next)) {
        **cur = ViolaJonesFaceDetector::INVALID_FACE;
      }
  
    // deal with the last detection as a special case
    if (!goodPair(*cur, *prev)) {
      **cur = ViolaJonesFaceDetector::INVALID_FACE;
    }
  }



  void FaceDetector::process(frame_number_t frnumber) {   
    if (frnumber >= getFrameSource().size())
      throw std::out_of_range("Tried to detect face in a frame whose index exceeds the total number of frames!");

    assert(getFrameSource().size() > 0);
    getBlackBoard().set(frnumber, FACEDETECTOR_BLACKBOARD_ENTRY, 
                        scaleAndTranslate(getRawFace(frnumber)));
  }



  bool FaceDetector::processRangeImplementation(frame_number_t first, 
                                                frame_number_t last,
                                                UiCallback* uiCallback) {
    for (size_t i = first; i < last; ++i) {
      if (slmotion::debug > 1) {
        // if (slmotion::debug == 1)
        //   std::cerr << '\r';
        std::cerr << i + 1 - first << '/' << last - first;
      }
      process(i);
      if (uiCallback != NULL && !(*uiCallback)((i-first+1)*100./(last-first)))
        return false;
    }

    if (slmotion::debug > 1) 
      std::cerr << std::endl;

    std::vector<Rect*> detections = getDetectionsForRange(first, last);

    if (detections.size() > 2) {
      if (slmotion::debug > 1)
	std::cerr << "Removing bad detections (moved too much etc.)" << std::endl;
      removeBadDetections(detections);
      
      if (slmotion::debug > 1)
	std::cerr << "Interpolating missing frames" << std::endl;
      interpolate(detections);
    }

    return true;
  }



  cv::Rect FaceDetector::scaleAndTranslate(const cv::Rect& original) {
    if (original == INVALID_FACE)
      return INVALID_FACE;

    return Rect(original.x + (original.width)*(1-scale.first)/2.0
                + translation.first,
                original.y + (original.height)*(1-scale.second)/2.0
                + translation.second,
                original.width * scale.first,
                original.height * scale.second);
  }



  /**
   * Interpolate missing detections for the given range
   */
  void FaceDetector::interpolate(std::vector<Rect*>& rects) {
    if (rects.size() == 0)
      return;

    // Step one: find one detection
    size_t i;
    for(i = 0; i < rects.size() && *rects[i] == ViolaJonesFaceDetector::INVALID_FACE;
        ++i);
    
    // Verify that a detection _exists_, otherwise, do nothing
    if (i >= rects.size())
      return;

    size_t firstIndex = i; // first detection index
    // Fill any missing detections before the first detection
    for(auto it = rects.begin(); it < rects.begin() + firstIndex; ++it)
      **it = *rects[firstIndex];

    if(*rects.front() == INVALID_FACE)
      return; // cannot interpolate if there were NO valid detections

    // interpolate any gaps
    std::pair<size_t,size_t> indices = findGap(rects, firstIndex);
    for (; indices.first < rects.size() && indices.second < rects.size();
         indices = findGap(rects, indices.second))
      interpolateGap(rects, indices);

    // Finally, if required, fill in the last detection
    if (indices.first < rects.size())
      for(auto it = rects.begin() + indices.first; it < rects.end(); ++it)
        **it = *rects[indices.first-1];
  }



  boost::program_options::options_description FaceDetector::getConfigurationFileOptionsDescription() const {
    boost::program_options::options_description opts;
    opts.add_options()
      ("FaceDetector.cascade", 
       boost::program_options::value<string>()->default_value(getDefaultHaarCascadeFile()),
       "Sets the cascade filename. A valid cascade file is required for the classifier to work "
       "properly.")
      ("FaceDetector.scale", boost::program_options::value<string>()->default_value("1.0:1.0"),
       "Input format is <double> | <double>:<double>\n"
       "Scales the face by a factor. If only one factor is specified, both the "
       "width and the height of the face rectangle will be scaled equally. If "
       "two colon-separated factors are specified, the first factor will be used "
       "for scaling the width and the second argument for scaling the height. "
       "E.g. scale=1.5 or scale=0.8:0.7")
      ("FaceDetector.translate", boost::program_options::value<string>()->default_value("0:0"),
       "Input format is <int>:<int>\n"
       "Translates the face by some amount of pixels. The first parameter "
       "specifies the number of pixels with respect to X axis, and the second "
       "parameter with respect to Y axis.")
      ("FaceDetector.maxmove", boost::program_options::value<int>()->default_value(25),
       "Sets the maximum number of pixels that the detected face can move with "
       "respect to its position in the previous frame. If the face was not "
       "detected in the previous frame, the distance will be considered with "
       "respect to the last detected position.")
      ("FaceDetector.maxareachange", boost::program_options::value<double>()->default_value(0.5),
       "Sets the maximum proportional change in the area of the face detected "
       "in respect to the previous frame, or the latest detected face. E.g. if "
       "set to 0.5, the face may change its size up to 50 % between detections.")
      ("FaceDetector.minneighbours", boost::program_options::value<int>()->default_value(2),
       "Minimum number of overlapping neighbouring detections required for "
       "a detection to be considered valid. Larger number means more accurate "
       "detections (in terms of false positives), but at the cost of possibly "
       "missing some valid detections.")
      ("FaceDetector.scalefactor", boost::program_options::value<double>()->default_value(1.2),
       "How big a jump there is between different scales; accuracy can be "
       "increased by moving this value closer to 1.0, but this will increase "
       "computation time.");
    return opts;
  }



  Component* FaceDetector::createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const {
    FaceDetector faceDetector(blackBoard, frameSource);

    // if (vm.count("face-cascade"))
    //   config.faceDetectorCascadeFilename = vm["face-cascade"].as<string>();
    // else if (vm.count("facedetection.cascade"))
    //   config.faceDetectorCascadeFilename = vm["facedetection.cascade"].as<string>();

    if (configuration.count("FaceDetector.cascade"))
      faceDetector.setCascadeFilename(configuration["FaceDetector.cascade"].as<std::string>());

    if (configuration.count("FaceDetector.scale")) {
      string s = configuration["FaceDetector.scale"].as<string>();
      double facescalex = 1.0;
      double facescaley = 1.0;
      if (sscanf(s.c_str(), "%lf:%lf", &facescalex, &facescaley) == 2) 
        faceDetector.setScale(facescalex, facescaley);
      else if (sscanf(s.c_str(), "scale=%lf", &facescalex) == 1) 
        faceDetector.setScale(facescalex, facescalex);
      else 
        throw ConfigurationFileException("A double or a colon-separated pair of doubles is expected after scale=");
    }

    if (configuration.count("FaceDetector.translate")) {
      std::string s = configuration["FaceDetector.translate"].as<string>();
      std::pair<int,int> translation(0,0);
      if (sscanf(s.c_str(), "%i:%i", &translation.first,
                 &translation.second) == 2)
        faceDetector.setTranslation(translation.first, 
                                    translation.second);
      else 
        throw ConfigurationFileException("A colon-separated pair of integer values is expected after scale=");
    }

    if (configuration.count("FaceDetector.maxmove"))
      faceDetector.setMaxMove(configuration["FaceDetector.maxmove"].as<int>());

    if (configuration.count("FaceDetector.maxareachange"))
      faceDetector.setMaxAreaChange(configuration["FaceDetector.maxareachange"].as<double>());

    if (configuration.count("FaceDetector.minneighbours"))
      faceDetector.setMinNeighbours(configuration["FaceDetector.minneighbours"].as<int>());

    if (configuration.count("FaceDetector.scalefactor"))
      faceDetector.setScaleFactor(configuration["FaceDetector.scalefactor"].as<double>());

    return new FaceDetector(faceDetector);
  }
}
