#ifndef SLMOTION_FACE_DETECTOR
#define SLMOTION_FACE_DETECTOR

#include "ViolaJonesFaceDetector.hpp"

namespace slmotion {
  const double FACEDETECTOR_DEFAULT_MAX_MOVE = 25; ///< Default value constant
  const double FACEDETECTOR_DEFAULT_MAX_AREA_CHANGE = 0.5; ///< Default value constant



  /**
   * The default interpolated Viola-Jones-style face detector
   */
  class FaceDetector : public ViolaJonesFaceDetector {
  public:
    /**
     * Returns the component name
     */
    virtual std::string getComponentName() {
      return "Face Detector";
    }



    /**
     * Constructs a new face detector object
     *
     * @param faceDetectorOpts Options struct as defined above
     * @param blackBoard A pointer to a black board object. The board must 
     * outlive the face detector.
     * @param frameSource A pointer to a frame source object. The source 
     * must outlive the face detector.
     */
    explicit FaceDetector(BlackBoard* blackBoard, FrameSource* frameSource,
                          const std::string& cascadeFilenameArgument = getDefaultHaarCascadeFile())
      : 
      ViolaJonesFaceDetector(blackBoard, frameSource, cascadeFilenameArgument),
      scale(std::make_pair(1.0,1.0)),
      translation(std::make_pair(0, 0)),
      maxMove(FACEDETECTOR_DEFAULT_MAX_MOVE),
      maxAreaChange(FACEDETECTOR_DEFAULT_MAX_AREA_CHANGE)
    {
    }



    /**
     * Performs raw face detection in the given frame, without 
     * interpolation. However, scaling and translation are applied.
     */
    void process(frame_number_t framenumber);



    inline void setScale(double x, double y) {
      scale = std::make_pair(x, y);
    }



    inline void setTranslation(int x, int y) {
      translation = std::make_pair(x, y);
    }



    inline void setMaxMove(double d) {
      maxMove = d;
    }



    inline void setMaxAreaChange(double d) {
      maxAreaChange = d;
    }



    inline virtual property_set_t getProvided() const {
      return property_set_t { FACEDETECTOR_BLACKBOARD_ENTRY };
    }



    virtual std::string getShortDescription() const {
      return "Viola-Jones face detector";
    }



    virtual std::string getLongDescription() const {
      return "Viola-Jones face detector that interpolates any missing frames.";
    }



    virtual property_set_t getRequirements() const {
      return property_set_t();
    }

    virtual std::string getShortName() const {
      return "FaceDetector";
    }

    virtual std::string getComponentName() const {
      return "Face Detector";
    }

    FaceDetector(bool) : ViolaJonesFaceDetector(true) {}

    virtual void reset() { }

    virtual boost::program_options::options_description getConfigurationFileOptionsDescription() const;

    virtual boost::program_options::options_description getCommandLineOptionsDescription() const {
      return boost::program_options::options_description();
    }

  private:
    virtual Component* createComponentImpl(const boost::program_options::variables_map& configuration, BlackBoard* blackBoard, FrameSource* frameSource) const;



    /**
     * Returns a copy of the original rectangle that has been scaled and
     * translated according to the object settings, with the exception of
     * INVALID_FACE which will remain invalid.
     */
    cv::Rect scaleAndTranslate(const cv::Rect& original);



    /**
     * Iterates over the associated frames and performs detection in each 
     * frame, the last frame number is exclusive. After detection has been
     * performed for all frames, any detections missing between two 
     * successful detections are linearly interpolated. Any unsuccessful 
     * detections before the first and after the last successful detections
     * are replaced by the first and last successful detections, 
     * respectfully.
     *
     * I.e. for i in [first,last): detect and interpolate
     *
     * @param first First frame to process (default 0)
     * @param last Last frame to process (default maximum). If the last 
     * frame exceeds the boundaries of the frame source, only frames up to 
     * the maximum are processed.
     */
    bool processRangeImplementation(frame_number_t first,
                                    frame_number_t last,
                                    UiCallback* uiCallback);



    /**
     * Sets the cascade filename in use, resets the cascade object, and checks 
     * that it was opened successfully, or throws an exception.
     */
    void setAndVerifyCascadeFilename(const std::string& filename);
    


    /**
     * Detects a face in the image using a Haar classifier
     *
     * @param frame Source frame; assumed to be a 3-channel BGR image.
     * @param facelist Output vector where found faces are stored
     * @param mask Should contain a skin filter mask or NULL if no mask is to be used.
     *
     * @return true if faces were found, sotring faces as Rect objects in the
     * facelist vector.
     */
    // bool detectFace(const cv::Mat& frame, std::vector<cv::Rect>& facelist, const cv::Mat* mask = NULL);

    /**
     * Performs a raw face detection and stores the result in the
     * cache, and returns the result. No interpolation is applied, and no 
     * other frames are considered; these are left for other functions. If 
     * multiple face candidates are detected, an INVALID_FACE is returned.
     *
     * @param img Input image. Assumed to be in BGR.
     *
     * @return The raw detection result, or INVALID_FACE in case of failure.
     */
    cv::Rect detectFace(const cv::Mat& img);



    /**
     * Reads in frames from the assembly line, optionally filling it
     * when necessary. Any frames before the first and after the last
     * possible detection, are considered equal to the first and last
     * detection, respectively. Any gaps between two successful
     * detections are filled in using linear interpolation.
     */
    cv::Rect getFaceAndInterpolate(const cv::Mat& currentFrame,
                                   size_t frnumber);



    /**
     * Interpolate missing detections for the given range
     */
    static void interpolate(std::vector<cv::Rect*>& rects);



    /**
     * To save computation time, store face detection results in this vector
     * during the first pass so that the expensive classification need not 
     * be done twice. Invalid detections are stored as INVALID_FACEs. Note 
     * that this cache MUST be cleared whenever the state needs to be reset.
     */
    std::pair<double,double> scale; ///< first = scaling wrt x axis (1.0 == no scaling), second = scaling wrt y axis
    std::pair<int,int> translation; ///< first = translation wrt x axis, second = translation wrt y axis
    double maxMove; ///< The maximum number of pixels that the detected face can move between frames 
    double maxAreaChange; ///< How great an amount the area of the detected face can change between frames 
  };
}
#endif
